# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2018 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import pymobius.forensics.vfs
import pymobius.forensics.registry.installed_programs
from pymobius.forensics.registry import *
import pymobius.p2p.application
import decoder_sd
import decoder_library_dat
import decoder_profile_xml
import decoder_searches_dat

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# All Date/Times are stored in Coordinated Universal Time (UTC).
# @see https://msdn.microsoft.com/pt-br/library/windows/desktop/ms724397(v=vs.85).aspx
#
# References:
#   . Shareaza 2.7.10.2 source code
#
# Shareaza main files:
#   . Searches.dat - Stores search history and search results
#   . Library1.dat and Library2.dat - Stores configuration for local folders and files
#   . *.sd files - Download control files (one per downloading file)
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Generic dataholder
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class dataholder (object):
  pass

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief iterate through Shareaza AppData folders
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def iter_shareaza_folders (vfs):
  for root in vfs:
    users_folder = root.get_child_by_name ('Users', False)

    if not users_folder:
      users_folder = root.get_child_by_name ('Documents and Settings', False)

    if users_folder:
      for entry in users_folder.get_children ():
        if entry.is_folder ():
          username = entry.name
          shareaza_folder = \
		  entry.get_child_by_path ('AppData/Roaming/Shareaza', False) or \
                  entry.get_child_by_path ('Application Data/Shareaza', False)

          if shareaza_folder:
            yield username, entry, shareaza_folder

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Retrieves Shareaza P2P activity data
# @author Eduardo Aguiar
# This function is planned to run in an independent thread. The idea here
# is to gather all activity data and only then write data to the model
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def retrieve (model):
  ant = Ant (model)
  ant.run ()

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief forensics: P2P Shareaza
# @author Eduardo Aguiar
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Ant (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, model):
    self.__model = model	# P2P model
    self.__item = model.item

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief run
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def run (self):

    # create local data model
    self.__data = dataholder ()
    self.__data.application = pymobius.p2p.application.application ()
    self.__data.application.id = 'shareaza'
    self.__data.application.name = 'Shareaza'

    self.__data.accounts = []
    self.__data.searches = []
    self.__data.local_files = []
    self.__data.remote_files = []

    # retrieve data
    self.__retrieve_registry_data ()
    self.__retrieve_app_data ()
    self.__normalize_data ()

    # update P2P model
    self.__model.applications.append (self.__data.application)
    self.__model.accounts += self.__data.accounts
    self.__model.searches += self.__data.searches
    self.__model.local_files += self.__data.local_files
    self.__model.remote_files += self.__data.remote_files

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief retrieve registry data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_registry_data (self):
    ant = pymobius.forensics.registry.main.Ant (self.__item)

    # Program version
    for registry in ant.get_data ():
      for program in pymobius.forensics.registry.installed_programs.get (registry):
        if program.display_name.startswith ('Shareaza '):
          self.__data.application.versions.add (program.version)
    
    # Shareaza folders
    #for username, key in pymobius.forensics.registry.iter_hkey_users (registry):
    #  shareaza_key = key.get_key_by_path ('Software\\Shareaza\\Shareaza')

    #  if shareaza_key:
    #    user_path = get_data_as_string (shareaza_key.get_data_by_name ('UserPath'))
    #    collection_path = get_data_as_string (shareaza_key.get_data_by_path ('Downloads\\CollectionPath'))
    #    complete_path = get_data_as_string (shareaza_key.get_data_by_path ('Downloads\\CompletePath'))
    #    incomplete_path = get_data_as_string (shareaza_key.get_data_by_path ('Downloads\\IncompletePath'))
    #    torrent_path = get_data_as_string (shareaza_key.get_data_by_path ('Downloads\\TorrentPath'))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief retrieve data from disk files
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_app_data (self):

    # get item VFS
    vfs = pymobius.forensics.vfs.get_item_vfs (self.__item)
    if not vfs:
      raise Exception, "Datasource is not available"

    # scan VFS root entries
    for username, user_folder, shareaza_folder in iter_shareaza_folders (vfs):

      data_folder = shareaza_folder.get_child_by_name ('Data', False)
      self.__retrieve_data_folder (data_folder, username)

      incomplete_folder = \
        user_folder.get_child_by_path ('AppData/Local/Shareaza/Incomplete', False) or \
        user_folder.get_child_by_path ('Application Data/Shareaza/Incomplete', False)
      self.__retrieve_incomplete_folder (incomplete_folder, username)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief retrieve Data folder evidence files
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_data_folder (self, folder, username):
    if not folder:
      return

    f = folder.get_child_by_name ('Searches.dat', False)
    decoder_searches_dat.retrieve (self.__item, self.__data, f, username)
  
    f = folder.get_child_by_name ('Library1.dat', False)
    if not f:
      f = folder.get_child_by_name ('Library2.dat', False)
    decoder_library_dat.retrieve (self.__item, self.__data, f, username)

    f = folder.get_child_by_name ('Profile.xml', False)
    decoder_profile_xml.retrieve (self.__item, self.__data, f, username)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief retrieve incomplete control files
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_incomplete_folder (self, folder, username):
  
    if not folder or not folder.is_folder ():
      return

    for f in folder.get_children ():
      if f.name.endswith ('.sd'):
        decoder_sd.retrieve (self.__item, self.__data, f, username)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief normalize retrieved data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __normalize_data (self):

    # sort and remove duplicated remote files
    remote_files = [ (f.timestamp, f) for f in self.__data.remote_files ]
    timestamp, peer_ip, size, name = None, None, None, None
    self.__data.remote_files = []

    for (timestamp, f) in sorted (remote_files):
      if (timestamp, peer_ip, size, name) != (f.timestamp, f.peer.ip, f.size, f.name):
        self.__data.remote_files.append (f)
        timestamp, peer_ip, size, name = f.timestamp, f.peer.ip, f.size, f.name
