# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2018 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import CDownload
import CXMLElement
import mobius
import pymobius.forensics.p2p

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief generic data holder
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class dataholder (object):
  pass

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Retrieve data from *.sd download control files
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def retrieve (item, data, incomplete_folder, username):
  if not incomplete_folder or not incomplete_folder.is_folder ():
    return

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # try to open .sd files
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  children = [ c for c in incomplete_folder.get_children () if c.name.endswith ('.sd') and c.is_file () ]

  for child in children:
    retrieve_sd (item, data, child, username)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Retrieve data from *.sd download file
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def retrieve_sd (item, data, child, username):
  reader = child.new_reader ()

  if not reader:
    return

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # decode file
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  decoder = mobius.decoder.mfc_decoder (reader)
  f = CDownload.decode (decoder)

  if not f:
    return

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # add file to local files
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  bytes_downloaded = f.size - f.bytes_to_download
  bytes_to_download = f.bytes_to_download

  if f.size:
    bytes_downloaded = '%d (%.2f%%)' % (bytes_downloaded, float (bytes_downloaded) * 100.0 / f.size)
    bytes_to_download = '%d (%.2f%%)' % (bytes_to_download, float (bytes_to_download) * 100.0 / f.size)

  # attributes
  f.username = username
  f.app_id = 'shareaza'
  f.app = 'Shareaza'
  f.shared = 'yes' if f.b_shared else 'no'
  f.complete = 'no'

  # hashes
  f.hashes = []

  for t in ('sha1', 'tiger', 'ed2k', 'bth', 'md5'):
    h = dataholder ()
    h.type = t
    h.value = getattr (f, 'hash_%s' % t, None)
    f.hashes.append (h)

  # tags
  f.tags = set ()
  if f.b_shared:
    f.tags.add ('shared')

  # metadata
  f.metadata = []
  f.metadata.append (('Bytes downloaded', bytes_downloaded))
  f.metadata.append (('Bytes to download', bytes_to_download))
  f.metadata += CXMLElement.get_metadata_from_pxml (f.pxml)

  data.local_files.append (f)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # add file + sources to remote files
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  for source in f.sources:
    rf = dataholder ()
    rf.name = f.name
    rf.size = f.size
    rf.url = source.url
    rf.timestamp = source.last_seen_time
    rf.username = username
    rf.app_id = 'shareaza'
    rf.app = 'Shareaza'
    rf.hashes = f.hashes
    rf.metadata = CXMLElement.get_metadata_from_pxml (f.pxml)

    # peer data
    peer = dataholder ()
    peer.ip = source.ip or source.server_ip
    peer.port = source.port
    peer.nickname = source.nick
    peer.country = source.country_name or source.country_code
    peer.application = ''
    rf.peer = peer

    # tags
    rf.tags = set ()

    # add remote file
    data.remote_files.append (rf)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # add file to kff.ignore
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  pymobius.forensics.p2p.set_handled (item, child)
