# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2018 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import CLibrary
import CXMLElement
import mobius
import pymobius.forensics.p2p

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief generic data holder
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class dataholder (object):
  pass

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief get folders from list recursively
# @param folder_list folder list
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def iter_folders (folder_list):
  for folder in folder_list:
    yield folder

    for child in iter_folders (folder.children):
      yield child

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Retrieve data from Library1.dat
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def retrieve (item, model, data_folder, username):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # try to open either Library1.dat or Library2.dat files
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  f = data_folder.get_child_by_name ('Library1.dat', False)

  if not f:
    f = data_folder.get_child_by_name ('Library2.dat', False)

  if not f:
    return

  reader = f.new_reader ()
  if not reader:
    return

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # decode file
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  decoder = mobius.decoder.mfc_decoder (reader)
  clib = CLibrary.decode (decoder)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # process data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  for folder in iter_folders (clib.folders):

    # add files to local files list
    for clf in folder.files:
      lf = new_local_file (folder, clf, username)
      model.local_files.append (lf)

      # add files with known sources to the remote_files list
      for source in clf.sources:
        rf = new_remote_file (lf, source, username)
        model.remote_files.append (rf)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # add file to ignored kff
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  pymobius.forensics.p2p.set_handled (item, f)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Create local file from CLibraryFile
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def new_local_file (folder, clf, username):
  f = dataholder ()
  f.name = clf.name
  f.size = clf.size

  f.path = folder.path + '\\' + clf.name
  f.shared = 'yes' if clf.b_shared else 'no'
  f.complete = 'yes'
  f.username = username
  f.app_id = 'shareaza'
  f.app = 'Shareaza'

  f.rating = clf.rating
  f.comments = clf.comments
  f.share_tags = clf.share_tags
  f.pxml = clf.pxml

  # hashes
  f.hashes = []

  for t in ('sha1', 'tiger', 'ed2k', 'bth', 'md5'):
    h = dataholder ()
    h.type = t
    h.value = getattr (clf, 'hash_%s' % t, None)
    f.hashes.append (h)

  # tags
  f.tags = set ()
  if clf.b_shared:
    f.tags.add ('shared')

  # metadata
  f.metadata = []
  f.metadata.append (('Last modification date/time', clf.last_modification_time))
  f.metadata.append (('Uploads started', clf.uploads_started))
  f.metadata.append (('Rating', clf.rating))
  f.metadata.append (('Comments', clf.comments))
  f.metadata.append (('Share tags', clf.share_tags))
  f.metadata += CXMLElement.get_metadata_from_pxml (clf.pxml)

  return f

 # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Create remote file from File and Source
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def new_remote_file (f, source, username):
  rf = dataholder ()
  rf.name = f.name
  rf.size = f.size
  rf.url = source.url
  rf.timestamp = source.timestamp
  rf.username = username
  rf.app_id = 'shareaza'
  rf.app = 'Shareaza'
  rf.peer_ip = source.ip
  rf.hashes = f.hashes

  # peer data
  peer = dataholder ()
  peer.ip = source.ip
  peer.port = 0
  peer.nickname = ''
  peer.country = ''
  peer.application = ''
  rf.peer = peer

  # tags
  rf.tags = set ()

  # metadata
  rf.metadata = []
  rf.metadata.append (('Rating', f.rating))
  rf.metadata.append (('Comments', f.comments))
  rf.metadata.append (('Share Tags', f.share_tags))
  rf.metadata += CXMLElement.get_metadata_from_pxml (f.pxml)

  return rf
