# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import pymobius.mediator
import mobius
import inspect

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mediator = pymobius.mediator.Mediator ()	# global mediator

STARTTIME = None
DEBUG = False

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Generic data holder
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Data (object):
  pass

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Generate data dump
# @param s data
# @return data dump in hex and chars
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def dump (s):
  i = 0
  l = len (s)
  text = ''
  
  while i < l:
    data = s[i : min (i + 16, l)]
    text += '%08x  ' % i
    text += ' '.join ('%02x' % ord (c) for c in data[:8])
    text += '  '
    text += ' '.join ('%02x' % ord (c) for c in data[8:])
    text += '   ' * (16 - len (data))
    text += '  '
    text += ''.join (c if 31 < ord (c) < 127 else '.' for c in data)
    text += '\n'
    i += 16

  return text

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Convert generic ID into name
# @param i ID
# @return Name
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def id_to_name (i):
  return i.replace ('_', ' ').replace ('-', ' ').capitalize ()

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Convert value to bytes
# @param value
# @return Value as byte sequence
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def to_bytes (value):
  if value is None:
    return bytes ()

  elif isinstance (value, str):
    return bytes (value)

  elif isinstance (value, unicode):
    return value.encode ('utf-8')

  return bytes (value)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Convert value to string
# @param value
# @return value as string
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def to_string (value):
  if value is None:
    return ''

  elif isinstance (value, str):
    return value

  elif isinstance (value, unicode):
    return value.encode ('utf-8')

  else:
    return str (value)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Convert object to dict
# @param obj Object
# @return dict
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def object_to_dict (obj):
  attrs = {}

  for name, value in inspect.getmembers (obj):
    if not callable (value) and not name.startswith ('__'):
      attrs[name] = value
      
  return attrs

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Create object from dict
# @param d Dict
# @return Object
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def object_from_dict (d):
  obj = Data ()
    
  for name, value in d.items ():
    setattr (obj, name, value)

  return obj	

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Add file to item's kff ignored file
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def set_handled (item, f):

  # create file reader
  if not f:
    return

  reader = f.new_reader ()
  if not reader:
    return

  # calculate hash sha2-512
  h = mobius.crypt.hash ('sha2_512')
  data = reader.read (65536)

  while data:
    h.update (data)
    data = reader.read (65536)

  # add to kff file
  case = item.case
  path = case.create_path ('hashset/%04d-handled.ignore' % item.uid)
  fp = open (path, 'a')
  fp.write ('%s\n' % h.get_hex_digest ())
  fp.close ()
