# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import pymobius
import pymobius.vfs
import mobius
import datetime
import traceback

ANT_ID = 'trash-can-entries'
ANT_NAME = 'Trash Can entries'
ANT_VERSION = '1.0'

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Ant: trash-can
# @author Eduardo Aguiar
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Ant (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, item):
    self.id = ANT_ID
    self.name = ANT_NAME
    self.version = ANT_VERSION
    self.__item = item

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Run ant
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def run (self):
    if not self.__item.datasource:
      return

    self.__entries = []
    self.__retrieve_win7 ()
    self.__save_data ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from Vista/Win7/Win8/Win10 OS
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_win7 (self):
    vfs = pymobius.vfs.get_item_vfs (self.__item)
    
    for root_folder in vfs:
      recycle_bin_folder = root_folder.get_child_by_name ('$RECYCLE.BIN', False)

      if recycle_bin_folder:
        self.__retrieve_win7_process_recycle_bin_folder (recycle_bin_folder)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from $RECYCLE_BIN folder
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_win7_process_recycle_bin_folder (self, folder):
    for child in folder.get_children ():
      if child.is_folder () and child.name.startswith (u'S-'):
        self.__retrieve_win7_process_sid_folder (child)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from <SID> subfolder
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_win7_process_sid_folder (self, folder):
    for child in folder.get_children ():
      if child.name.startswith (u'$I') and child.is_file () and not child.is_deleted ():
        self.__retrieve_win7_process_info_file (child, folder)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from $I info file
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_win7_process_info_file (self, f, sid_folder):
    try:
      decoder = mobius.decoder.data_decoder (f.new_reader ())
      rname = '$R' + f.name[2:]

      e = pymobius.Data ()
      version = decoder.get_uint64_le ()
      e.name = f.name
      e.size = decoder.get_uint64_le ()
      e.deletion_time = decoder.get_nt_datetime ()

      e.metadata = mobius.pod.map ()
      e.metadata.set ('version', version)
      e.metadata.set ('sid', sid_folder.name)
      e.metadata.set ('info-file-path', f.path)
      e.metadata.set ('restore-file-name', rname)
      
      path_size = decoder.get_uint32_le ()
      e.path = decoder.get_string_by_size (path_size * 2, 'UTF-16LE')

      r = sid_folder.get_child_by_name (rname)

      if r and r.exists ():
        e.metadata.set ('restore-file-path', r.path)
        e.metadata.set ('restore-file-size', r.size)
        e.metadata.set ('restore-is-available', True)

      else:
        e.metadata.set ('restore-is-available', False)

      self.__entries.append (e)
    except Exception, e:
      mobius.core.logf ('WRN ' + str (e) + ' ' + traceback.format_exc ())

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save data into model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __save_data (self):
    case = self.__item.case
    transaction = case.new_transaction ()

    # remove old data
    self.__item.remove_trash_can_entries ()

    # save entries
    for e in self.__entries:
      entry = self.__item.new_trash_can_entry (e.path, e.size)
      entry.name = e.name
      entry.deletion_time = e.deletion_time
      entry.metadata = e.metadata

    # set ant run
    self.__item.set_ant (ANT_ID, ANT_NAME, ANT_VERSION)
    transaction.commit ()
