// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "sector_reader_impl.h"
#include "imagefile_impl.h"
#include <mobius/io/file.h>
#include <mobius/exception.inc>
#include <stdexcept>

namespace mobius
{
namespace imagefile
{
namespace raw
{
namespace
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Constants
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint64_t CHUNK_SIZE = 64 * 1024;	// 64 Kb

} // namespace

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create sector_reader from imagefile_impl
//! \param impl imagefile_impl object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
sector_reader_impl::sector_reader_impl (const imagefile_impl& impl)
 : sectors_ (impl.get_sectors ()),
   sector_size_ (impl.get_sector_size ()),
   chunk_idx_ (impl.get_size () / CHUNK_SIZE + 1)
{
  if (sector_size_ > CHUNK_SIZE)
    throw std::invalid_argument (MOBIUS_EXCEPTION_MSG ("sector_size is greater than CHUNK_SIZE"));

  if (CHUNK_SIZE % sector_size_)
    throw std::invalid_argument (MOBIUS_EXCEPTION_MSG ("sector_size is not divisor of CHUNK_SIZE"));

  auto f = impl.get_file ();
  stream_ = f.new_reader ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Check if sector is available
//! \param sector Sector
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
sector_reader_impl::is_available (sector_type sector)
{
  return bool (sector < sectors_);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Read sector
//! \param sector Sector
//! \return Data from sector
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::bytearray
sector_reader_impl::read (sector_type sector)
{
  if (sector >= sectors_)
    return mobius::bytearray ();

  // read data chunk, if necessary
  auto pos = sector * sector_size_;
  auto chunk_idx = pos / CHUNK_SIZE;
  
  if (chunk_idx != chunk_idx_)
    {
      stream_.seek (chunk_idx * CHUNK_SIZE);
      chunk_data_ = stream_.read (CHUNK_SIZE);
      chunk_idx_ = chunk_idx;
    }

  // read sector data from current data chunk
  auto sector_pos = pos % CHUNK_SIZE;

  if (sector_pos < chunk_data_.size ())
    return (chunk_data_.slice (sector_pos, sector_pos + sector_size_ - 1));
  
  return mobius::bytearray (sector_size_);
}

} // namespace raw
} // namespace imagefile
} // namespace mobius
