// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief C++ API module wrapper
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "imagefile.h"
#include "io/reader.h"
#include "io/writer.h"
#include "api.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief tp_new (default constructor)
//! \param type type object
//! \param args argument list
//! \param kwds keywords dict
//! \return new application object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_new (PyTypeObject *type, PyObject *args, PyObject *)
{
  const char *arg_url = nullptr;
  const char *arg_type = "autodetect";

  if (!PyArg_ParseTuple (args, "|ss", &arg_url, &arg_type))
    return nullptr;

  PyObject *ret = type->tp_alloc (type, 0);

  if (ret)
    {
      try
        {
          if (arg_url)
            ((imagefile_imagefile_o *) ret)->obj = new mobius::imagefile::imagefile (arg_url, arg_type);

          else
            ((imagefile_imagefile_o *) ret)->obj = new mobius::imagefile::imagefile ();
        }
      catch (const std::runtime_error& e)
        {
          Py_DECREF (ret);
          PyErr_SetString (PyExc_IOError, e.what ());
          ret = nullptr;
        }
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief tp_dealloc
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
tp_dealloc (imagefile_imagefile_o *self)
{
  delete self->obj;
  self->ob_type->tp_free ((PyObject*) self);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief is_available getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_is_available (imagefile_imagefile_o *self)
{
  return PyBool_FromLong (self->obj->is_available ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief is_valid getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_is_valid (imagefile_imagefile_o *self)
{
  return PyBool_FromLong (self->obj->operator bool ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief url getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_url (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_url ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief type getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_type (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_type ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief size getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_size (imagefile_imagefile_o *self)
{
  return PyInt_FromLong (self->obj->get_size ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief size setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
tp_setter_size (imagefile_imagefile_o *self, PyObject *value, void *)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'size' attribute");
      return -1;
    }

  if (!PyInt_Check (value) && !PyLong_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'size' attribute");
      return -1;
    }

  // set size
  std::uint64_t arg_size = PyInt_AsUnsignedLongLongMask (value);
  self->obj->set_size (arg_size);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief sectors getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_sectors (imagefile_imagefile_o *self)
{
  return PyInt_FromLong (self->obj->get_sectors ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief sectors setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
tp_setter_sectors (imagefile_imagefile_o *self, PyObject *value, void *)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'sectors' attribute");
      return -1;
    }

  if (!PyInt_Check (value) && !PyLong_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'sectors' attribute");
      return -1;
    }

  // set sectors
  std::uint64_t arg_sectors = PyInt_AsUnsignedLongLongMask (value);
  self->obj->set_sectors (arg_sectors);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief sector_size getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_sector_size (imagefile_imagefile_o *self)
{
  return PyInt_FromLong (self->obj->get_sector_size ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief sector_size setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
tp_setter_sector_size (imagefile_imagefile_o *self, PyObject *value, void *)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'sector_size' attribute");
      return -1;
    }

  if (!PyInt_Check (value) && !PyLong_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'sector_size' attribute");
      return -1;
    }

  // set sector_size
  std::uint64_t arg_sector_size = PyInt_AsUnsignedLongLongMask (value);
  self->obj->set_sector_size (arg_sector_size);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief segments getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_segments (imagefile_imagefile_o *self)
{
  return PyInt_FromLong (self->obj->get_segments ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief segments setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
tp_setter_segments (imagefile_imagefile_o *self, PyObject *value, void *)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'segments' attribute");
      return -1;
    }

  if (!PyInt_Check (value) && !PyLong_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'segments' attribute");
      return -1;
    }

  // set segments
  std::uint64_t arg_segments = PyInt_AsUnsignedLongLongMask (value);
  self->obj->set_segments (arg_segments);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief segment_size getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_segment_size (imagefile_imagefile_o *self)
{
  return PyInt_FromLong (self->obj->get_segment_size ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief segment_size setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
tp_setter_segment_size (imagefile_imagefile_o *self, PyObject *value, void *)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'segment_size' attribute");
      return -1;
    }

  if (!PyInt_Check (value) && !PyLong_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'segment_size' attribute");
      return -1;
    }

  // set segment_size
  std::uint64_t arg_segment_size = PyInt_AsUnsignedLongLongMask (value);
  self->obj->set_segment_size (arg_segment_size);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief drive_vendor getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_drive_vendor (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_drive_vendor ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief drive_vendor setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
tp_setter_drive_vendor (imagefile_imagefile_o *self, PyObject *value, void *)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'drive_vendor' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'drive_vendor' attribute");
      return -1;
    }

  // set drive_model
  const char *arg_drive_vendor = PyString_AsString (value);
  self->obj->set_drive_vendor (arg_drive_vendor);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief drive_model getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_drive_model (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_drive_model ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief drive_model setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
tp_setter_drive_model (imagefile_imagefile_o *self, PyObject *value, void *)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'drive_model' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'drive_model' attribute");
      return -1;
    }

  // set drive_model
  const char *arg_drive_model = PyString_AsString (value);
  self->obj->set_drive_model (arg_drive_model);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief drive_serial_number getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_drive_serial_number (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_drive_serial_number ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief drive_serial_number setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
tp_setter_drive_serial_number (imagefile_imagefile_o *self, PyObject *value, void *)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'drive_serial_number' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'drive_serial_number' attribute");
      return -1;
    }

  // set drive_serial_number
  const char *arg_drive_serial_number = PyString_AsString (value);
  self->obj->set_drive_serial_number (arg_drive_serial_number);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief acquisition_user getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_acquisition_user (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_acquisition_user ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief acquisition_user setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
tp_setter_acquisition_user (imagefile_imagefile_o *self, PyObject *value, void *)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'acquisition_user' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'acquisition_user' attribute");
      return -1;
    }

  // set acquisition_user
  const char *arg_acquisition_user = PyString_AsString (value);
  self->obj->set_acquisition_user (arg_acquisition_user);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief acquisition_time getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_acquisition_time (imagefile_imagefile_o *self)
{
  return PyDateTime_from_datetime (self->obj->get_acquisition_time ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief acquisition_time setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
tp_setter_acquisition_time (imagefile_imagefile_o *self, PyObject *value, void *)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'acquisition_time' attribute");
      return -1;
    }

  if (!check_PyDateTime (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'acquisition_time' attribute");
      return -1;
    }

  // set acquisition_time
  self->obj->set_acquisition_time (PyDateTime_as_datetime (value));

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief acquisition_tool getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_acquisition_tool (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_acquisition_tool ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief acquisition_tool setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
tp_setter_acquisition_tool (imagefile_imagefile_o *self, PyObject *value, void *)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'acquisition_tool' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'acquisition_tool' attribute");
      return -1;
    }

  // set acquisition_tool
  const char *arg_acquisition_tool = PyString_AsString (value);
  self->obj->set_acquisition_tool (arg_acquisition_tool);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief acquisition_platform getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_acquisition_platform (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_acquisition_platform ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief acquisition_platform setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
tp_setter_acquisition_platform (imagefile_imagefile_o *self, PyObject *value, void *)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'acquisition_platform' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'acquisition_platform' attribute");
      return -1;
    }

  // set acquisition_platform
  const char *arg_acquisition_platform = PyString_AsString (value);
  self->obj->set_acquisition_platform (arg_acquisition_platform);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief hash_md5 getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_hash_md5 (imagefile_imagefile_o *self)
{
  return PyString_FromString (self->obj->get_hash_md5 ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief hash_md5 setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
tp_setter_hash_md5 (imagefile_imagefile_o *self, PyObject *value, void *)
{
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'hash_md5' attribute");
      return -1;
    }

  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'hash_md5' attribute");
      return -1;
    }

  // set hash_md5
  const char *arg_hash_md5 = PyString_AsString (value);
  self->obj->set_hash_md5 (arg_hash_md5);

  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief new_reader method
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_new_reader (imagefile_imagefile_o *self, PyObject *)
{
  return PyMobius_IO_Reader_from_cpp (self->obj->new_reader ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief api.new_writer
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_new_writer (imagefile_imagefile_o *self, PyObject *)
{
  return PyMobius_IO_Writer_from_cpp (self->obj->new_writer ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief getters and setters structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyGetSetDef tp_getset[] =
{
  {
    (char *) "is_available",
    (getter) tp_getter_is_available,
    (setter) 0,
    (char *) "check if imagefile is available", NULL
  },
  {
    (char *) "is_valid",
    (getter) tp_getter_is_valid,
    (setter) 0,
    (char *) "check if imagefile is valid", NULL
  },
  {
    (char *) "url",
    (getter) tp_getter_url,
    (setter) 0,
    (char *) "imagefile URL", NULL
  },
  {
    (char *) "type",
    (getter) tp_getter_type,
    (setter) 0,
    (char *) "imagefile type", NULL
  },
  {
    (char *) "size",
    (getter) tp_getter_size,
    (setter) tp_setter_size,
    (char *) "imagefile size in bytes", NULL
  },
  {
    (char *) "sectors",
    (getter) tp_getter_sectors,
    (setter) tp_setter_sectors,
    (char *) "number of sectors", NULL
  },
  {
    (char *) "sector_size",
    (getter) tp_getter_sector_size,
    (setter) tp_setter_sector_size,
    (char *) "sector size in bytes", NULL
  },
  {
    (char *) "segments",
    (getter) tp_getter_segments,
    (setter) tp_setter_segments,
    (char *) "number of file segments", NULL
  },
  {
    (char *) "segment_size",
    (getter) tp_getter_segment_size,
    (setter) tp_setter_segment_size,
    (char *) "segment size in bytes", NULL
  },
  {
    (char *) "drive_vendor",
    (getter) tp_getter_drive_vendor,
    (setter) tp_setter_drive_vendor,
    (char *) "drive vendor name", NULL
  },
  {
    (char *) "drive_model",
    (getter) tp_getter_drive_model,
    (setter) tp_setter_drive_model,
    (char *) "drive model", NULL
  },
  {
    (char *) "drive_serial_number",
    (getter) tp_getter_drive_serial_number,
    (setter) tp_setter_drive_serial_number,
    (char *) "drive serial number", NULL
  },
  {
    (char *) "acquisition_user",
    (getter) tp_getter_acquisition_user,
    (setter) tp_setter_acquisition_user,
    (char *) "acquisition user name", NULL
  },
  {
    (char *) "acquisition_time",
    (getter) tp_getter_acquisition_time,
    (setter) tp_setter_acquisition_time,
    (char *) "acquisition date and time", NULL
  },
  {
    (char *) "acquisition_tool",
    (getter) tp_getter_acquisition_tool,
    (setter) tp_setter_acquisition_tool,
    (char *) "acquisition tool", NULL
  },
  {
    (char *) "acquisition_platform",
    (getter) tp_getter_acquisition_platform,
    (setter) tp_setter_acquisition_platform,
    (char *) "acquisition platform", NULL
  },
  {
    (char *) "hash_md5",
    (getter) tp_getter_hash_md5,
    (setter) tp_setter_hash_md5,
    (char *) "MD5 hash", NULL
  },
  {NULL, NULL, NULL, NULL, NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef tp_methods[] =
{
  {
    (char *) "new_reader",
    (PyCFunction) tp_f_new_reader,
    METH_VARARGS,
    "create new reader"
  },
  {
    (char *) "new_writer",
    (PyCFunction) tp_f_new_writer,
    METH_VARARGS,
    "create new writer"
  },
  {NULL, NULL, 0, NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief imagefle: type structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyTypeObject imagefile_imagefile_t =
{
  PyVarObject_HEAD_INIT (NULL, 0)
  "mobius.imagefile.imagefile",               		// tp_name
  sizeof (imagefile_imagefile_o),              		// tp_basicsize
  0,                                       		// tp_itemsize
  (destructor) tp_dealloc,  	                   	// tp_dealloc
  0,                                       		// tp_print
  0,                                       		// tp_getattr
  0,                                       		// tp_setattr
  0,                                       		// tp_compare
  0,                                       		// tp_repr
  0,                                       		// tp_as_number
  0,                                       		// tp_as_sequence
  0,                                       		// tp_as_mapping
  0,                                       		// tp_hash
  0,                                       		// tp_call
  0,                                       		// tp_str
  0,                                       		// tp_getattro
  0,                                       		// tp_setattro
  0,                                       		// tp_as_buffer
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,		// tp_flags
  "imagefile class",                     		// tp_doc
  0,                                       		// tp_traverse
  0,                                       		// tp_clear
  0,                                       		// tp_richcompare
  0,                                       		// tp_weaklistoffset
  0,                                       		// tp_iter
  0,                                       		// tp_iternext
  tp_methods,                              		// tp_methods
  0,                                       		// tp_members
  tp_getset,                	                   	// tp_getset
  0,                             	              	// tp_base
  0,                                       		// tp_dict
  0,                                       		// tp_descr_get
  0,                                       		// tp_descr_set
  0,                                       		// tp_dictoffset
  0,                                       		// tp_init
  0,                                       		// tp_alloc
  tp_new,                                  		// tp_new
  0,                                       		// tp_free
  0,                                       		// tp_is_gc
  0,                                       		// tp_bases
  0,                                       		// tp_mro
  0,                                       		// tp_cache
  0,                                       		// tp_subclasses
  0,                                       		// tp_weaklist
  0,                                       		// tp_del
  0,                                       		// tp_version_tag
};
