// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/core/application.h>
#include <mobius/io/file.h>
#include <mobius/io/folder.h>
#include <unistd.h>
#include <iostream>
#include <iomanip>

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief show file info
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
show_file (const mobius::io::file& f, int level = 0)
{
  std::cout << std::endl;
  const std::string indent (level * 2, ' ');

  if (!f || !f.exists ())
    {
      std::cout << indent << "? " << f.get_url () << std::endl;
      return;
    }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // show metadata
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::cout << indent << ". " << f.get_url () << std::endl;
  std::cout << indent << "  Name: " << f.get_name () << std::endl;
  std::cout << indent << "  Short name: " << f.get_short_name () << std::endl;
  std::cout << indent << "  Inode: " << f.get_inode () << std::endl;
  std::cout << indent << "  Is deleted: " << std::boolalpha << f.is_deleted () << std::endl;
  std::cout << indent << "  Is reallocated: " << std::boolalpha << f.is_reallocated () << std::endl;
  std::cout << indent << "  Is hidden: " << std::boolalpha << f.is_hidden () << std::endl;
  std::cout << indent << "  Is regular file: " << std::boolalpha << f.is_regular_file () << std::endl;
  std::cout << indent << "  Size: " << f.get_size () << " bytes" << std::endl;
  std::cout << indent << "  User: " << f.get_user_name () << " (" << f.get_user_id () << ')' << std::endl;
  std::cout << indent << "  Group: " << f.get_group_name () << " (" << f.get_group_id () << ')' << std::endl;
  std::cout << indent << "  Permissions: " << std::oct << f.get_permissions () << std::dec << std::endl;
  std::cout << indent << "  Last access time (atime): " << f.get_access_time () << std::endl;
  std::cout << indent << "  Last modification time (mtime): " << f.get_modification_time () << std::endl;
  std::cout << indent << "  Last metadata time (ctime): " << f.get_metadata_time () << std::endl;
  std::cout << indent << "  Creation time: " << f.get_creation_time () << std::endl;
  std::cout << indent << "  Deletion time: " << f.get_deletion_time () << std::endl;
  std::cout << indent << "  Backup time: " << f.get_backup_time () << std::endl;

  auto parent = f.get_parent ();
  std::cout << indent << "  Parent folder: " << parent.get_url () << std::endl;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // show some bytes...
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  if (f.is_regular_file ())
    {
      try
        {
          auto reader = f.new_reader ();
          auto data = reader.read (16);

          std::cout << indent << "  First 16 bytes:";
          for (auto b : data)
            std::cout << ' ' << std::hex << int (b) << std::dec;
          std::cout << std::endl;
        }
      catch (const std::runtime_error& e)
        {
          std::cerr << "Warning: " << e.what () << std::endl;
        }
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief show folder info
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
show_folder (const mobius::io::folder& folder, int level = 0)
{
  std::cout << std::endl;
  const std::string indent (level * 2, ' ');

  if (!folder || !folder.exists ())
    {
      std::cout << indent << "? " << folder.get_url () << std::endl;
      return;
    }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // show metadata
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::cout << indent << "+ " << folder.get_url () << std::endl;
  std::cout << indent << "  User: " << folder.get_user_name () << " (" << folder.get_user_id () << ')' << std::endl;
  std::cout << indent << "  Group: " << folder.get_group_name () << " (" << folder.get_group_id () << ')' << std::endl;
  std::cout << indent << "  Permissions: " << std::oct << folder.get_permissions () << std::dec << std::endl;
  std::cout << indent << "  Last access time (atime): " << folder.get_access_time () << std::endl;
  std::cout << indent << "  Last modification time (mtime): " << folder.get_modification_time () << std::endl;
  std::cout << indent << "  Last metadata time (ctime): " << folder.get_metadata_time () << std::endl;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // handle entries
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  try
    {
      for (const auto& entry: folder.get_children ())
        {
          if (entry.is_folder ())
            show_folder (entry.get_folder (), level + 1);

          else if (entry.is_file ())
            show_file (entry.get_file (), level + 1);
        }
    }
  catch (const std::runtime_error& e)
    {
      std::cerr << "Warning: " << e.what () << std::endl;
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief show usage text
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
usage ()
{
  std::cerr << std::endl;
  std::cerr << "use: dirtree [OPTIONS] <url1> [url2] ..." << std::endl;
  std::cerr << "e.g: dirtree file:///etc" << std::endl;
  std::cerr << std::endl;
  //std::cerr << "options are:" << std::endl;
  //std::cerr << "  -o offset\tstarting sector of the filesystem" << std::endl;
  //std::cerr << std::endl;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief main function
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
int
main (int argc, char **argv)
{
  mobius::core::application app;
  std::cerr << app.get_name () << " v" << app.get_version () << std::endl;
  std::cerr << app.get_copyright () << std::endl;
  std::cerr << "DirTree v1.2" << std::endl;
  std::cerr << "by Eduardo Aguiar" << std::endl;
  std::cerr << std::endl;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // process command line
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  int opt;

  while ((opt = getopt (argc, argv, "")) != EOF)
    {
      /*if (opt == 'o')
        {
          offset = atol (optarg) * 512;
          break;
        }
      else
        {
          //std::cerr << "Error: Invalid option '-" << char (opt) << "'" << std::endl;
          usage ();
          exit (EXIT_FAILURE);
        }*/
    }

  if (optind == argc)
    {
      std::cerr << "Error: You must pass a valid path" << std::endl;
      usage ();
      exit (EXIT_FAILURE);
    }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // show resources
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  while (optind < argc)
    {
      auto folder = mobius::io::new_folder_by_path (argv[optind]);

      try
        {
          show_folder (folder);
        }
      catch (const std::exception& e)
        {
          std::cerr <<  "Error: " << e.what () << std::endl;
          exit (EXIT_FAILURE);
        }

      optind++;
    }

  return EXIT_SUCCESS;
}
