// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief  C++ API module wrapper
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <pymobius.h>
#include "file.h"
#include "folder.h"
#include "path.h"
#include "reader.h"
#include "sector_reader.h"
#include "writer.h"
#include "uri.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief new_reader_from_sector_reader
//! \param self function object
//! \param args argument list
//! \return Python object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
func_new_reader_from_sector_reader (PyObject *, PyObject *args)
{
  // parse input args
  io_sector_reader_o * arg_sector_reader;

  if (!PyArg_ParseTuple (args, "O!", &io_sector_reader_t, &arg_sector_reader))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyMobius_IO_Reader_from_cpp (
              mobius::io::new_reader_from_sector_reader (* (arg_sector_reader->obj))
            );
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Module methods
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *func_get_current_folder (PyObject *, PyObject *);
PyObject *func_join_path (PyObject *, PyObject *);
PyObject *func_new_entry_by_path (PyObject *, PyObject *);
PyObject *func_new_entry_by_url (PyObject *, PyObject *);
PyObject *func_new_file_by_path (PyObject *, PyObject *);
PyObject *func_new_file_by_url (PyObject *, PyObject *);
PyObject *func_new_folder_by_path (PyObject *, PyObject *);
PyObject *func_new_folder_by_url (PyObject *, PyObject *);
PyObject *func_new_path_from_win (PyObject *, PyObject *);
PyObject *func_new_uri_from_path (PyObject *, PyObject *);
PyObject *func_to_win_path (PyObject *, PyObject *);

static PyMethodDef module_methods[] =
{
  {
    "get_current_folder",
    func_get_current_folder,
    METH_VARARGS,
    "Get current folder"
  },
  {
    "join_path",
    func_join_path,
    METH_VARARGS,
    "Join paths"
  },
  {
    "new_entry_by_path",
    func_new_entry_by_path,
    METH_VARARGS,
    "Create entry by path"
  },
  {
    "new_entry_by_url",
    func_new_entry_by_url,
    METH_VARARGS,
    "Create entry by URL"
  },
  {
    "new_file_by_path",
    func_new_file_by_path,
    METH_VARARGS,
    "Create file by path"
  },
  {
    "new_file_by_url",
    func_new_file_by_url,
    METH_VARARGS,
    "Create file by URL"
  },
  {
    "new_folder_by_path",
    func_new_folder_by_path,
    METH_VARARGS,
    "Create folder by path"
  },
  {
    "new_folder_by_url",
    func_new_folder_by_url,
    METH_VARARGS,
    "Create folder by URL"
  },
  {
    "new_path_from_win",
    func_new_path_from_win,
    METH_VARARGS,
    "Create path from Win path"
  },
  {
    "new_uri_from_path",
    func_new_uri_from_path,
    METH_VARARGS,
    "Create URI from path"
  },
  {
    "new_reader_from_sector_reader",
    func_new_reader_from_sector_reader,
    METH_VARARGS,
    "Create reader from sector_reader object"
  },
  {
    "to_win_path",
    func_to_win_path,
    METH_VARARGS,
    "Convert path to Win path string"
  },
  {
    NULL,
    NULL,
    0,
    NULL
  } // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Module datatypes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr struct
{
  const char *name;
  PyTypeObject *type;
} types[] =

{
  {"file", &io_file_t},
  {"folder", &io_folder_t},
  {"path", &io_path_t},
  {"reader", &io_reader_t},
  {"sector_reader", &io_sector_reader_t},
  {"writer", &io_writer_t},
  {"uri", &io_uri_t},
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create module
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
new_io_module ()
{
  // initialize module
  PyObject* module = Py_InitModule3 (
                       "mobius.io",
                       module_methods,
                       "Mobius Forensic Toolkit mobius.io module"
                     );

  // build types
  for (const auto& t : types)
    {
      if (PyType_Ready (t.type) < 0)
        return nullptr;
    }

  // add types
  for (const auto& t : types)
    {
      Py_INCREF (t.type);
      PyModule_AddObject (module, t.name, (PyObject *) t.type);
    }

  // return module
  return module;
}
