# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import pymobius.app.chrome
import pymobius.app.firefox
import pymobius.app.internet_explorer
import pymobius

ATTR_NAME = 'data.cookies'

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Model class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class model (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, item):
    self.__item = item

    if item.has_attribute (ATTR_NAME):
      self.__entries = item.get_attribute (ATTR_NAME)
      self.__entries_loaded = True

    else:
      self.__entries = []
      self.__entries_loaded = False

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get data from model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get (self):
    if not self.__entries_loaded:
      self.__retrieve_entries ()
      self.__item.set_attribute (ATTR_NAME, self.__entries)
      self.__entries_loaded = True

    return self.__entries

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get data from model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def refresh (self):
    self.__item.remove_attribute (ATTR_NAME)
    self.__entries = []
    self.__entries_loaded = False

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_entries (self):
    self.__retrieve_chrome ()
    self.__retrieve_firefox ()
    self.__retrieve_internet_explorer ()
    
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from Google Chrome
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_chrome (self):
    model = pymobius.app.chrome.model (self.__item)
    
    for profile in model.get_profiles ():
      for entry in profile.get_cookies ():
        cookie = pymobius.Data ()
        cookie.username = profile.username
        cookie.itemname = self.__item.get_attribute ('name')
        cookie.name = entry.name
        cookie.value = entry.value
        cookie.domain = entry.domain
        cookie.creation_time = entry.creation_time
        cookie.profile_id = profile.name
        cookie.profile_path = profile.path
        cookie.app_id = 'chrome'
        cookie.app_name = 'Google Chrome'

        cookie.metadata = []
        cookie.metadata.append (('Last access date/time', str (entry.last_access_time)[:19]))

        self.__entries.append (cookie)
        
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from Mozilla Firefox
  # @see http://doxygen.db48x.net/mozilla/html/interfacensIDownloadManager.html
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_firefox (self):
    model = pymobius.app.firefox.model (self.__item)
    
    for profile in model.get_profiles ():
      for entry in profile.get_cookies ():
        cookie = pymobius.Data ()
        cookie.username = profile.username
        cookie.itemname = self.__item.get_attribute ('name')
        cookie.name = entry.name
        cookie.value = entry.value
        cookie.domain = entry.domain
        cookie.creation_time = entry.creation_time
        cookie.profile_id = profile.name
        cookie.profile_path = profile.path
        cookie.app_id = 'firefox'
        cookie.app_name = 'Mozilla Firefox'

        cookie.metadata = []
        cookie.metadata.append (('Last access date/time', str (entry.last_accessed_time)[:19]))

        self.__entries.append (cookie)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from Internet Explorer
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_internet_explorer (self):
    model = pymobius.app.internet_explorer.model (self.__item)
    
    for profile in model.get_profiles ():
      for entry in profile.get_cookies ():
        cookie = pymobius.Data ()
        cookie.username = profile.username
        cookie.itemname = self.__item.get_attribute ('name')
        cookie.name = entry.name
        cookie.value = entry.value
        cookie.domain = entry.domain
        cookie.creation_time = entry.creation_time
        cookie.profile_id = profile.name
        cookie.profile_path = profile.path
        cookie.app_id = 'internet_explorer'
        cookie.app_name = 'Internet Explorer'

        cookie.metadata = []
        cookie.metadata.append (('Expiration date/time', str (entry.expiration_time)[:19]))

        self.__entries.append (cookie)
