# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import os
import os.path
import tempfile
import sqlite3
import pymobius
import mobius

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Skype Profile class
# @author Eduardo Aguiar
# @see https://arxiv.org/pdf/1603.05369.pdf
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Profile (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, folder, item):
    self.__folder = folder
    self.name = folder.name
    self.path = folder.path.replace ('/', '\\')
    self.__item = item
    self.__main_db_path = None

    # set attributes
    self.__file_transfers_loaded = False
    self.__file_transfers = []

    # retrieve main.db
    f = folder.get_child_by_name ('main.db')
    if not f:
      return

    reader = f.new_reader ()
    if not reader:
      return

    # create temporary .sqlite local file
    ext = os.path.splitext (f.name)[1]
    fd, path = tempfile.mkstemp (suffix=ext)

    fp = open (path, 'w')
    fp.write (reader.read ())
    fp.close ()

    self.__main_db_path = path
    import shutil
    shutil.copy (path, '/tmp/skype/main-%s.db' % self.name)

    # get account data
    db = sqlite3.connect (self.__main_db_path)

    cursor = db.execute ('''
       SELECT skypename,
              fullname
         FROM accounts
        WHERE id = 1''')

    row = cursor.fetchone ()

    if row:
      self.account_id = row[0]
      self.account_name = row[1]
      
    else:
      self.account_id = None
      self.account_name = None

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Destroy object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __del__ (self):
    if self.__main_db_path:
      os.remove (self.__main_db_path)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get file transfers
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_file_transfers (self):
    if not self.__file_transfers_loaded:
      self.__load_file_transfers ()

    return self.__file_transfers

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load file transfers
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __load_file_transfers (self):
    db = sqlite3.connect (self.__main_db_path)

    SQL_STATEMENT = '''
       SELECT partner_handle,
              partner_dispname,
              status,
              starttime,
              finishtime,
              filepath,
              filename,
              filesize,
              bytestransferred,
              type
         FROM transfers'''
  
    for row in db.execute (SQL_STATEMENT):
      ft = pymobius.Data ()
      ft.status = row[2]
      ft.start_time = mobius.datetime.new_datetime_from_unix_timestamp (row[3])
      ft.finish_time = mobius.datetime.new_datetime_from_unix_timestamp (row[4])
      ft.path = row[5]
      ft.filename = row[6]
      ft.size = row[7]
      ft.bytes_transferred = row[8]
      ft.type = row[9]
      
      # set from/to accounts
      if ft.type == 1:	# receive
        ft.from_skype_account = row[0]
        ft.from_skype_name = row[1]
        ft.to_skype_account = self.account_id
        ft.to_skype_name = self.account_name
        
      elif ft.type == 2:		# send
        ft.from_skype_account = self.account_id
        ft.from_skype_name = self.account_name
        ft.to_skype_account = row[0]
        ft.to_skype_name = row[1]

      self.__file_transfers.append (ft)

    # end
    db.close ()
    self.__file_transfers_loaded = True
