// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/tsk/entry.h>
#include <mobius/tsk/exception.h>
#include <tsk/libtsk.h>
#include <string>
#include <ctime>
#include <mobius/tsk/exception.inc>

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief convert date timestamp to string representation
//! \param t UNIX timestamp
//! \param nano nanoseconds
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static const std::string
get_datetime (std::time_t t, uint32_t)
{
  if (t == 0)
    return std::string ();

  struct tm *tm = gmtime (&t);
  char buffer[256];
  strftime (buffer, sizeof (buffer), "%Y-%m-%d %T", tm);
  return std::string (buffer);

}

namespace mobius
{
namespace tsk
{

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief constructor
//!\param fs_dir_p pointer to TSK_FS_DIR structure
//!\param path base path for entries
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
entry_list::entry_list (TSK_FS_DIR *fs_dir_p, const std::string& path)
  : fs_dir_p_ (fs_dir_p), path_ (path)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief destructor
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
entry_list::~entry_list ()
{
  if (counter_.is_unique ())
    {
      tsk_fs_dir_close (fs_dir_p_);
      fs_dir_p_ = 0;
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get list size
//!\return number of entries in the list
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::size_t
entry_list::get_length () const
{
  unsigned int count = 0;

  if (fs_dir_p_)
    count = tsk_fs_dir_getsize (fs_dir_p_);

  return count;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get an entry from the list
//!\return entry
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const entry
entry_list::operator[] (std::size_t idx) const
{
  TSK_FS_FILE * fs_file_p = tsk_fs_dir_get (fs_dir_p_, idx);
  if (!fs_file_p)
    throw MOBIUS_TSK_EXCEPTION;

  mobius::tsk::entry entry (fs_file_p);
  entry.set_path (path_ + "/" + entry.get_name ());

  return entry;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief constructor
//!\param fs_file_p pointer to TSK_FS_FILE structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
entry::entry (TSK_FS_FILE *fs_file_p)
  : fs_file_p_ (fs_file_p),
    inode_ (0),
    is_deleted_ (false),
    is_reallocated_ (false),
    size_ (0),
    mode_ (0),
    uid_ (0),
    gid_ (0)
{
  __read_fs_name ();
  __read_fs_meta ();
  __read_fs_attr ();

  is_reallocated_ = is_deleted_ &&
                    fs_file_p->meta &&
                    (fs_file_p->meta->flags & TSK_FS_NAME_FLAG_ALLOC);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief entry destructor
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
entry::~entry ()
{
  if (counter_.is_unique ())
    {
      tsk_fs_file_close (fs_file_p_);
      fs_file_p_ = 0;
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief check if entry is a folder
//!\return true if entry is a folder, false otherwise
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
entry::is_folder () const
{
  return type_ == TSK_FS_META_TYPE_DIR;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief check if entry is deleted
//!\return true if entry is deleted, false otherwise
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
entry::is_deleted () const
{
  return is_deleted_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief check if entry is reallocated
//
//! An entry is reallocated if it is deleted and points to a data descriptor
//! used by another non-deleted entry.
//!\return true if entry is reallocated, false otherwise
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
entry::is_reallocated () const
{
  return is_reallocated_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get pointer to the internal structure
//!\return pointer to TSK_FS_FILE structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
TSK_FS_FILE *
entry::get_pointer ()
{
  return fs_file_p_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get entry inode
//!\return inode number
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inode_t
entry::get_inode () const
{
  return inode_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get entry type
//!\return type, as defined by TSK_FS_NAME_TYPE_ENUM
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
int
entry::get_type () const
{
  return type_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get entry size
//!\return size in bytes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
offset_t
entry::get_size () const
{
  return size_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get entry permission mode
//!\return permission mode as defined in Unix (777, 755, ...)
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
int
entry::get_mode () const
{
  return mode_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get user ID
//!\return user ID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
int
entry::get_uid () const
{
  return uid_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get group ID
//!\return group ID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
int
entry::get_gid () const
{
  return gid_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get entry name
//!\return name
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
entry::get_name () const
{
  return name_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get entry short name
//!\return short name
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
entry::get_short_name () const
{
  return short_name_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get last access date and time
//!\return datetime as a string (e.g. 2015-08-25 14:00:05)
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
entry::get_atime () const
{
  return atime_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get last data modification date and time
//!\return datetime as a string (e.g. 2015-08-25 14:00:05)
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
entry::get_mtime () const
{
  return mtime_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get last metadata modification date and time
//!\return datetime as a string (e.g. 2015-08-25 14:00:05)
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
entry::get_ctime () const
{
  return ctime_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get creation date and time
//!\return datetime as a string (e.g. 2015-08-25 14:00:05)
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
entry::get_crtime () const
{
  return crtime_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get deletion date and time
//!\return datetime as a string (e.g. 2015-08-25 14:00:05)
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
entry::get_dtime () const
{
  return dtime_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get number of streams of this entry
//!\return stream count
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::size_t
entry::get_stream_count () const
{
  return streams_.size ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get entry stream by idx
//!\param idx stream index, starting from 0
//!\return stream
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const stream
entry::get_stream_by_idx (std::size_t idx) const
{
  return streams_[idx];
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get the first stream with a given type
//!\param type stream type
//!\return stream
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const stream
entry::get_stream_by_type (int type) const
{
  for (auto s : streams_)
    {
      if (s.get_type () == type)
        return s;
    }

  return stream ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get entry's children
//!\return an entry_list containing children of this entry
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const entry_list
entry::get_children () const
{
  TSK_FS_DIR *fs_dir_p = 0;

  if (type_ == TSK_FS_META_TYPE_DIR)
    fs_dir_p = tsk_fs_dir_open_meta (fs_file_p_->fs_info, inode_);

  return entry_list (fs_dir_p, path_);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief get entry path
//!\return full path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
entry::get_path () const
{
  return path_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief set entry path
//!\param path new path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
entry::set_path (const std::string& path)
{
  path_ = path;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief read data from TSK_FS_NAME structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
entry::__read_fs_name ()
{
  if (fs_file_p_->name)
    {
      inode_ = fs_file_p_->name->meta_addr;
      is_deleted_ = fs_file_p_->name->flags & TSK_FS_NAME_FLAG_UNALLOC;
      type_ = fs_file_p_->name->type;

      if (fs_file_p_->name->name)
        name_ = fs_file_p_->name->name;

      if (fs_file_p_->name->shrt_name)
        short_name_ = fs_file_p_->name->shrt_name;
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief read data from TSK_FS_META structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
entry::__read_fs_meta ()
{
  if (!fs_file_p_->meta)
    {
      if (!inode_)
        return;

      int rc = fs_file_p_->fs_info->file_add_meta (fs_file_p_->fs_info, fs_file_p_, inode_);
      if (rc)
        throw MOBIUS_TSK_EXCEPTION;
    }

  // fill attributes
  if (fs_file_p_->meta)
    {
      inode_ = fs_file_p_->meta->addr;
      type_ = fs_file_p_->meta->type;
      size_ = fs_file_p_->meta->size;
      mode_ = fs_file_p_->meta->mode;
      uid_ = fs_file_p_->meta->uid;
      gid_ = fs_file_p_->meta->gid;

      // fill timestamps
      atime_ = get_datetime (fs_file_p_->meta->atime, fs_file_p_->meta->atime_nano);
      mtime_ = get_datetime (fs_file_p_->meta->mtime, fs_file_p_->meta->mtime_nano);
      ctime_ = get_datetime (fs_file_p_->meta->ctime, fs_file_p_->meta->ctime_nano);
      crtime_ = get_datetime (fs_file_p_->meta->crtime, fs_file_p_->meta->crtime_nano);

      if (fs_file_p_->fs_info->ftype & TSK_FS_TYPE_EXT_DETECT)
        dtime_ = get_datetime (fs_file_p_->meta->time2.ext2.dtime, fs_file_p_->meta->time2.ext2.dtime_nano);
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief read data from TSK_FS_ATTR structures
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
entry::__read_fs_attr ()
{
  std::size_t count = tsk_fs_file_attr_getsize (fs_file_p_);

  for (std::size_t i = 0; i < count; i++)
    {
      const TSK_FS_ATTR *fs_attr_p = tsk_fs_file_attr_get_idx (fs_file_p_, i);

      if (fs_attr_p)
        {
          stream stream (fs_attr_p, i);
          streams_.push_back (stream);
        }
    }
}

} // namespace tsk
} // namespace mobius
