// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/imagefile/imagefile_impl_split.h>
#include <mobius/imagefile/segment_array_base.h>
#include <mobius/io/file.h>
#include <mobius/io/reader.h>
#include <mobius/io/reader_impl_base.h>
#include <mobius/io/writer.h>
#include <mobius/io/writer_impl_base.h>
#include <mobius/uri.h>
#include <mobius/exception.inc>
#include <stdexcept>

namespace mobius
{
namespace imagefile
{
namespace
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief segment_array subclass
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=        
class segment_array_split : public segment_array_base
{
public:
  explicit segment_array_split (const std::string&);
  segment_array_split (const segment_array_split&) = default;
  segment_array_split (segment_array_split&&) = default;
  segment_array_split& operator= (const segment_array_split&) = default;
  segment_array_split& operator= (segment_array_split&&) = default;
  
private:
  virtual const std::string get_next_extension (const std::string&) const override;
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create segment_array_split
//! \param url URL
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
segment_array_split::segment_array_split (const std::string& url)
 : segment_array_base (url)
{
}
 
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief return next extension
//! \param extension
//! \return next extension
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::string
segment_array_split::get_next_extension (const std::string& extension) const
{
  std::string tmp_extension = extension;
  int pos = extension.length () - 1;
  bool carry = true;

  while (pos >= 0 && carry)
    {
      if (tmp_extension[pos] == '9')
        tmp_extension[pos] = '0';

      else if (tmp_extension[pos] == 'z')
        tmp_extension[pos] = 'a';

      else if (tmp_extension[pos] == 'Z')
        tmp_extension[pos] = 'A';

      else
        {
          ++tmp_extension[pos];
          carry = false;
        }

      if (carry)
        --pos;
    }

  return tmp_extension;
}

} // namespace

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief split imagefile reader implementation class
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class reader_impl_split : public mobius::io::reader_impl_base
{
public:
  explicit reader_impl_split (const imagefile_impl_split&);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief check if reader is seekable
  //! \return true/false
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  virtual bool
  is_seekable () const override
  {
    return true;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief check if <b>reader.get_size</b> is available
  //! \return true/false
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  virtual bool
  is_sizeable () const override
  {
    return true;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get data size
  //! \return data size in bytes
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  virtual size_type
  get_size () const override
  {
    return size_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get read position
  //! \return read position in bytes from the beginning of data
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  virtual offset_type
  tell () const override
  {
    return pos_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get read position
  //! \return read position in bytes from the beginning of data
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  virtual bool
  eof () const override
  {
    return pos_ >= size_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // virtual methods
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  virtual const mobius::bytearray read (size_type) override;
  virtual void seek (offset_type, whence_type = whence_type::beginning) override;

private:
  size_type size_;
  size_type segment_size_;
  size_type segment_idx_;       // current segment idx
  size_type pos_ = 0;
  mobius::io::reader stream_;
  segment_array_split segments_;

  void _set_stream ();
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief constructor
//! \param url URL
//! \param size imagefile size
//! \param segment_size segment size
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
reader_impl_split::reader_impl_split (const imagefile_impl_split& imagefile_impl)
  : size_ (imagefile_impl.get_size ()),
    segments_ (imagefile_impl.get_url ())
{
  segments_.scan ();
  
  if (segments_.get_size () > 0)
    {
      segment_size_ = segments_[0].get_size ();
      segment_idx_ = segments_.get_size ();
    }
  else
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("segment files not found"));
  
  _set_stream ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief set read position
//! \param offset offset in bytes
//! \param w either beginning, current or end
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
reader_impl_split::seek (offset_type offset, whence_type w)
{
  // calculate offset from the beginning of data
  offset_type abs_offset;

  if (w == whence_type::beginning)
    abs_offset = offset;

  else if (w == whence_type::current)
    abs_offset = pos_ + offset;

  else if (w == whence_type::end)
    abs_offset = size_ - 1 + offset;

  else
    throw std::invalid_argument (MOBIUS_EXCEPTION_MSG ("invalid whence_type"));

  // update current pos, if possible
  if (abs_offset >= 0 && size_type (abs_offset) < size_)
    {
      pos_ = abs_offset;
      _set_stream ();
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief read bytes from reader
//! \param size size in bytes
//! \return bytearray containing data
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const mobius::bytearray
reader_impl_split::read (size_type size)
{
  if (!stream_)
    _set_stream ();

  mobius::bytearray data = stream_.read (size);
  pos_ += data.size ();
  size -= data.size ();

  while (size > 0 && pos_ < size_)
    {
      _set_stream ();
      mobius::bytearray tmp = stream_.read (size);
      pos_ += tmp.size ();
      size -= tmp.size ();
      data+= tmp;
    }

  return data;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief set read stream according to current position
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
reader_impl_split::_set_stream ()
{
  size_type segment_idx = pos_ / segment_size_;
  size_type pos = pos_ % segment_size_;

  if (segment_idx != segment_idx_)
    {
      const auto& segment = segments_[segment_idx];
      stream_ = segment.new_reader ();
      segment_idx_ = segment_idx;
    }

  stream_.seek (pos);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief split imagefile writer implementation
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class writer_impl_split : public mobius::io::writer_impl_base
{
public:
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // constructors and assignment operators
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  explicit writer_impl_split (const imagefile_impl_split&);
  writer_impl_split (const writer_impl_split&) = default;
  writer_impl_split (writer_impl_split&&) = default;
  writer_impl_split& operator= (const writer_impl_split&) = default;
  writer_impl_split& operator= (writer_impl_split&&) = default;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // virtual methods
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  virtual size_type write (const mobius::bytearray&) override;
  virtual void seek (offset_type, whence_type = whence_type::beginning) override;
  virtual void flush () override;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief check if reader is seekable
  //! \return true/false
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  virtual bool
  is_seekable () const override
  {
    return true;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get current write position
  //! \return write position in bytes from the beginning of data
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  virtual offset_type
  tell () const override
  {
    return pos_;
  }

private:
  size_type segment_size_;
  size_type pos_ = 0;
  size_type size_ = 0;
  size_type segment_idx_;
  mobius::io::writer stream_;
  segment_array_split segments_;

  size_type _write_data (size_type, const mobius::bytearray&);
  void _set_stream (size_type);
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief constructor
//! \param imagefile_impl imagefile implementation object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
writer_impl_split::writer_impl_split (const imagefile_impl_split& imagefile_impl)
 : segment_size_ (imagefile_impl.get_segment_size ()),
   segment_idx_ (1),
   segments_ (imagefile_impl.get_url ())
{
  _set_stream (0);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief set write position
//! \param offset offset in bytes
//! \param w either beginning, current or end
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
writer_impl_split::seek (offset_type offset, whence_type w)
{
  // calculate offset from the beginning of data
  offset_type abs_offset;

  if (w == whence_type::beginning)
    abs_offset = offset;

  else if (w == whence_type::current)
    abs_offset = pos_ + offset;

  else if (w == whence_type::end)
    abs_offset = size_ - 1 + offset;

  else
    throw std::invalid_argument (MOBIUS_EXCEPTION_MSG ("invalid whence_type"));

  // update current pos, if possible
  if (abs_offset >= 0)
    pos_ = abs_offset;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief write bytes to writer
//! \param data a bytearray
//! \return number of bytes written
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
writer_impl_split::size_type
writer_impl_split::write (const mobius::bytearray& data)
{
  // if write position is ahead of size_, fill gap with zeroes    
  if (size_ < pos_)
    {
      constexpr unsigned int BLOCK_SIZE = 65536;    //!< write block size
      size_type count = pos_ - size_;

      mobius::bytearray buffer (BLOCK_SIZE);
      buffer.fill (0);

      while (count >= size_type (BLOCK_SIZE))
        {
          size_type bytes = _write_data (size_, buffer);
          count -= bytes;
          size_ += bytes;
        }
        
      if (count > 0)
        {
          size_type bytes = _write_data (size_, buffer.slice (0, count - 1));
          size_ += bytes;
        }
    }

  // write data
  size_type bytes = _write_data (pos_, data);
  pos_ += bytes;
  size_ = std::max (size_, pos_);

  // return number of bytes written
  return bytes;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief write data block, splitting into segment files, if necessary
//! \param pos write position
//! \param data data
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
writer_impl_split::size_type
writer_impl_split::_write_data (size_type pos, const mobius::bytearray& data)
{
  mobius::bytearray buffer = data;

  while (!buffer.empty ())
    {
      _set_stream (pos);

      // write data up to segment size
      size_type segment_left = segment_size_ - stream_.tell ();
      
      if (buffer.size () <= segment_left)
        {
          pos += stream_.write (buffer);
          buffer.clear ();
        }

      else
        {
          pos += stream_.write (buffer.slice (0, segment_left - 1));
          buffer = buffer.slice (segment_left, buffer.size () - 1);
        }
    }

  return data.size ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief set write stream according to current position
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
writer_impl_split::_set_stream (size_type pos)
{
  size_type segment_idx = pos / segment_size_;
  size_type stream_pos = pos % segment_size_;

  if (segment_idx != segment_idx_)
    {
      auto& segment = segments_[segment_idx];
      stream_ = segment.new_writer ();
      segment_idx_ = segment_idx;
    }

  stream_.seek (stream_pos);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief flush data to file
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
writer_impl_split::flush ()
{
  stream_.flush ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief check if URL is an instance of imagefile split
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
imagefile_impl_split::is_instance (const std::string& url)
{
  mobius::uri uri (url);
  return uri.get_extension () == "001";
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief construct object
//! \param url imagefile URL
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
imagefile_impl_split::imagefile_impl_split (const std::string& url)
  : imagefile_impl_base (url)
{
  segment_array_split segments (url);
  segments.scan ();
  
  if (segments.get_size () > 0)
    {
      const auto& first_segment = segments[0];
      set_segment_size (first_segment.get_size ());
      set_acquisition_user (first_segment.get_user_name ());
      set_acquisition_time (first_segment.get_last_modification_time ());
      
      size_type size = 0;
      for (auto segment : segments)
        size += segment.get_size ();
      
      set_segments (segments.get_size ());
      set_size (size);
      set_sectors ((size + get_sector_size () - 1) / get_sector_size ());
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create new reader for imagefile
//! \return reader
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const mobius::io::reader
imagefile_impl_split::new_reader () const
{
  return mobius::io::reader (std::make_shared <reader_impl_split> (*this));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create new writer for imagefile
//! \return writer
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const mobius::io::writer
imagefile_impl_split::new_writer () const
{
  return mobius::io::writer (std::make_shared <writer_impl_split> (*this));
}

} // namespace imagefile
} // namespace mobius
