# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
XML_ENCODING='utf-8'

import libxml2
import os.path
import gtk

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Part
# =i=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Part (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.category = None
    self.number = None
    self.attributes = {}

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_attribute (self, id):
    return self.attributes.get (id)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Persistence layer for PartCatalogue
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Pickle (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load parts.xml file
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load (self, path):
    parts = []

    if os.path.exists (path):
      doc = libxml2.parseFile (path)
      node = doc.getRootElement ()
      parts = self.load_parts (node)

    return parts

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <parts>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_parts (self, node):
    parts = []

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'part':
        part = self.load_part (node)
        parts.append (part)

      node = node.next

    return parts

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <part>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_part (self, node):
    part = Part ()
    part.category = self.__get_prop (node, 'category')
    part.number = self.__get_prop (node, 'number')
    part.attributes = {}

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'attribute':
        id, value = self.load_attribute (node)
        part.attributes[id] = value

      node = node.next

    return part

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <attribute>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_attribute (self, node):
    id = self.__get_prop (node, 'id')
    value = self.__get_prop (node, 'value')

    return id, value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save XML
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save (self, path, parts):
    doc = libxml2.newDoc ('1.0')
    node = self.save_catalogue (parts)
    doc.addChild (node)
    doc.saveFormatFileEnc (path, XML_ENCODING, 1)
    doc.freeDoc ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <catalogue>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_catalogue (self, parts):
    node = libxml2.newNode ('catalogue')

    for part in parts:
      child = self.save_part (part)
      node.addChild (child)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <part>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_part (self, part):
    node = libxml2.newNode ('part')
    self.__set_prop (node, 'category', part.category)
    self.__set_prop (node, 'number', part.number)

    for id, value in part.attributes.iteritems ():
      if value != '' and value != None:
        child = self.save_attribute (id, value)
        node.addChild (child)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <attribute>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_attribute (self, id, value):
    node = libxml2.newNode ('attribute')
    self.__set_prop (node, 'id', id)
    self.__set_prop (node, 'value', value)
    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __get_prop (self, node, name):
    value = node.prop (name)
    if value:
      value = value.decode (XML_ENCODING)
    return value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __set_prop (self, node, name, value):
    if value != None:
      node.setProp (name, value.encode (XML_ENCODING))

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Manage part catalogue model
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Model (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, mediator):
    self.mediator = mediator
    self.loaded = False

    self.mediator.advertise ('part.get-list', self.svc_part_get_list)
    self.mediator.advertise ('part.set-list', self.svc_part_set_list)
    self.mediator.connect ('item.attribute-modified', self.on_item_attribute_modified)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.get-list
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_get_list (self):
    if not self.loaded:
      self.__load_model ()
    return self.__parts.itervalues ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.set-list
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_set_list (self, parts):
    self.__parts = {}
    self.__parts.update (((part.category, part.number), part) for part in parts)
    self.__save_model ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Load model from XML
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __load_model (self):
    pickle = Pickle ()
    path = self.mediator.call ('app.get-path', 'part-catalogue.xml')

    self.__parts = {}
    self.__parts.update ((((part.category, part.number), part) for part in pickle.load (path)))
    self.loaded = True

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save model into XML
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __save_model (self):
    pickle = Pickle ()
    path = self.mediator.call ('app.get-path', 'part-catalogue.xml')
    pickle.save (path, self.__parts.values ())

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Event handler: on_item_attribute_modified
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_item_attribute_modified (self, item, attr, text, new_text):
    if attr == 'part_number' and new_text != '':
      if not self.loaded:
        self.__load_model ()
      part = self.__parts.get ((item.category, new_text))

      if not part:
        part = Part ()
        part.category = item.category
        part.number = new_text
        self.mediator.emit ('part.new', part)

      # fill item attributes
      for attr, value in part.attributes.iteritems ():
        self.mediator.call ('item.set-attribute', item, attr, value)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Windows constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
(ITEM_TYPE, ITEM_ICON, ITEM_NAME, ITEM_OBJ) = range (4)
(ATTR_ID, ATTR_NAME, ATTR_VALUE) = range (3)
(TYPE_CATEGORY, TYPE_PART) = range (2)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Widget
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Widget (gtk.VBox):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, mediator):
    gtk.VBox.__init__ (self)
    self.mediator = mediator
    self.set_border_width (5)
    self.set_spacing (5)
    self.show ()

    hpaned = gtk.HPaned ()
    hpaned.show ()
    self.pack_start (hpaned)

    # part treeview
    frame = gtk.Frame ()
    frame.show ()
    hpaned.pack1 (frame, True)

    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    frame.add (sw)

    model = gtk.TreeStore (int, gtk.gdk.Pixbuf, str, object)
    model.set_sort_column_id (ITEM_NAME, gtk.SORT_ASCENDING)

    self.treeview = gtk.TreeView (model)
    self.treeview.set_headers_visible (False)
    self.treeview.set_enable_tree_lines (True)
    self.treeview.show ()
    sw.add (self.treeview)
    self.treeview.connect ('cursor-changed', self.on_treeitem_selected)

    renderer = gtk.CellRendererPixbuf ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'pixbuf', ITEM_ICON)
    self.treeview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ITEM_NAME)
    self.treeview.append_column (tvcolumn)

    # attribute listview
    frame = gtk.Frame ()
    frame.show ()
    hpaned.pack2 (frame)

    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    frame.add (sw)

    datastore = gtk.ListStore (str, str, str)
    self.attribute_listview = gtk.TreeView (datastore)
    self.attribute_listview.set_rules_hint (True)
    self.attribute_listview.set_enable_search (False)

    renderer = gtk.CellRendererText ()

    tvcolumn = gtk.TreeViewColumn ('Name')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ATTR_NAME)
    tvcolumn.set_resizable (True)
    self.attribute_listview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    renderer.set_property ('editable', True)
    renderer.connect ('edited', self.on_attribute_edited)

    tvcolumn = gtk.TreeViewColumn ('Value')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ATTR_VALUE)
    self.attribute_listview.append_column (tvcolumn)

    self.attribute_listview.show ()
    sw.add (self.attribute_listview)

    # buttons
    hbox = gtk.HBox ()
    hbox.show ()
    self.pack_start (hbox, False)

    self.remove_button = gtk.Button (stock=gtk.STOCK_REMOVE)
    self.remove_button.set_sensitive (False)
    self.remove_button.connect ('clicked', self.on_part_remove)
    self.remove_button.show ()
    hbox.pack_start (self.remove_button, False)

    button = gtk.Button (stock=gtk.STOCK_CLOSE)
    button.connect ('clicked', self.on_extension_close)
    button.show ()
    hbox.pack_end (button, False)
    
    self.save_button = gtk.Button (stock=gtk.STOCK_SAVE)
    self.save_button.set_sensitive (False)
    self.save_button.connect ('clicked', self.on_model_save)
    self.save_button.show ()
    hbox.pack_end (self.save_button, False)

    self.part_icon = self.mediator.call ('ui.render-icon', ICON_PART_DATA, 32, 32)
    self.category_iter = {}
    self.is_modified = False

    for part in self.mediator.call ('part.get-list'):
      self.add_part (part)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief add part to treeview
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def add_part (self, part):
    model = self.treeview.get_model ()

    # add category, if necessary
    iter = self.category_iter.get (part.category)
    if not iter:
      category = self.mediator.call ('category.get', part.category)
      iter = model.append (None, (TYPE_CATEGORY, category.icon, category.name, category))
      self.category_iter[part.category] = iter

    # add part to category
    return model.append (iter, (TYPE_PART, self.part_icon, part.number, part))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief add new part to treeview
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def add_new_part (self, part):
    iter = self.add_part (part)
    self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief select a given part in treeview
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def select_part (self, part):
    model = self.treeview.get_model ()
    iter = model.get_iter_first ()

    while iter:
      child = model.iter_children (iter)

      while child:
        tree_part = model.get_value (child, ITEM_OBJ)
        if part == tree_part:
          path = model.get_path (child)
          self.treeview.expand_to_path (path)
          selection = self.treeview.get_selection ()
          selection.select_iter (child)
          self.on_treeitem_selected (self.treeview)
          break
        child = model.iter_next (child)
      iter = model.iter_next (iter)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief show save/ignore/cancel dialog if there are modified items
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def can_destroy (self):

    # if model was modified, show save/ignore/cancel dialog
    if self.is_modified:
      dialog = gtk.MessageDialog (None,
                  gtk.DIALOG_MODAL,
                  gtk.MESSAGE_QUESTION,
                  gtk.BUTTONS_YES_NO,
                  "Save changes before closing?")
      dialog.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
      rc = dialog.run ()
      dialog.destroy ()

      if rc == gtk.RESPONSE_CANCEL:
        return False

      elif rc == gtk.RESPONSE_YES:
        self.on_model_save (None)

    return True

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle close event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_extension_close (self, widget, *args):
    self.working_area.on_delete_event (widget, *args)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle model modification
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_model_modified (self):
    self.is_modified = True
    self.save_button.set_sensitive (True)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle model-save event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_model_save (self, widget, *args):
    model = self.treeview.get_model ()
    iter = model.get_iter_first ()
    parts = []

    while iter:
      child = model.iter_children (iter)
      while child:
        part = model.get_value (child, ITEM_OBJ)
        parts.append (part)
        child = model.iter_next (child)
      iter = model.iter_next (iter)

    self.mediator.call ('part.set-list', parts)
    self.is_modified = False
    self.save_button.set_sensitive (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle treeitem-selected event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_treeitem_selected (self, widget, *args):
    model, iter = widget.get_selection ().get_selected ()

    if iter:
      item_type = model.get_value (iter, ITEM_TYPE) 

      if item_type == TYPE_PART:
        part = model.get_value (iter, ITEM_OBJ) 
        self.on_part_selected (part)
      else:
        category = model.get_value (iter, ITEM_OBJ) 
        self.on_category_selected (category)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle part-selected event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_part_selected (self, part):
    self.attribute_listview.set_sensitive (True)
    self.remove_button.set_sensitive (True)

    # populate attribute-listview
    model = self.attribute_listview.get_model ()
    model.clear ()

    for a in self.mediator.call ('category.get-attribute-list', part.category):
      if a.id != 'part_number':
        model.append ((a.id, a.name, part.get_attribute (a.id)))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle part-unselected event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_part_unselected (self):
    self.attribute_listview.set_sensitive (False)
    self.remove_button.set_sensitive (False)

    # clear attribute-listview
    model = self.attribute_listview.get_model ()
    model.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category-selected event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_selected (self, category):
    self.attribute_listview.set_sensitive (False)
    self.remove_button.set_sensitive (False)

    # clear attribute-listview
    model = self.attribute_listview.get_model ()
    model.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle part-remove event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_part_remove (self, widget, *args):
    selection = self.treeview.get_selection ()
    model, iter = selection.get_selected ()

    if iter:
      part = model.get_value (iter, ITEM_OBJ)

      dialog = gtk.MessageDialog (None,
        gtk.DIALOG_MODAL,
        gtk.MESSAGE_QUESTION,
        gtk.BUTTONS_YES_NO,
        "You are about to remove part '%s'. Are you sure?" % part.number)

      rc = dialog.run ()
      dialog.destroy ()

      if rc != gtk.RESPONSE_YES:
        return

      # remove part
      has_next = model.remove (iter)

      if has_next:
        selection.select_iter (iter)
      else:
        self.on_part_unselected ()

      # signal model modification
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute-edited event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_edited (self, cell, path, new_text, *args):
    model = self.attribute_listview.get_model ()
    iter = model.get_iter_from_string (path)
    text = cell.get_property ('text')

    if text != new_text:
      model.set_value (iter, ATTR_VALUE, new_text)
      id = model.get_value (iter, ATTR_ID)
 
      treemodel, iter = self.treeview.get_selection ().get_selected ()
      if iter:
        part = treemodel.get_value (iter, ITEM_OBJ)
        part.attributes[id] = new_text

      self.on_model_modified ()

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief 32x32 PNG icon
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''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='''

ICON_PART_DATA = '''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'''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'part_catalogue'
    self.name = 'Part Catalogue'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.0'
    self.description = 'Part/partnumber catalogue and manager'
    self.mediator = mediator.new_client_mediator ()
    self.icon_data = ICON_DATA

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
    self.model = Model (self.mediator)
    self.mediator.call ('toolbox.add', 'part_catalogue', ICON_DATA, 'Part\nCatalogue', self.on_activate)
    self.mediator.connect ('part.new', self.on_part_new)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    self.mediator.call ('ui.working-area.del', self.id)
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Open working area
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def open_working_area (self):
    working_area = self.mediator.call ('ui.working-area.get', self.id)

    if not working_area:
      widget = Widget (self.mediator)

      working_area = self.mediator.call ('ui.working-area.new', self.id)
      working_area.set_default_size (600, 400)
      working_area.set_title (self.name)
      working_area.set_icon (self.icon_data)
      working_area.set_widget (widget)

    working_area.show ()
    return working_area

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_activate
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_activate (self, id):
    working_area = self.open_working_area ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: part.new
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_part_new (self, part):
    working_area = self.open_working_area ()

    widget = working_area.get_widget ()
    widget.add_new_part (part)
    widget.select_part (part)
