# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008, 2009 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import libxml2
import re
import os.path

ATTR_EXP = re.compile (r'\$\{(.*?)\}')
XML_ENCODING='utf-8'

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Mobius object
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Object (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.uid = -1
    self.category = ''
    self.__attributes = {}
    self.__capabilities = set ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief set attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_attribute (self, id, value):
    self.__attributes[id] = value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief get attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_attribute (self, id):
    return self.__attributes.get (id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief remove attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def remove_attribute (self, id):
    if id in self.__attributes:
      self.__attributes.pop (id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief has attribute?
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def has_attribute (self, id):
    return id in self.__attributes

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief return attributes iterator
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def iter_attributes (self):
    return self.__attributes.iteritems ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief add capability
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_capability (self, id):
    self.__capabilities.add (id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief remove capability
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def remove_capability (self, id):
    self.__capabilities.discard (id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief has capability?
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def has_capability (self, id):
    return id in self.__capabilities

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief iter capabilities
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def iter_capabilities (self):
    return iter (self.__capabilities)


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Case class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Case (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.filename = None
    self.icon_data = ICON_CASE_DATA
    self.__rootdir = None
    self.__next_uid = 1
    self.__is_new = True
    self.__is_modified = False
    self.__objects = {}
    self.__attributes = {}
    self.__children = []

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return iterator to children
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __iter__ (self):
    return iter (self.__children)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief set attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_attribute (self, id, value):
    self.__attributes[id] = value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_attribute (self, id):
    return self.__attributes.get (id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return attribute iterator
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def iter_attributes (self):
    return self.__attributes.iteritems ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set root directory
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_rootdir (self, path):
    self.__rootdir = path

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return root directory
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_rootdir (self):
    return self.__rootdir

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set next uid
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_next_uid (self, uid):
    self.__next_uid = uid

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return next uid
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_next_uid (self):
    return self.__next_uid

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Change case modification flag
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_modified (self, flag):
    self.__is_modified = flag

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return True if case is modified
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def is_modified (self):
    return self.__is_modified

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Change case new flag
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_new (self, flag):
    self.__is_new = flag

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return True if case is new
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def is_new (self):
    return self.__is_new

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_object (self, obj):
    obj.case = self
    self.__objects[obj.uid] = obj

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_object (self, uid):
    return self.__objects.get (uid)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add child
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_child (self, child):
    child.parent = self
    child.case = self
    self.__children.append (child)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Remove child
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def remove_child (self, child):
    self.__children.remove (child)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return True if case has at least a child
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def has_child (self):
    return len (self.__children) > 0

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Create item
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def create_item (self, category):
    item = Item ()
    item.category = category.id
    item.case = self
    item.uid = self.__next_uid
    self.__objects[item.uid] = item
    self.__next_uid += 1

    return item

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Case Item
# =i=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-i=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Item (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, parent=None):
    self.parent = parent
    self.category = None
    self.case = None

    # control variables
    self.__children = []
    self.__attributes = {}

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return iterator to children
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __iter__ (self):
    return iter (self.__children)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_attribute (self, id, value):
    self.__attributes[id] = value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_attribute (self, id):
    value = self.__attributes.get (id)
    return value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Check if item has a given attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def has_attribute (self, id):
    return id in self.__attributes

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return attribute iterator
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def iter_attributes (self):
    return self.__attributes.iteritems ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add child
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_child (self, child):
    child.parent = self
    child.case = self.case
    self.__children.append (child)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Remove child
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def remove_child (self, child):
    self.__children.remove (child)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Check whether item has at least a child
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def has_child (self):
    return len (self.__children) > 0

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# case icon data (PNG 32x32)
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_CASE_DATA = '''
iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAABGdBTUEAALGPC/xhBQAAAAZiS0dE
AP8A/wD/oL2nkwAAAAlwSFlzAAAN1wAADdcBQiibeAAAAAd0SU1FB9gHDxctIjTZVloAAAdmSURB
VFjDtVdrbBTXFf7unZl9+bHrfWCDbQzGBhu7QMwCBYfwMKkBA00NSmPiqo+A2tCoUqukClGqtmmq
FEgRjVSIqxYRhVQy0ATUgmxCABcDNjVyEtPYIQ6xedjx296d3Zmd1+2P9WvXi0kseqWruTtzdr7v
nPPdM+cCUxzXBCy6AMRPZsMOIPlB76EPMsjLq68uKLhWPv5eA2CzJuEdlwfHGEBiglfiZ8yO/7IK
CFN1Evn59S+cOtUhHz7c3u/11h8CLvDHAO5jN84opTACGyB/5MSfIoDfQKLxLt4N3YOoNqOPHUTm
lMCTkmpn7tjRdKuxsZs1Nnaz6uoOrbDwWk2901IpbYbCysBYGVj3CvQ2OvADAGBHsUSvxseyDCYz
MKUTOvsL1kW/Oyvr1C/i448dAAD+fgSysrjDTz3lmO3zKSCEwGYDt3/WwcfmJRqGJX4sdZ4MuKQh
/EZ8Cbl6Lsq0AqSPRiMB1LCjAMA5APB4TmQXFDgPybKR19oa+MZ9CWRn1z5fXp5UKEkqZJmAEMBx
fC/mDh6D3aFE6oYA6TuQoZTgeS07UlMsDgCPbOACv3Rp6NUtW9K279w5J7209NKbwPbemASs1tqZ
eXmmXW43LH6/CkKAaWcrMPfLo3AkBSONzYCxCtC2AYiPLehLnxfMKSlB7b59y7zz5tm53bs/bL18
ueelkedRBBiZPbv2SFGRefYI+PSrRzH/7mG4Pf5IyxTAWAtoG2PvJUXh8ecDO2GhTxe+/bbXxPME
t2+LqKnpOgM8PRCTQHp6zW/Xr7cWSpIGWQbSGv+Bgs4KpKQMRoSc5QF6MaAvvk+NqMtH5TvP4Xvl
TyA1Nc4kyxo4jmDPnubP6us7Xh5vO0rAbj+/OCeH/5HNZpj8fgWzWk5jec+bSJ0+DpwD2HJALQ1H
IHpIkgn79zwLR+IT2PXsAnAcgSxroJQgENDQ0NB/HnjGH4PAFavHI1fk5uqpfr+OOW0fYHXfQWSk
949ZJgDG6jA4zBPB/33xEZyo/Am+vbkYHo8FkhT2OjwpKipa2xsaBl6O/h8PAMnJwbe8XhT4/Tqy
Oq+g2HcQmRlj4CwDMIoArSicAkXhYTJpAABRtGDv73ch3laCJ7dlg1IyCk5pGFzTNNTV9V8eUX4E
gYSE91/MyVE2qSojM7qbsMl3EHOze8fyvQjQSwB9PqDrFAf+WI6q0wU4e/HnOFu1DO8d/yFWr1oF
h0OAJGnDoGHgMAkD1VX3Bm7eDP4ull54TeMu9PcrL5jNqrWk/RAWLO0ajQ1bCahbAeYEPrmRgX2v
PYf5uY9hxXIJJd/aC+/iRVi7Jg0ch3Ehp6AUowQ4jqCp4d4Xfv+2lpgEJGlt/Z07Vdfj4wOPdzE3
gPZw2FcByvfHVLL7ly+iaM0S8DxFYqIZGz1FsNn4qHBHes5xBKKowRA7Wyf9GoZC3Ct379LBi2wZ
hvrCm5qxyE2amMAhFDIgSTokSQNjDJKkDU8dsqyPPhuZfr+Kf711Ebn5GXsnJSBJj9f29lquf6A/
gqY2Tzj9QwBYZGEJ14cxgDDo+N9ja1HUcPLIJVb05HKs25pzoq7uzoJJ+wFV5f/QPWDzNQ8Nb/Ae
gPjCS5/PCsa4CLD7kRgBr6q8ahRt9ZKUFAsEgZtFKb3a0tL3zH0JaNr6c6IYf/20byEGeylIL0Dv
DBeYoBmGQaNA9Rie6wgENFz8Z4O+ctNC6nabwfMUgkAhCJxNEMhfu7rEvzPGEmN2RIyZXj8jPSo2
fu4CJIDcHalwAjSNGwWJDvfIPVFUce1co/bNdfM5t9syAjx8Da9NJmobn9yoz8iGMyqmNX4ykAww
gAxXYVG0QAnxMXI+FolgUEfT5SZ14aO5vMttjQLnIAiUmUz0VYfD+h1CiH+SntC6v1pcGhrso8Aw
gYBogSxzE8I9QiQY1NDyn2Y1f8Vc3u0xx/CcBgSB/25KSsKvCCHsAU1p8cn31fXXP2x1hSOgAqJo
haKwqC03Fom2pptq7pJM3uk0k/HA4fxz7TxPCjMzHce/clcsw/5Gy0CKQToB0gf4fDZomjFB7cGg
hi8/u6XOWTiLJjnNZJzghq/8WUGg3vz85I++ZlteXHkpsPSWv42CtgHdXXaoqjZB+YN327WM3FQk
Oc1cGHTUe0MQ6Ctr1szc4PXO6J3SuaBKKdrTeMPFSFe4DiiKEan8nk4tZfYMZndahXG5hiBwTBDI
9o0bs39NCDGmfDDpR9nf2oamDZEhwO+zIBQaU77h69Y9M11aktssRAuO52mvruee+Krt/yQnI8Ja
A2kntW6CoSErZFmFJOng5D7DMd0ZsjutlhiCg8nExSUkfJH1EAgAaXLTTzta4nRZDNcAGwYNhydO
drgstomC4yAIdFAQuOINGzI/fSgEfoyOoNEsXVGHKOwmkblcZiXBFW8bFlyPINAbgkA/FQROEgTa
bTbza0tL59V+nRMY/yAD+bb+2u0uz8pZLukWkjzvCQKtoVSvP3Jkc8+ITUVFgw1g9rKyvE487MEA
cn5N+hb8n8b/AM4MzLllmV4aAAAAAElFTkSuQmCC'''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Persistence layer for case
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Pickle (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __get_prop (self, node, name):
    value = node.prop (name)
    if value:
      value = value.decode (XML_ENCODING)
    return value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __set_prop (self, node, name, value):
    if value != None:
      node.setProp (name, value.encode (XML_ENCODING))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load application data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load (self, path):
    doc = libxml2.parseFile (path)
    node = doc.getRootElement ()

    case = self.load_case (node)
    case.filename = path
    case.set_new (False)
    
    return case

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load case from node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_case (self, node):
    case = Case ()
    case.set_rootdir (self.__get_prop (node, 'rootdir'))
    case.set_next_uid (int (self.__get_prop (node, 'next_uid') or '1'))

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'item':
        child = self.load_item (node, case)
        child.parent = case
        case.add_child (child)
        case.add_object (child)

      elif node.type == 'element' and node.name == 'attribute':
        id, value = self.load_attribute (node)
        case.set_attribute (id, value)

      node = node.next

    return case

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load item from node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_item (self, node, case):
    item = Item ()
    item.uid = int (self.__get_prop (node, 'uid'))
    item.case = case
    item.category = self.__get_prop (node, 'category')

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'item':
        child = self.load_item (node, case)
        item.add_child (child)
        case.add_object (child)

      elif node.type == 'element' and node.name == 'attribute':
        id, value = self.load_attribute (node)
        item.set_attribute (id, value)

      node = node.next

    return item

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load attribute from node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_attribute (self, node):
    name = self.__get_prop (node, 'name')
    value = self.__get_prop (node, 'value')
    return name, value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save case into XML file
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save (self, case):
    doc = libxml2.newDoc ('1.0')
    node = self.save_case (case)
    doc.addChild (node)
    doc.saveFormatFileEnc (case.filename, XML_ENCODING, 1)
    doc.freeDoc ()

    case.set_new (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_case (self, case):
    node = libxml2.newNode ('case')
    self.__set_prop (node, 'rootdir', case.get_rootdir ())
    self.__set_prop (node, 'next_uid', str (case.get_next_uid ()))

    # attributes
    for name, value in case.iter_attributes ():
      child = self.save_attribute (name, value)
      node.addChild (child)

    # items
    for item in case:
      child = self.save_item (item)
      node.addChild (child)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save item
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_item (self, item):
    node = libxml2.newNode ('item')
    self.__set_prop (node, 'uid', str (item.uid))
    self.__set_prop (node, 'category', item.category)

    # attributes
    for id, value in item.iter_attributes ():
      child = self.save_attribute (id, value)
      node.addChild (child)

    # sub-items
    for subitem in item:
      child = self.save_item (subitem)
      node.addChild (child)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save atttribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_attribute (self, name, value):
    node = libxml2.newNode ('attribute')
    self.__set_prop (node, 'name', name)
    self.__set_prop (node, 'value', value)

    return node

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Icon
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''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'''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'case-model'
    self.name = 'Case Model'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.0'
    self.description = 'Manages case data'
    self.mediator = mediator.new_client_mediator ()
    self.icon_data = ICON_DATA

    # specific data
    self.__next_uid = 1
    self.__next_new_id = 1
    self.__cases = {}

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
    self.mediator.advertise ('case.new', self.svc_case_new)
    self.mediator.advertise ('case.open', self.svc_case_open)
    self.mediator.advertise ('case.save', self.svc_case_save)
    self.mediator.advertise ('case.close', self.svc_case_close)
    self.mediator.advertise ('case.get-path', self.svc_case_get_path)
    self.mediator.advertise ('case.any-modified', self.svc_case_any_modified)
    self.mediator.advertise ('case.save-all', self.svc_case_save_all)
    self.mediator.advertise ('item.set-attribute', self.svc_item_set_attribute)
    self.mediator.advertise ('item.expand-expression', self.svc_item_expand_expression)
    self.mediator.advertise ('item.expand-initvalues', self.svc_item_expand_initvalues)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief service: case.new
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def svc_case_new (self):
    case = Case ()
    case.uid = self.__next_uid
    case.set_attribute ('id', 'untitled%02d' % self.__next_new_id)
    case.set_attribute ('name', 'Untitled Case #%02d' % self.__next_new_id)
    case.filename = 'untitled%02d.case' % self.__next_new_id

    self.__cases[case.uid] = case
    self.__next_uid += 1
    self.__next_new_id += 1
    self.mediator.emit ('case-created', case)

    return case

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief service: case.open
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def svc_case_open (self, path):
    pickle = Pickle ()
    case = pickle.load (path)
    case.uid = self.__next_uid

    self.__cases[case.uid] = case
    self.__next_uid += 1
    self.mediator.emit ('case-opened', case)

    return case

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief service: case.save
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def svc_case_save (self, case):
    pickle = Pickle ()
    pickle.save (case)
    case.set_modified (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief service: case.close
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def svc_case_close (self, case):
    self.__cases.pop (case.uid)
    self.mediator.emit ('case-closed', case)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief service: case.get-path
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def svc_case_get_path (self, case, *args):
    path = os.path.join (case.get_rootdir (), *args)

    dirname = os.path.dirname (path)
    if not os.path.exists (dirname):
      os.makedirs (dirname)

    return path

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief service: case.any-modified
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def svc_case_any_modified (self):
    for case in self.__cases.itervalues ():
      if case.is_modified ():
        return True
    return False

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief service: case.save-all
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def svc_case_save_all (self):
    for case in self.__cases.itervalues ():
      if case.is_modified ():
        self.svc_case_save (case)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief service: item.set-attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_item_set_attribute (self, item, attr, value):
    old_value = item.get_attribute (attr)

    if old_value != value:
      item.set_attribute (attr, value)
      self.mediator.emit ('item.attribute-modified', item, attr, old_value, value)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief service: item.expand-expression
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_item_expand_expression (self, item, expr):

    def repl (match):
      id = match.group (1)
      return item.get_attribute (id)

    return ATTR_EXP.sub (repl, expr)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  # @brief Service: item.expand-initvalues
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_item_expand_initvalues (self, item):
    for attr in self.mediator.call ('category.iter-attributes', item.category):
      if attr.initvalue:
        value = self.mediator.call ('item.expand-expression', item, attr.initvalue)
        self.mediator.call ('item.set-attribute', item, attr.id, value)
