# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008, 2009 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
XML_ENCODING='utf-8'
METHOD_NEW_PART, METHOD_MERGE, METHOD_REPLACE = range (3)

import libxml2
import os.path

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Part
# =i=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Part (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.category = None
    self.id = None
    self.attributes = {}

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_attribute (self, id):
    return self.attributes.get (id)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @deprecated Persistence layer for Part Catalogue prior to Mobius 0.4.4
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Pickle (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load parts.xml file
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load (self, path):
    parts = []

    if os.path.exists (path):
      doc = libxml2.parseFile (path)
      node = doc.getRootElement ()
      parts = self.load_parts (node)

    return parts

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <parts>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_parts (self, node):
    parts = []

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'part':
        part = self.load_part (node)
        parts.append (part)

      node = node.next

    return parts

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <part>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_part (self, node):
    part = Part ()
    part.category = self.__get_prop (node, 'category')
    part.id = self.__get_prop (node, 'number')
    part.attributes = {}

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'attribute':
        id, value = self.load_attribute (node)
        part.attributes[id] = value

      node = node.next

    return part

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <attribute>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_attribute (self, node):
    id = self.__get_prop (node, 'id')
    value = self.__get_prop (node, 'value')

    return id, value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __get_prop (self, node, name):
    value = node.prop (name)
    if value:
      value = value.decode (XML_ENCODING)
    return value

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief icon data
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''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'''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'part-model'
    self.name = 'Part Model'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.0'
    self.description = 'Part/part-number model'
    self.mediator = mediator.copy ()
    self.icon_data = ICON_DATA

    # specific data
    self.__loaded = False
    self.__parts = {}
    self.__path = self.mediator.call ('app.get-path', 'part-catalogue.xml')

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
    self.mediator.advertise ('part.new', self.svc_part_new)
    self.mediator.advertise ('part.set', self.svc_part_set)
    self.mediator.advertise ('part.get', self.svc_part_get)
    self.mediator.advertise ('part.remove', self.svc_part_remove)
    self.mediator.advertise ('part.catalogue.iter', self.svc_part_catalogue_iter)
    self.mediator.advertise ('part.catalogue.load', self.svc_part_catalogue_load)
    self.mediator.advertise ('part.catalogue.save', self.svc_part_catalogue_save)
    self.mediator.advertise ('part.catalogue.export', self.svc_part_catalogue_export)
    self.mediator.advertise ('part.catalogue.import', self.svc_part_catalogue_import)
    self.mediator.connect ('item.attribute-modified', self.on_item_attribute_modified)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Load catalogue
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def load_catalogue (self, path):

    # @deprecated read catalogue version prior to Mobius 0.4.4, too
    data = open (path).read ()

    if '<catalogue>' in data:
      pickle = Pickle ()
      parts = {}
      for part in pickle.load (path):
        cat_dict = parts.setdefault (part.category, {})
        cat_dict[part.id] = part
    else:
      parts = self.mediator.call ('xml.unpickle', path)

    return parts

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Load on demand
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def load_on_demand (self):
    if not self.__loaded:
      if os.path.exists (self.__path):
        self.__parts = self.load_catalogue (self.__path)
      else:
        self.__parts = {}
      self.__loaded = True

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.new
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_new (self, category, part_id):
    self.load_on_demand ()

    part = Part ()
    part.category = category
    part.id = part_id
    self.svc_part_set (part)

    return part

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.set
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_set (self, part):
    self.load_on_demand ()

    cat_dict = self.__parts.setdefault (part.category, {})
    cat_dict[part.id] = part

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.get
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_get (self, category, part_id):
    self.load_on_demand ()

    cat_dict = self.__parts.get (category, {})
    part = cat_dict.get (part_id)

    return part

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.remove
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_remove (self, part):
    self.load_on_demand ()

    cat_dict = self.__parts.get (part.category, {})
    old_part = cat_dict.pop (part.id, None)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.catalogue.iter
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_catalogue_iter (self):
    self.load_on_demand ()

    for cat in self.__parts.itervalues ():
      for part in cat.itervalues ():
        yield part

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.catalogue.load
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_catalogue_load (self):
    self.__loaded = False
    self.load_on_demand ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.catalogue.save
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_catalogue_save (self):
    self.load_on_demand ()
    self.mediator.call ('xml.pickle', self.__path, self.__parts)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.catalogue.export
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_catalogue_export (self, path):
    self.load_on_demand ()
    self.mediator.call ('xml.pickle', path, self.__parts)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.catalogue.import
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_catalogue_import (self, path, method):
    self.load_on_demand ()

    # load catalogue
    parts = self.load_catalogue (path)

    # import
    if method in (METHOD_NEW_PART, METHOD_MERGE):
      for cat_dict in parts.itervalues ():
        for part in cat_dict.itervalues ():
          old_part = self.svc_part_get (part.category, part.id)

          if not old_part:
            self.svc_part_set (part)

          elif method == METHOD_MERGE:
            for attr_id, attr_value in part.attributes.iteritems ():
              if attr_value:
                old_part.attributes[attr_id] = attr_value

    elif method == METHOD_REPLACE:
      self.__parts = parts

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Event handler: on_item_attribute_modified
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_item_attribute_modified (self, item, attr, text, new_text):
    if attr == 'part_id' and new_text != '':
      part = self.svc_part_get (item.category, new_text)

      if not part:
        self.mediator.emit ('part.unknown', item.category, new_text)

      else:    # fill item attributes
        for attr, value in part.attributes.iteritems ():
          self.mediator.call ('item.set-attribute', item, attr, value)
