# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008, 2009 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
XML_ENCODING='utf-8'

import libxml2
import os.path
import gtk

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Category
# =i=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Category (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.parent = None
    self.id = None
    self.icon = None
    self.icon_data = None
    self.is_dataholder = False
    self.__attributes = []
    self.__attribute_dict = {}

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_attribute (self, attr):
    self.__attributes.append (attr)
    self.__attribute_dict [attr.id] = attr
    attr.parent = self

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_attribute (self, id):
    return self.__attribute_dict.get (id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get attribute list
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_attribute_list (self):
    return self.__attributes[:]

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Clear attributes
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def clear_attributes (self):
    self.__attributes = []
    self.__attribute_dict = {}

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return icon for item
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_icon (self, item):
    return self.icon

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Attribute
# =i=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Attribute (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.parent = None
    self.id = None
    self.name = None
    self.description = None
    self.default = None

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Model class holds categories
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Model (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, mediator):
    self.mediator = mediator
    self.mediator.advertise ('category.get', self.svc_category_get)
    self.mediator.advertise ('category.get-icon-data', self.svc_category_get_icon_data)
    self.mediator.advertise ('category.get-list', self.svc_category_get_list)
    self.mediator.advertise ('category.set-list', self.svc_category_set_list)
    self.mediator.advertise ('category.get-attribute-list', self.svc_category_get_attribute_list)
    self.loaded = False

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: category.get
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_get (self, id):
    if not self.loaded:
      self.__load_model ()
    return self.__categories.get (id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: category.get-icon-data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_get_icon_data (self, id):
    if not self.loaded:
      self.__load_model ()

    category = self.__categories.get (id)
    if category:
      icon_data = category.icon_data
    else:
      icon_data = ICON_UNKNOWN_CATEGORY

    return icon_data

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: category.get-list
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_get_list (self):
    self.__load_model ()
    return self.__categories.values ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: category.set-list
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_set_list (self, categories):
    self.__categories = {}
    self.__categories.update (((cat.id, cat) for cat in categories))
    self.__save_model ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: category.get-attribute-list
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_get_attribute_list (self, id):
    if not self.loaded:
      self.__load_model ()

    attr_list = []
    category = self.__categories.get (id)

    if category:
      attr_list = category.get_attribute_list ()

    return attr_list

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Load model from XML
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __load_model (self):
    pickle = Pickle ()
    path = self.mediator.call ('app.get-path', 'category.xml')

    self.__categories = {}
    self.__categories.update (((cat.id, cat) for cat in pickle.load (path)))
    self.loaded = True

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Save model to XML
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __save_model (self):
    pickle = Pickle ()
    path = self.mediator.call ('app.get-path', 'category.xml')
    pickle.save (path, self.__categories.values ())

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Persistence layer for category database
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Pickle (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __get_prop (self, node, name):
    value = node.prop (name)
    if value:
      value = value.decode (XML_ENCODING)
    return value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __set_prop (self, node, name, value):
    if value != None:
      node.setProp (name, value.encode (XML_ENCODING))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load categories
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load (self, path):
    categories = []

    if os.path.exists (path):
      doc = libxml2.parseFile (path)
      node = doc.getRootElement ()
      categories = self.load_categories (node)

    return categories

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <categories>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_categories (self, node):
    categories = []

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'category':
        cat = self.load_category (node)
        categories.append (cat)

      node = node.next

    return categories

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <category> from node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_category (self, node):
    cat = Category ()
    cat.id = self.__get_prop (node, 'id')
    cat.name = self.__get_prop (node, 'name')
    cat.is_dataholder = self.__get_prop (node, 'is_dataholder') == 'True'

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'attribute':
        cat.add_attribute (self.load_attribute (node))

      elif node.type == 'element' and node.name == 'icon':
        cat.icon_data = self.load_icon (node)

      node = node.next

    return cat

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <attribute> from node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_attribute (self, node):
    attr = Attribute ()
    attr.id = self.__get_prop (node, 'id')
    attr.name = self.__get_prop (node, 'name')
    attr.description = self.__get_prop (node, 'description')
    attr.default = self.__get_prop (node, 'default')

    return attr

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <icon> from node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_icon (self, node):
    icon = None
    node = node.children

    while node:
      if node.type == 'text':
        icon = node.getContent ()

      node = node.next

    return icon

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save XML
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save (self, path, categories):
    doc = libxml2.newDoc ('1.0')
    node = self.save_categories (categories)
    doc.addChild (node)
    doc.saveFormatFileEnc (path, XML_ENCODING, 1)
    doc.freeDoc ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <categories>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_categories (self, categories):
    node = libxml2.newNode ('categories')

    for cat in categories:
      child = self.save_category (cat)
      node.addChild (child)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <category>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_category (self, cat):
    node = libxml2.newNode ('category')
    self.__set_prop (node, 'id', cat.id)
    self.__set_prop (node, 'name', cat.name)
    self.__set_prop (node, 'is_dataholder', str (cat.is_dataholder))

    for a in cat.get_attribute_list ():
      child = self.save_attribute (a)
      node.addChild (child)

    if cat.icon_data:
      child = self.save_icon_data (cat.icon_data)
      node.addChild (child)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <attribute>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_attribute (self, attr):
    node = libxml2.newNode ('attribute')
    self.__set_prop (node, 'id', attr.id)
    self.__set_prop (node, 'name', attr.name)
    self.__set_prop (node, 'description', attr.description)
    self.__set_prop (node, 'default', attr.default)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <icon>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_icon_data (self, data):
    node = libxml2.newNode ('icon')
    node.addContent (data)
    return node

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Windows constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
(CATEGORY_ICON, CATEGORY_NAME, CATEGORY_OBJ) = range (3)
(ATTR_ID, ATTR_NAME, ATTR_DEFAULT) = range (3)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Widget
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Widget (gtk.VBox):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, mediator):
    gtk.VBox.__init__ (self)
    self.mediator = mediator.new_client_mediator ()
    self.set_border_width (5)
    self.set_spacing (5)
    self.show ()

    hpaned = gtk.HPaned ()
    hpaned.show ()
    self.pack_start (hpaned)

    # category list
    frame = self.build_category_list ()
    frame.show ()
    hpaned.pack1 (frame, True)

    # notebook
    notebook = gtk.Notebook ()
    notebook.show ()
    hpaned.pack2 (notebook)

    # page: general
    vbox1 = gtk.VBox ()
    vbox1.show ()
    notebook.append_page (vbox1, gtk.Label ('General'))

    table = gtk.Table (4, 3)
    table.set_row_spacings (10)
    table.set_col_spacings (5)
    table.show ()
    vbox1.pack_start (table, False, False)

    label = gtk.Label ()
    label.set_markup ('<b>ID</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    table.attach (label, 0, 1, 0, 1, gtk.FILL, 0)

    self.category_id_entry = gtk.Entry ()
    self.category_id_entry.connect ('changed', self.on_category_id_changed)
    self.category_id_entry.show ()
    table.attach (self.category_id_entry, 1, 3, 0, 1, gtk.FILL, 0)

    label = gtk.Label ()
    label.set_markup ('<b>Name</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    table.attach (label, 0, 1, 1, 2, gtk.FILL, 0)

    self.category_name_entry = gtk.Entry ()
    self.category_name_entry.connect ('changed', self.on_category_name_changed)
    self.category_name_entry.show ()
    table.attach (self.category_name_entry, 1, 3, 1, 2, yoptions=0)

    label = gtk.Label ()
    label.set_markup ('<b>Icon</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    table.attach (label, 0, 1, 2, 3, gtk.FILL, 0)

    self.category_icon_button = gtk.Button ()
    self.category_icon_button.connect ('clicked', self.on_category_icon_edit)
    self.category_icon_button.show ()

    self.category_image = gtk.Image ()
    self.category_image.set_from_stock (gtk.STOCK_MISSING_IMAGE, gtk.ICON_SIZE_BUTTON)
    self.category_image.show ()
    self.category_icon_button.add (self.category_image)

    table.attach (self.category_icon_button, 1, 2, 2, 3, 0, 0)

    placeholder = gtk.Label (' ')
    placeholder.show ()
    table.attach (placeholder, 2, 3, 2, 3, yoptions=0)

    self.category_is_dataholder_cb = gtk.CheckButton ('Is dataholder')
    self.category_is_dataholder_cb.connect ('toggled', self.on_category_is_dataholder_toggled)
    self.category_is_dataholder_cb.show ()
    table.attach (self.category_is_dataholder_cb, 1, 3, 3, 4, yoptions=0)

    # page: attributes
    vbox_page = gtk.VBox ()
    vbox_page.set_spacing (5)
    vbox_page.show ()
    notebook.append_page (vbox_page, gtk.Label ('Attributes'))
    
    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    vbox_page.pack_start (sw)

    # listview
    datastore = gtk.ListStore (str, str, str)
    self.attribute_listview = gtk.TreeView (datastore)
    self.attribute_listview.set_rules_hint (True)
    self.attribute_listview.set_enable_search (False)
    self.attribute_listview.connect ('cursor-changed', self.on_attribute_selected)

    renderer = gtk.CellRendererText ()
    renderer.set_property ('editable', True)
    renderer.connect ('edited', self.on_attribute_edited, ATTR_ID)

    tvcolumn = gtk.TreeViewColumn ('ID')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ATTR_ID)
    tvcolumn.set_resizable (True)
    self.attribute_listview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    renderer.set_property ('editable', True)
    renderer.connect ('edited', self.on_attribute_edited, ATTR_NAME)

    tvcolumn = gtk.TreeViewColumn ('Name')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ATTR_NAME)
    self.attribute_listview.append_column (tvcolumn)
    self.attribute_listview.show ()
    sw.add (self.attribute_listview)

    # default value
    hbox = gtk.HBox ()
    hbox.show ()
    vbox_page.pack_start (hbox, False)

    label = gtk.Label ()
    label.set_markup ('<b>Default value</b>')
    label.show ()
    hbox.pack_start (label, False)

    self.default_value_entry = gtk.Entry ()
    self.default_value_entry.connect ('changed', self.on_default_value_changed)
    self.default_value_entry.show ()
    hbox.pack_start (self.default_value_entry)

    # attribute buttons
    hbox = gtk.HBox ()
    hbox.show ()
    vbox_page.pack_start (hbox, False)

    self.add_attr_button = gtk.Button (stock=gtk.STOCK_ADD)
    self.add_attr_button.connect ('clicked', self.on_attribute_add)
    self.add_attr_button.show ()
    hbox.pack_start (self.add_attr_button, False)
    
    self.remove_attr_button = gtk.Button (stock=gtk.STOCK_REMOVE)
    self.remove_attr_button.set_sensitive (False)
    self.remove_attr_button.connect ('clicked', self.on_attribute_remove)
    self.remove_attr_button.show ()
    hbox.pack_start (self.remove_attr_button, False)

    self.up_attr_button = gtk.Button (stock=gtk.STOCK_GO_UP)
    self.up_attr_button.set_sensitive (False)
    self.up_attr_button.connect ('clicked', self.on_attribute_up)
    self.up_attr_button.show ()
    hbox.pack_start (self.up_attr_button, False)

    self.down_attr_button = gtk.Button (stock=gtk.STOCK_GO_DOWN)
    self.down_attr_button.set_sensitive (False)
    self.down_attr_button.connect ('clicked', self.on_attribute_down)
    self.down_attr_button.show ()
    hbox.pack_start (self.down_attr_button, False)

    # buttons
    hbox = gtk.HBox ()
    hbox.show ()
    self.pack_start (hbox, False)

    button = gtk.Button (stock=gtk.STOCK_ADD)
    button.connect ('clicked', self.on_category_add)
    button.show ()
    hbox.pack_start (button, False)
    
    self.remove_button = gtk.Button (stock=gtk.STOCK_REMOVE)
    self.remove_button.connect ('clicked', self.on_category_remove)
    self.remove_button.set_sensitive (False)
    self.remove_button.show ()
    hbox.pack_start (self.remove_button, False)

    button = gtk.Button (stock=gtk.STOCK_CLOSE)
    button.connect ('clicked', self.on_extension_close)
    button.show ()
    hbox.pack_end (button, False)
    
    self.save_button = gtk.Button (stock=gtk.STOCK_SAVE)
    self.save_button.set_sensitive (False)
    self.save_button.connect ('clicked', self.on_model_save)
    self.save_button.show ()
    hbox.pack_end (self.save_button, False)

    # build attribute dict
    self.category_attributes = {}

    for category in self.mediator.call ('category.get-list'):
      datastore = gtk.ListStore (str, str, str)

      for a in category.get_attribute_list ():
        datastore.append ((a.id, a.name, a.default))

      self.category_attributes[category.id] = datastore

    # flags
    self.is_modified = False	# categories are modified
    self.is_selecting = False	# selecting new category

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Build category list
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def build_category_list (self):
    datastore = gtk.ListStore (gtk.gdk.Pixbuf, str, object)
    datastore.set_sort_column_id (CATEGORY_NAME, gtk.SORT_ASCENDING)
    categories = self.mediator.call ('category.get-list')
    #categories.sort ()

    for category in categories:
      icon = self.mediator.call ('ui.render-icon', category.icon_data, 32, 32)
      datastore.append ((icon, category.name, category))

    # listview
    frame = gtk.Frame ()

    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_NEVER, gtk.POLICY_AUTOMATIC)
    sw.show ()
    frame.add (sw)

    self.category_listview = gtk.TreeView (datastore)
    self.category_listview.set_rules_hint (True)
    self.category_listview.set_headers_visible (False)
    self.category_listview.connect ('cursor-changed', self.on_category_selected)
    self.category_listview.show ()
    sw.add (self.category_listview)

    renderer = gtk.CellRendererPixbuf ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'pixbuf', CATEGORY_ICON)
    self.category_listview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', CATEGORY_NAME)
    self.category_listview.append_column (tvcolumn)

    return frame

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief show save/ignore/cancel dialog if there are modified items
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_widget_stopped (self):
    model = self.category_listview.get_model ()

    # if model was modified, show save/ignore/cancel dialog
    if self.is_modified:
      dialog = gtk.MessageDialog (None,
                  gtk.DIALOG_MODAL,
                  gtk.MESSAGE_QUESTION,
                  gtk.BUTTONS_YES_NO,
                  "Save changes before closing?")
      dialog.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
      rc = dialog.run ()
      dialog.destroy ()

      if rc == gtk.RESPONSE_CANCEL:
        return True

      elif rc == gtk.RESPONSE_YES:
        self.on_model_save ()

    # clean-up code
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle model modification
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_model_modified (self):
    self.is_modified = True
    self.save_button.set_sensitive (True)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief save model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_model_save (self, *args):
    model = self.category_listview.get_model ()
    categories = []

    for icon, name, category in model:
      datastore_attr = self.category_attributes.get (category.id)
      category.clear_attributes ()

      for id, name, default in datastore_attr:
        attr = Attribute ()
        attr.id = id
        attr.name = name
        attr.default = default
        category.add_attribute (attr)

      categories.append (category)

    self.mediator.call ('category.set-list', categories)
    self.is_modified = False
    self.save_button.set_sensitive (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_selected (self, widget, *args):
    model, iter = widget.get_selection ().get_selected ()

    if iter:
      self.is_selecting = True

      icon = model.get_value (iter, CATEGORY_ICON)
      category = model.get_value (iter, CATEGORY_OBJ)
      datastore_attr = self.category_attributes.get (category.id)

      self.category_id_entry.set_text (category.id or '')
      self.category_name_entry.set_text (category.name or '')
      self.category_image.set_from_pixbuf (icon)
      self.category_is_dataholder_cb.set_active (category.is_dataholder)
      self.attribute_listview.set_model (datastore_attr)

      self.category_id_entry.set_sensitive (True)
      self.category_name_entry.set_sensitive (True)
      self.category_icon_button.set_sensitive (True)
      self.category_is_dataholder_cb.set_sensitive (True)

      self.add_attr_button.set_sensitive (True)
      self.remove_button.set_sensitive (True)
      self.on_attribute_unselected ()
      self.is_selecting = False

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category unselection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_unselected (self):
    self.is_selecting = True
    self.remove_button.set_sensitive (False)

    self.category_id_entry.set_text ('')
    self.category_id_entry.set_sensitive (False)
    self.category_name_entry.set_text ('')
    self.category_name_entry.set_sensitive (False)
    self.category_image.clear ()
    self.category_icon_button.set_sensitive (False)
    self.category_is_dataholder_cb.set_active (False)
    self.category_is_dataholder_cb.set_sensitive (False)

    self.add_attr_button.set_sensitive (False)
    self.attribute_listview.set_model (None)
    self.is_selecting = False

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category add
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_add (self, widget, *args):
    icon = self.mediator.call ('ui.render-icon', ICON_NEW_CATEGORY, 32, 32)
    name = '<NEW CATEGORY>'

    category = Category ()
    category.id = '<NEW ID>'
    category.name = name
    category.icon = icon
    category.icon_data = ICON_NEW_CATEGORY

    model = self.category_listview.get_model ()
    model.append ((icon, name, category))

    datastore = gtk.ListStore (str, str, str)
    self.category_attributes[category.id] = datastore
    self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category remove
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_remove (self, widget, *args):
    selection = self.category_listview.get_selection ()
    model, iter = selection.get_selected ()

    if iter:
      category = model.get_value (iter, CATEGORY_OBJ)

      dialog = gtk.MessageDialog (None,
        gtk.DIALOG_MODAL,
        gtk.MESSAGE_QUESTION,
        gtk.BUTTONS_YES_NO,
        "You are about to remove category '%s'. Are you sure?" % category.name)

      rc = dialog.run ()
      dialog.destroy ()

      if rc != gtk.RESPONSE_YES:
        return

      # remove category
      has_next = model.remove (iter)

      if has_next:
        selection.select_iter (iter)
      else:
        self.on_category_unselected ()

      # remove attribute datastore
      self.category_attributes.pop (category.id)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category id edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_id_changed (self, widget, *args):
    model, iter = self.category_listview.get_selection ().get_selected ()

    if self.is_selecting == False and iter:
      value = self.category_id_entry.get_text ()
      category = model.get_value (iter, CATEGORY_OBJ)
      old_id = category.id
      category.id = value

      datastore = self.category_attributes.pop (old_id)
      self.category_attributes[category.id] = datastore
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category name edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_name_changed (self, widget, *args):
    model, iter = self.category_listview.get_selection ().get_selected ()

    if self.is_selecting == False and iter:
      value = self.category_name_entry.get_text ()
      category = model.get_value (iter, CATEGORY_OBJ)
      category.name = value
      model.set_value (iter, CATEGORY_NAME, value)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category icon edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_icon_edit (self, widget, *args):

    # choose file
    fs = gtk.FileChooserDialog ('Choose 32x32 icon filename')
    fs.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
    fs.add_button (gtk.STOCK_OK, gtk.RESPONSE_OK)

    filter = gtk.FileFilter ()
    filter.add_pattern ('*.png')
    filter.add_pattern ('*.jpg')
    filter.add_pattern ('*.gif')
    filter.add_pattern ('*.svg')
    fs.set_filter (filter)

    rc = fs.run ()
    filename = fs.get_filename ()
    fs.destroy ()

    if rc != gtk.RESPONSE_OK:
      return

    # read file
    fp = open (filename)
    data = fp.read ()
    fp.close ()

    # set new icon
    model, iter = self.category_listview.get_selection ().get_selected ()

    if iter:
      category = model.get_value (iter, CATEGORY_OBJ)
      category.icon_data = data.encode ('base64')
      category.icon = self.mediator.call ('ui.render-icon', category.icon_data, 32, 32)

      self.category_image.set_from_pixbuf (category.icon)
      model.set_value (iter, CATEGORY_ICON, category.icon)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category is dataholder toggled
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_is_dataholder_toggled (self, widget, *args):
    model, iter = self.category_listview.get_selection ().get_selected ()

    if self.is_selecting == False and iter:
      category = model.get_value (iter, CATEGORY_OBJ)
      category.is_dataholder = widget.get_active ()
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_selected (self, widget, *args):
    model, iter = widget.get_selection ().get_selected ()

    if iter:
      id = model.get_value (iter, ATTR_ID)
      name = model.get_value (iter, ATTR_NAME)
      default = model.get_value (iter, ATTR_DEFAULT)
      row = model.get_path (iter)[0]

      self.default_value_entry.set_sensitive (True)
      self.default_value_entry.set_text (default or '')
      self.remove_attr_button.set_sensitive (True)

      if row > 0:
        self.up_attr_button.set_sensitive (True)
      else:
        self.up_attr_button.set_sensitive (False)

      if model.iter_next (iter) != None:
        self.down_attr_button.set_sensitive (True)
      else:
        self.down_attr_button.set_sensitive (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute unselection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_unselected (self, *args):
    self.remove_attr_button.set_sensitive (False)
    self.up_attr_button.set_sensitive (False)
    self.down_attr_button.set_sensitive (False)
    self.default_value_entry.set_text ('')
    self.default_value_entry.set_sensitive (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute add
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_add (self, widget, *args):
    model = self.attribute_listview.get_model ()
    iter = model.append (('<ID>', '<NAME>', ''))
    path = model.get_path (iter)
    column = self.attribute_listview.get_column (ATTR_ID)
    self.attribute_listview.set_cursor (path, column, True)
    self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute remotion
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_remove (self, widget, *args):
    selection = self.attribute_listview.get_selection ()
    model, iter = selection.get_selected ()

    if iter:
      has_next = model.remove (iter)

      if has_next:
        selection.select_iter (iter)
      else:
        self.on_attribute_unselected ()

      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle move up
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_up (self, widget, *args):
    model, iter = self.attribute_listview.get_selection ().get_selected ()

    if iter:
      dest_row = model.get_path (iter)[0] - 1
      dest_iter = model.get_iter (dest_row)
      model.move_before (iter, dest_iter)

      if dest_row == 0:
        self.up_attr_button.set_sensitive (False)

      self.down_attr_button.set_sensitive (True)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle move down
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_down (self, widget, *args):
    model, iter = self.attribute_listview.get_selection ().get_selected ()

    if iter:
      dest_row = model.get_path (iter)[0] + 1
      dest_iter = model.get_iter (dest_row)
      model.move_after (iter, dest_iter)

      if dest_row + 1 >= len (model):
        self.down_attr_button.set_sensitive (False)

      self.up_attr_button.set_sensitive (True)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_edited (self, cell, path, new_text, col, *args):
    model = self.attribute_listview.get_model ()
    iter = model.get_iter_from_string (path)
    text = cell.get_property ('text')

    if text != new_text:
      model.set_value (iter, col, new_text)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle default value edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_default_value_changed (self, widget, *args):
    selection = self.attribute_listview.get_selection ()
    model, iter = selection.get_selected ()

    if iter:
      value = self.default_value_entry.get_text ()
      model.set_value (iter, ATTR_DEFAULT, value)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle cancel button
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_extension_close (self, widget, *args):
    self.working_area.on_delete_event (widget, *args)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief icon data
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''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'''

ICON_NEW_CATEGORY = '''
iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAABGdBTUEAALGPC/xhBQAAAAZiS0dE
AP8A/wD/oL2nkwAAAAlwSFlzAAAN1wAADdcBQiibeAAAAAd0SU1FB9gHEAw6I15U6QIAAAMiSURB
VFjD7ddLaFxlFAfw351HnDxN05gapYoiVjREqSk+0LYQxHapG+tacCHiLmA36kJRXFmtohDEldLW
YAXBVSKE2IoPtKJQxFrFkqTWkpo6ySSdmevi3pDLdB6ZFIqLHPi4353zcc7/O//zuMOGrEsyD3P9
Yfp+of87rnsN/VfJedvTDMxxKORkyImQ0TKbz+O+Zq2lmgfQfj/DndyJa9CLRwIe7cJjVyME3XQf
ZGeeUyFnQrbl6ZjASNNk1tF14vbqqotTzD9BuY0yLpT4dxyTuLficIgTKFWzFNT2n3uJ3H4G5iOm
kqs9x3MdbI1p+Cnkg4sEy5E+afajXjyI401GIIUnW9jVGzmptsL47PaAB7rIxb8nUyvI8+HeWgAa
JGFYsWqdqWsjhbZ1VkHaakhrsRU00F9RGaZjliqBXJnTegCG4oSpAiCdAJKqAyjABL5ttgwzw2w9
EpXL6SGybVFSZWuACGqAebvIC9O0dnGym4UQl9YAoOcABzYxHfL89+RK7IgBVItEusL5yv7VJebe
xZ/ccZDiIn5rBCBNto8t2BxweFPkeCXJ0lVApKpQchrheXzJ3CRzh/BQrKgLIEdruPraKepwYcL4
CoBMYp+qALKEYCm6PXHov1hLEuaZL0ZGL+CZBd4q0BJTkE3kQi0q0tiGcm/kODvEjT9y8yuN6iyW
jltYvIdPljn+MTPYcUM05jMVK1vnORXyR4m+EaYG+WyAmU/xdwMA+XF+v4mZSyweJXuWnbu5tSLs
mYpoJJ3D3ix/DbGvh+Es4yV+nsWxRrNgmXNPoT1+H6EooiGoyIOxEqMFzpUJUgzi5fZoEF6L91oT
d0xHObb2cZxf3ZbjmyUBvFninVPMH8E3OMv0XXz1Okd72NXU0Ms07lPZRK2nUShhAe8n6vsMweNM
76nR3luvYBasVMLK88UW3rib/rHVc1uOMbqHfalm/WTqX//zGHyyz6cwU6LUjdvwK0GJQsB4FTM/
ZNf5SVYcY7KLicHLdYV/KH4dU4HZ3Ty7n/T2y0dkoROz6gzzRpJLdLXSGoZbpsZXy9LG/6kN+V/K
f7T3xDg+BD+/AAAAAElFTkSuQmCC'''

ICON_UNKNOWN_CATEGORY = ''' PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiIHN0YW5kYWxvbmU9Im5vIj8+Cjwh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'''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'category-manager'
    self.name = 'Category Manager'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.1'
    self.description = 'Category manager'
    self.mediator = mediator.new_client_mediator ()
    self.icon_data = ICON_DATA

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
    self.model = Model (self.mediator)
    self.mediator.call ('toolbox.add', self.id, self.icon_data, 'Category\nManager', self.on_activate)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    self.mediator.call ('ui.working-area.del', self.id)
    #self.mediator.call ('toolbox.remove', self.id)
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_activate
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_activate (self, item_id):
    working_area = self.mediator.call ('ui.working-area.get', self.id)

    if not working_area:
      widget = Widget (self.mediator)

      working_area = self.mediator.call ('ui.working-area.new', self.id)
      working_area.set_default_size (600, 400)
      working_area.set_title (self.name)
      working_area.set_icon (self.icon_data)
      working_area.set_widget (widget)

    working_area.show ()
