# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import gtk
import mobius.config

(EXTENSION_ICON,
EXTENSION_NAME,
EXTENSION_VERSION,
EXTENSION_OBJECT) = range (4)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension manager dialog
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Dialog (gtk.Dialog):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Build widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, *args):
    gtk.Dialog.__init__ (self,
     '%s - Extension Manager' % mobius.config.APP_TITLE, None, gtk.DIALOG_MODAL,
      (gtk.STOCK_ADD, 1, gtk.STOCK_REMOVE, 2, gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE))
    self.set_position (gtk.WIN_POS_CENTER)
    self.set_default_size (580, 380)
    self.set_type_hint (gtk.gdk.WINDOW_TYPE_HINT_DIALOG)
    self.set_border_width (10)

    self.vbox.set_spacing (5)

    # listview
    datastore = gtk.ListStore (gtk.gdk.Pixbuf, str, str, object)
    extensions = [(e.name, e) for e in mobius.app.get_resource_list ('extension')]
    extensions.sort ()

    for name, extension in extensions:
      icon = mobius.mediator.call ('ui.render-icon', extension.icon_data, 32, 32)
      datastore.append ((icon, name, extension.version, extension))

    self.listview = gtk.TreeView (datastore)
    self.listview.show ()

    renderer = gtk.CellRendererPixbuf ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'pixbuf', EXTENSION_ICON)
    self.listview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ('Extension')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', EXTENSION_NAME)
    self.listview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ('Version')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', EXTENSION_VERSION)
    self.listview.append_column (tvcolumn)

    self.vbox.pack_start (self.listview)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Run dialog
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def run (self):
    rc = None

    while rc != gtk.RESPONSE_CLOSE:
      if rc == 1:
        self.add_extension ()

      elif rc == 2:   # remove
        self.remove_extension ()
 
      rc = gtk.Dialog.run (self)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add entry to listview, checking if it already exists
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_listview_entry (self, extension):
    model = self.listview.get_model ()
    icon = getattr (extension, 'icon', None)
    in_listview = False

    # check if extension is in listview 
    iter = model.get_iter_first ()

    while iter and not in_listview:
      ext = model.get_value (iter, EXTENSION_OBJECT)

      if ext.id == extension.id:
        in_listview = True

      elif ext.name > extension.name:
        break

      else:
        iter = model.iter_next (iter)

    # if extension is already in listview, just update data
    if in_listview:
      model.set_value (iter, EXTENSION_ICON, icon)
      model.set_value (iter, EXTENSION_OBJECT, extension)
      model.set_value (iter, EXTENSION_NAME, extension.name)
      model.set_value (iter, EXTENSION_VERSION, extension.version)
  
    # otherwise, insert preserving alphabetic order
    else:
      model.insert_before (iter, (icon, extension.name, extension.version, extension))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_extension (self):

    # choose file
    fs = gtk.FileChooserDialog ('Add extension', parent=self)
    fs.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
    fs.add_button (gtk.STOCK_OK, gtk.RESPONSE_OK)

    filter = gtk.FileFilter ()
    filter.add_pattern ('*.py')
    fs.set_filter (filter)

    rc = fs.run ()
    path = fs.get_filename ()
    fs.destroy ()

    if rc != gtk.RESPONSE_OK:
      return

    # install extension
    extension = mobius.app.import_extension (path)

    if self.can_install (extension):

      # remove old version, first
      if mobius.app.get_resource ('extension', extension.id):
        mobius.app.uninstall_extension (extension.id)

      # install extension
      mobius.app.install_extension (extension)
      mobius.app.start_extension (extension)
      self.add_listview_entry (extension)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Remove selected extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def remove_extension (self):
    model, iter = self.listview.get_selection ().get_selected ()
    if not iter:
      return

    # get data from liststore
    name = model.get_value (iter, EXTENSION_NAME)
    extension = model.get_value (iter, EXTENSION_OBJECT)

    # show confirmation dialog
    dialog = gtk.MessageDialog (self,
             gtk.DIALOG_MODAL,
             gtk.MESSAGE_QUESTION,
             gtk.BUTTONS_YES_NO,
             "Are you sure you want to uninstall extension '%s'?" % name)
    rc = dialog.run ()
    dialog.destroy ()

    # if user choose yes, uninstall extension
    if rc == gtk.RESPONSE_YES:
      mobius.app.uninstall_extension (extension.id)
      model.remove (iter)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Check if extension can be installed
  # @return True for install, False otherwise
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def can_install (self, extension):
    app_extension = mobius.app.get_resource ('extension', extension.id)
    flag_install = False

    # check if extension is already installed
    if app_extension:
      version = [int (n) for n in extension.version.split ('.')]
      app_version = [int (n) for n in app_extension.version.split ('.')]

      # same version
      if version == app_version:
        dialog = gtk.MessageDialog (self,
           gtk.DIALOG_MODAL,
           gtk.MESSAGE_ERROR,
           gtk.BUTTONS_OK,
           "Extension '%s v%s' is already installed" % (extension.name, extension.version))
        rc = dialog.run ()
        dialog.destroy ()

      # older version
      elif version < app_version:
        dialog = gtk.MessageDialog (self,
           gtk.DIALOG_MODAL,
           gtk.MESSAGE_WARNING,
           gtk.BUTTONS_YES_NO,
           "Extension '%s v%s' is installed. Are you sure you want to downgrade to v%s'?" % (extension.name, app_extension.version, extension.version))
        rc = dialog.run ()
        dialog.destroy ()

        if rc == gtk.RESPONSE_YES:
          flag_install = True

      # newer version
      else:
        flag_install = True

    # new extension
    else:
      flag_install = True

    return flag_install
