# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
XML_ENCODING='utf-8'

import libxml2
import os.path
import gtk

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Part
# =i=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Part (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.category = None
    self.number = None
    self.attributes = {}

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get attribute
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_attribute (self, id):
    return self.attributes.get (id)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Persistence layer for PartCatalogue
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Pickle (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load parts.xml file
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load (self, path):
    parts = []

    if os.path.exists (path):
      doc = libxml2.parseFile (path)
      node = doc.getRootElement ()
      parts = self.load_parts (node)

    return parts

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <parts>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_parts (self, node):
    parts = []

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'part':
        part = self.load_part (node)
        parts.append (part)

      node = node.next

    return parts

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <part>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_part (self, node):
    part = Part ()
    part.category = self.__get_prop (node, 'category')
    part.number = self.__get_prop (node, 'number')
    part.attributes = {}

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'attribute':
        id, value = self.load_attribute (node)
        part.attributes[id] = value

      node = node.next

    return part

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <attribute>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_attribute (self, node):
    id = self.__get_prop (node, 'id')
    value = self.__get_prop (node, 'value')

    return id, value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save XML
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save (self, path, parts):
    doc = libxml2.newDoc ('1.0')
    node = self.save_catalogue (parts)
    doc.addChild (node)
    doc.saveFormatFileEnc (path, XML_ENCODING, 1)
    doc.freeDoc ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <catalogue>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_catalogue (self, parts):
    node = libxml2.newNode ('catalogue')

    for part in parts:
      child = self.save_part (part)
      node.addChild (child)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <part>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_part (self, part):
    node = libxml2.newNode ('part')
    self.__set_prop (node, 'category', part.category)
    self.__set_prop (node, 'number', part.number)

    for id, value in part.attributes.iteritems ():
      if value != '' and value != None:
        child = self.save_attribute (id, value)
        node.addChild (child)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <attribute>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_attribute (self, id, value):
    node = libxml2.newNode ('attribute')
    self.__set_prop (node, 'id', id)
    self.__set_prop (node, 'value', value)
    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __get_prop (self, node, name):
    value = node.prop (name)
    if value:
      value = value.decode (XML_ENCODING)
    return value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __set_prop (self, node, name, value):
    if value != None:
      node.setProp (name, value.encode (XML_ENCODING))

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Manage part catalogue model
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Model (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, mediator):
    self.mediator = mediator
    self.loaded = False

    self.mediator.advertise ('part.get-list', self.svc_part_get_list)
    self.mediator.advertise ('part.set-list', self.svc_part_set_list)
    self.mediator.connect ('item.attribute-modified', self.on_item_attribute_modified)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.get-list
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_get_list (self):
    if not self.loaded:
      self.__load_model ()
    return self.__parts.itervalues ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.set-list
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_set_list (self, parts):
    self.__parts = {}
    self.__parts.update (((part.category, part.number), part) for part in parts)
    self.__save_model ()

    if not self.loaded:
      self.__load_model ()
    return self.__parts.itervalues ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Load model from XML
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __load_model (self):
    pickle = Pickle ()
    path = self.mediator.call ('app.get-path', 'part-catalogue.xml')

    self.__parts = {}
    self.__parts.update ((((part.category, part.number), part) for part in pickle.load (path)))
    self.loaded = True

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save model into XML
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __save_model (self):
    pickle = Pickle ()
    path = self.mediator.call ('app.get-path', 'part-catalogue.xml')
    pickle.save (path, self.__parts.values ())

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Event handler: on_item_attribute_modified
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_item_attribute_modified (self, item, attr, text, new_text):
    if attr == 'part_number' and new_text != '':
      if not self.loaded:
        self.__load_model ()
      part = self.__parts.get ((item.category.id, new_text))

      if part:
        for attr, value in part.attributes.iteritems ():
          self.mediator.call ('item.set-attribute', item, attr, value)
      else:
         part = Part ()
         part.category = item.category.id
         part.number = new_text

         # @todo find a better place for this code
         self.window = Window (self.mediator)
         self.window.add_new_part (part)
         self.window.show ()
         self.window.select_part (part)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Windows constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
(ITEM_TYPE, ITEM_ICON, ITEM_NAME, ITEM_OBJ) = range (4)
(ATTR_ID, ATTR_NAME, ATTR_VALUE) = range (3)
(TYPE_CATEGORY, TYPE_PART) = range (2)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Window
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Window (gtk.Window):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, mediator):
    gtk.Window.__init__ (self)
    self.connect ('delete-event', self.on_delete)
    self.set_title (mediator.call ('app.get-window-title', 'Part Catalogue'))
    self.set_default_size (600, 400)
    self.mediator = mediator

    vbox = gtk.VBox ()
    vbox.set_border_width (5)
    vbox.set_spacing (5)
    vbox.show ()
    self.add (vbox)

    hpaned = gtk.HPaned ()
    hpaned.show ()
    vbox.pack_start (hpaned)

    # part treeview
    frame = gtk.Frame ()
    frame.show ()
    hpaned.pack1 (frame, True)

    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    frame.add (sw)

    model = gtk.TreeStore (int, gtk.gdk.Pixbuf, str, object)
    model.set_sort_column_id (ITEM_NAME, gtk.SORT_ASCENDING)

    self.treeview = gtk.TreeView (model)
    self.treeview.set_headers_visible (False)
    self.treeview.set_enable_tree_lines (True)
    self.treeview.show ()
    sw.add (self.treeview)
    self.treeview.connect ('cursor-changed', self.on_treeitem_selected)

    renderer = gtk.CellRendererPixbuf ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'pixbuf', ITEM_ICON)
    self.treeview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ITEM_NAME)
    self.treeview.append_column (tvcolumn)

    # attribute listview
    frame = gtk.Frame ()
    frame.show ()
    hpaned.pack2 (frame)

    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    frame.add (sw)

    datastore = gtk.ListStore (str, str, str)
    self.attribute_listview = gtk.TreeView (datastore)
    self.attribute_listview.set_rules_hint (True)
    self.attribute_listview.set_enable_search (False)

    renderer = gtk.CellRendererText ()

    tvcolumn = gtk.TreeViewColumn ('Name')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ATTR_NAME)
    tvcolumn.set_resizable (True)
    self.attribute_listview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    renderer.set_property ('editable', True)
    renderer.connect ('edited', self.on_attribute_edited)

    tvcolumn = gtk.TreeViewColumn ('Value')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ATTR_VALUE)
    self.attribute_listview.append_column (tvcolumn)

    self.attribute_listview.show ()
    sw.add (self.attribute_listview)

    # buttons
    hbox = gtk.HBox ()
    hbox.show ()
    vbox.pack_start (hbox, False)

    self.remove_button = gtk.Button (stock=gtk.STOCK_REMOVE)
    self.remove_button.set_sensitive (False)
    self.remove_button.connect ('clicked', self.on_part_remove)
    self.remove_button.show ()
    hbox.pack_start (self.remove_button, False)

    button = gtk.Button (stock=gtk.STOCK_CLOSE)
    button.connect ('clicked', self.on_extension_close)
    button.show ()
    hbox.pack_end (button, False)
    
    self.save_button = gtk.Button (stock=gtk.STOCK_SAVE)
    self.save_button.set_sensitive (False)
    self.save_button.connect ('clicked', self.on_model_save)
    self.save_button.show ()
    hbox.pack_end (self.save_button, False)

    self.part_icon = self.mediator.call ('icon.create', ICON_DATA)
    self.category_iter = {}
    self.is_modified = False

    for part in self.mediator.call ('part.get-list'):
      self.add_part (part)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief add part to treeview
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def add_part (self, part):
    model = self.treeview.get_model ()

    # add category, if necessary
    iter = self.category_iter.get (part.category)
    if not iter:
      category = self.mediator.call ('category.get', part.category)
      iter = model.append (None, (TYPE_CATEGORY, category.icon, category.name, category))
      self.category_iter[part.category] = iter

    # add part to category
    return model.append (iter, (TYPE_PART, self.part_icon, part.number, part))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief add new part to treeview
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def add_new_part (self, part):
    iter = self.add_part (part)
    self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief select a given part in treeview
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def select_part (self, part):
    model = self.treeview.get_model ()
    iter = model.get_iter_first ()

    while iter:
      child = model.iter_children (iter)

      while child:
        tree_part = model.get_value (child, ITEM_OBJ)
        if part == tree_part:
          path = model.get_path (child)
          self.treeview.expand_to_path (path)
          selection = self.treeview.get_selection ()
          selection.select_iter (child)
          self.on_treeitem_selected (self.treeview)
          break
        child = model.iter_next (child)
      iter = model.iter_next (iter)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief show save/ignore/cancel dialog if there are modified items
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def can_leave (self):

    # if model was modified, show save/ignore/cancel dialog
    if self.is_modified:
      dialog = gtk.MessageDialog (self,
                  gtk.DIALOG_MODAL,
                  gtk.MESSAGE_QUESTION,
                  gtk.BUTTONS_YES_NO,
                  "Save changes before closing?")
      dialog.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
      rc = dialog.run ()
      dialog.destroy ()

      if rc == gtk.RESPONSE_CANCEL:
        return False

      elif rc == gtk.RESPONSE_YES:
        self.on_model_save ()

    return True

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle destroy event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_delete (self, widget, *args):
    return not self.can_leave ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle close event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_extension_close (self, widget, *args):
    if self.can_leave ():
      self.destroy ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle model modification
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_model_modified (self):
    self.is_modified = True
    self.save_button.set_sensitive (True)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle model-save event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_model_save (self, widget, *args):
    model = self.treeview.get_model ()
    iter = model.get_iter_first ()
    parts = []

    while iter:
      child = model.iter_children (iter)
      while child:
        part = model.get_value (child, ITEM_OBJ)
        parts.append (part)
        child = model.iter_next (child)
      iter = model.iter_next (iter)

    self.mediator.call ('part.set-list', parts)
    self.is_modified = False
    self.save_button.set_sensitive (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle treeitem-selected event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_treeitem_selected (self, widget, *args):
    model, iter = widget.get_selection ().get_selected ()

    if iter:
      item_type = model.get_value (iter, ITEM_TYPE) 

      if item_type == TYPE_PART:
        part = model.get_value (iter, ITEM_OBJ) 
        self.on_part_selected (part)
      else:
        category = model.get_value (iter, ITEM_OBJ) 
        self.on_category_selected (category)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle part-selected event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_part_selected (self, part):
    self.attribute_listview.set_sensitive (True)
    self.remove_button.set_sensitive (True)

    # populate attribute-listview
    category = self.mediator.call ('category.get', part.category)
    model = self.attribute_listview.get_model ()
    model.clear ()

    for a in category.get_attribute_list ():
      if a.id != 'part_number':
        model.append ((a.id, a.name, part.get_attribute (a.id)))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle part-unselected event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_part_unselected (self):
    self.attribute_listview.set_sensitive (False)
    self.remove_button.set_sensitive (False)

    # clear attribute-listview
    model = self.attribute_listview.get_model ()
    model.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category-selected event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_selected (self, category):
    self.attribute_listview.set_sensitive (False)
    self.remove_button.set_sensitive (False)

    # clear attribute-listview
    model = self.attribute_listview.get_model ()
    model.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle part-remove event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_part_remove (self, widget, *args):
    selection = self.treeview.get_selection ()
    model, iter = selection.get_selected ()

    if iter:
      part = model.get_value (iter, ITEM_OBJ)

      dialog = gtk.MessageDialog (self,
        gtk.DIALOG_MODAL,
        gtk.MESSAGE_QUESTION,
        gtk.BUTTONS_YES_NO,
        "You are about to remove part '%s'. Are you sure?" % part.number)

      rc = dialog.run ()
      dialog.destroy ()

      if rc != gtk.RESPONSE_YES:
        return

      # remove part
      has_next = model.remove (iter)

      if has_next:
        selection.select_iter (iter)
      else:
        self.on_part_unselected ()

      # signal model modification
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute-edited event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_edited (self, cell, path, new_text, *args):
    model = self.attribute_listview.get_model ()
    iter = model.get_iter_from_string (path)
    text = cell.get_property ('text')

    if text != new_text:
      model.set_value (iter, ATTR_VALUE, new_text)
      id = model.get_value (iter, ATTR_ID)
 
      treemodel, iter = self.treeview.get_selection ().get_selected ()
      if iter:
        part = treemodel.get_value (iter, ITEM_OBJ)
        part.attributes[id] = new_text

      self.on_model_modified ()

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief 32x32 PNG icon
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''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'''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'part_catalogue'
    self.name = 'Part Catalogue'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.0'
    self.description = 'Part/partnumber catalogue and manager'
    self.mediator = mediator
    self.icon = mediator.call ('icon.create', ICON_DATA)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
    self.model = Model (self.mediator)
    self.mediator.call ('toolbox.add', 'part_catalogue', ICON_DATA, 'Part\nCatalogue', self.on_clicked)
    self.window = None

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    if self.window:
      self.window.destroy ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_clicked
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_clicked (self, id):
    self.window = Window (self.mediator)
    self.window.show ()
