/*- <matfquad/net/socket/Address.cpp> -*- C++ -*-
 *
 *
 *  matfquad
 *  Copyright (C) 2012  Márcio Adriano Tavares Fernandes
 *
 *  This library is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
#include "matfquad/net/socket/Address.hpp"
#include <cerrno>
#include <cstring>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
using namespace matfquad::net::socket;

const int Address::NAMETOADDRESS_CANONNAME = AI_CANONNAME;
const int Address::NAMETOADDRESS_NUMERICHOST = AI_NUMERICHOST;
const int Address::NAMETOADDRESS_PASSIVE = AI_PASSIVE;
const int Address::NAMETOADDRESS_DEFAULT = Address::NAMETOADDRESS_CANONNAME|Address::NAMETOADDRESS_PASSIVE;

const int Address::ADDRESSTONAME_NAMEREQUIRED = NI_NAMEREQD;
const int Address::ADDRESSTONAME_NONFULLYQUALIFIEDDOMAINNAME = NI_NOFQDN;
const int Address::ADDRESSTONAME_NUMERICHOST = NI_NUMERICHOST;
const int Address::ADDRESSTONAME_NUMERICSERVICE = NI_NUMERICSERV;
const int Address::ADDRESSTONAME_DEFAULT = 0;

Address::Address()
{
  this->name_to_address_flags = Address::NAMETOADDRESS_DEFAULT;
  this->address_to_name_flags = Address::ADDRESSTONAME_DEFAULT;
  this->family = Family();
  this->type = Type();
  this->protocol = Protocol();
  this->address_length = 0;
  this->address = NULL;
  this->next = NULL;
}

Address::Address(std::string hostname, std::string service, int family, int type, int protocol) throw (address_error): Address(hostname, service, family, type, protocol, Address::NAMETOADDRESS_DEFAULT)
{
}

Address::Address(std::string hostname, std::string service, int family, int type, int protocol, int name_to_address_flags) throw (address_error): Address()
{
  struct addrinfo hints;
  memset(&hints, '\0', sizeof(addrinfo));
  hints.ai_flags = name_to_address_flags;
  hints.ai_family = family;
  hints.ai_socktype = type;
  hints.ai_protocol = protocol;
  struct addrinfo *res = NULL;
  if ( getaddrinfo(hostname.size() > 0 ? hostname.c_str() : NULL, service.size() > 0 ? service.c_str(): NULL, &hints, &res) != 0 )
  {
    int errnum = errno;
    throw address_error(gai_strerror(errnum));
  }
  this->assign((void *)res);
  freeaddrinfo(res);
  res = NULL;
}

Address::Address(std::string hostname, std::string service, Family *family, Type *type, Protocol *protocol) throw (address_error): Address(hostname, service, family, type, protocol, Address::NAMETOADDRESS_DEFAULT)
{
}

Address::Address(std::string hostname, std::string service, Family *family, Type *type, Protocol *protocol, int name_to_address_flags) throw (address_error): Address(hostname, service, family->getNumber(), type->getNumber(), protocol->getNumber(), name_to_address_flags)
{
}

Address::Address(void *address, size_t address_length, int family, int type, int protocol) throw (address_error): Address(address, address_length, family, type, protocol, Address::NAMETOADDRESS_DEFAULT, Address::ADDRESSTONAME_DEFAULT)
{
}

Address::Address(void *address, size_t address_length, int family, int type, int protocol, int name_to_address_flags, int address_to_name_flags) throw (address_error): Address()
{
  this->name_to_address_flags = name_to_address_flags;
  this->address_to_name_flags = address_to_name_flags;
  this->family = Family(family);
  this->type = Type(type);
  this->protocol = Protocol(protocol);
  if ( this->type.getNumber() == Type_DGRAM::getInstance()->getNumber() )
  {
    address_to_name_flags |= NI_DGRAM;
  }
  this->hostname.clear();
  this->hostname.resize(1025, '\0');
  if ( getnameinfo((struct sockaddr *)address, address_length, (char *)this->hostname.c_str(), this->hostname.size()-1, NULL, 0, address_to_name_flags) != 0 )
  {
    throw address_error(gai_strerror(errno));
  }
  this->hostname.resize(strlen(this->hostname.c_str()));
}

Address::Address(void *address, size_t address_length, Family *family, Type *type, Protocol *protocol) throw (address_error): Address(address, address_length, family->getNumber(), type->getNumber(), protocol->getNumber(), Address::NAMETOADDRESS_DEFAULT, Address::ADDRESSTONAME_DEFAULT)
{
}

Address::Address(void *address, size_t address_length, Family *family, Type *type, Protocol *protocol, int name_to_address_flags, int address_to_name_flags) throw (address_error): Address(address, address_length, family->getNumber(), type->getNumber(), protocol->getNumber(), name_to_address_flags, address_to_name_flags)
{
}

Address::Address(void *hints): Address()
{
  this->assign(hints);
}


Address::Address(Address *address): Address()
{
  this->assign(address);
}

Address::~Address()
{
  if ( this->address != NULL )
  {
    free(this->address);
    this->address = NULL;
  }
  if ( this->next != NULL )
  {
    delete this->next;
    this->next = NULL;
  }
}

void Address::assign(Address *address)
{
  this->clear();
  this->name_to_address_flags = address->name_to_address_flags;
  this->address_to_name_flags = address->address_to_name_flags;
  this->family = address->family;
  this->type = address->type;
  this->protocol = address->protocol;
  this->address_length = address->address_length;
  if ( address->address != NULL )
  {
    this->address = (void *)malloc(this->address_length);
    if ( this->address == NULL )
    {
      exit(EXIT_FAILURE);
    }
    memset(this->address, '\0', this->address_length);
    memcpy(this->address, address->address, this->address_length);
  }
  this->hostname.assign(address->hostname);
  if ( address->next != NULL )
  {
    this->next = new Address(address->next);
  }
}

void Address::assign(void *hints)
{
  this->clear();
  this->name_to_address_flags = ((struct addrinfo *)hints)->ai_flags;
  this->family = ((struct addrinfo *)hints)->ai_family;
  this->type = ((struct addrinfo *)hints)->ai_socktype;
  this->protocol = ((struct addrinfo *)hints)->ai_protocol;
  this->address_length = ((struct addrinfo *)hints)->ai_addrlen;
  if ( this->address_length > 0 )
  {
    this->address = (struct addrinfo *)malloc(this->address_length);
    if ( this->address == NULL )
    {
      exit(EXIT_FAILURE);
    }
    memset(this->address, '\0', this->address_length);
    memcpy(this->address, ((struct addrinfo *)hints)->ai_addr, this->address_length);
  }
  if ( ((struct addrinfo *)hints)->ai_canonname != NULL )
  {
    this->hostname.assign(((struct addrinfo *)hints)->ai_canonname);
  }
  if ( ((struct addrinfo *)hints)->ai_next != NULL )
  {
    this->next = new Address((void *)(((struct addrinfo *)hints)->ai_next));
  }
}

void Address::clear()
{
  this->name_to_address_flags = Address::NAMETOADDRESS_DEFAULT;
  this->address_to_name_flags = Address::ADDRESSTONAME_DEFAULT;
  this->family = Family();
  this->type = Type();
  this->protocol = Protocol();
  this->address_length = 0;
  if ( this->address != NULL )
  {
    free(this->address);
    this->address = NULL;
  }
  this->hostname.assign("");
  if ( this->next != NULL )
  {
    delete this->next;
    this->next = NULL;
  }
}

void *Address::getAddress()
{
  return this->address;
}

size_t Address::getAddressLength()
{
  return this->address_length;
}

int Address::getAddressToNameFlags()
{
  return this->address_to_name_flags;
}

Family Address::getFamily()
{
  return this->family;
}

int Address::getFamilyNumber()
{
  return this->family.getNumber();
}

std::string Address::getHostname()
{
  return this->hostname;
}

int Address::getNameToAddressFlags()
{
  return this->name_to_address_flags;
}

Address *Address::getNext()
{
  return this->next;
}

Protocol Address::getProtocol()
{
  return this->protocol;
}

int Address::getProtocolNumber()
{
  return this->protocol.getNumber();
}

Type Address::getType()
{
  return this->type;
}

int Address::getTypeNumber()
{
  return this->type.getNumber();
}
