/* marray/marray_uchar.h
 *
 * Copyright (C) 2003, 2004, 2007 Jordi Burguet-Castell
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   Free Software Foundation, Inc.
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */

/*
 * This file contains the basic declarations for a marray.
 */
#ifndef __MARRAY_uchar_H__
#define __MARRAY_uchar_H__

#include <stdlib.h>


#include <gsl/gsl_types.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_check_range.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_vector.h>

#include "marray_utilities.h"

#undef __BEGIN_DECLS
#undef __END_DECLS
#ifdef __cplusplus
# define __BEGIN_DECLS extern "C" {
# define __END_DECLS }
#else
# define __BEGIN_DECLS /* empty */
# define __END_DECLS /* empty */
#endif

__BEGIN_DECLS


/*
 * A marray is a struct with the rank of the marray (number of
 * indices it has), an array with the number of elements available for
 * each index, and an array to store the n1*n2*...*nm values.
 *
 * For the moment, there is no tda, as opossed to matrices, because it
 * would complicate quite a bit the algorithms and probably it is not
 * worth it.
 */
typedef struct
{
  unsigned int rank;
  size_t * dimension;
  size_t size;
  unsigned char * data;
} marray_uchar;


/*
 * There is not such a thing as "marray views", in contrast with the
 * case for gsl_matrix.
 */

/* Allocation */

marray_uchar *
marray_uchar_alloc(const unsigned int rank, const size_t * dimension);

marray_uchar *
marray_uchar_calloc(const unsigned int rank, const size_t * dimension);

marray_uchar *
marray_uchar_copy(marray_uchar * t);

void marray_uchar_free(marray_uchar * t);


/* Views */

/*
 * There are no views.
 */


/* Conversions */

gsl_matrix_uchar * marray_uchar_2matrix(marray_uchar * t);
gsl_vector_uchar * marray_uchar_2vector(marray_uchar * t);


/* Operations */

unsigned char marray_uchar_get(const marray_uchar * t, const size_t * indices);
void marray_uchar_set(marray_uchar * t, const size_t * indices, const unsigned char x);


unsigned char * marray_uchar_ptr(marray_uchar * t, const size_t * indices);
const unsigned char * marray_uchar_const_ptr(const marray_uchar * t, const size_t * indices);

void marray_uchar_set_zero(marray_uchar * t);
void marray_uchar_set_all(marray_uchar * t, unsigned char x);

int marray_uchar_fread(FILE * stream, marray_uchar * t);
int marray_uchar_fwrite(FILE * stream, const marray_uchar * t);
int marray_uchar_fscanf(FILE * stream, marray_uchar * t);
int marray_uchar_fprintf(FILE * stream, const marray_uchar * t, const char * format);

int marray_uchar_memcpy(marray_uchar * dest, const marray_uchar * src);
int marray_uchar_swap(marray_uchar * t1, marray_uchar * t2);

unsigned char marray_uchar_max(const marray_uchar * t);
unsigned char marray_uchar_min(const marray_uchar * t);
void marray_uchar_minmax(const marray_uchar * t, unsigned char * min_out, unsigned char * max_out);

void marray_uchar_max_index(const marray_uchar * t, size_t * indices);
void marray_uchar_min_index(const marray_uchar * t, size_t * indices);
void marray_uchar_minmax_index(const marray_uchar * t, size_t * imin, size_t * imax);

int marray_uchar_isnull(const marray_uchar * t);

int marray_uchar_add(marray_uchar * a, const marray_uchar * b);
int marray_uchar_sub(marray_uchar * a, const marray_uchar * b);
int marray_uchar_mul_elements(marray_uchar * a, const marray_uchar * b);
int marray_uchar_div_elements(marray_uchar * a, const marray_uchar * b);
int marray_uchar_scale(marray_uchar * a, const double x);
int marray_uchar_add_constant(marray_uchar * a, const double x);


/* inline functions if you are using GCC */

#ifdef HAVE_INLINE
extern inline
size_t
marray_uchar_position(const size_t * indices, const marray_uchar * t)
{
  size_t shift, position;
  unsigned int i;

  shift = t->size;
  position = 0;
  for (i = 0; i < t->rank; i++)
    {
#if GSL_RANGE_CHECK
      if (indices[i] >= t->dimension[i])
        return t->size;
#endif
      shift /= t->dimension[i];
      position += shift * indices[i];
    }

  return position;
}


extern inline 
unsigned char
marray_uchar_get(const marray_uchar * t, const size_t * indices)
{
  size_t position;

  position = marray_uchar_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VAL("index out of range", GSL_EINVAL, 0);
#endif

  return t->data[position];
}


extern inline
void
marray_uchar_set(marray_uchar * t, const size_t * indices, const unsigned char x)
{
  size_t position;
  
  position = marray_uchar_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VOID("index out of range", GSL_EINVAL);
#endif

  t->data[position] = x;
}


extern inline 
unsigned char *
marray_uchar_ptr(marray_uchar * t, const size_t * indices)
{
  size_t position;

  position = marray_uchar_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (unsigned char *) (t->data + position);
}


extern inline 
const unsigned char *
marray_uchar_const_ptr(const marray_uchar * t, const size_t * indices)
{
  size_t position;

  position = marray_uchar_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (const unsigned char *) (t->data + position);
} 

#endif

__END_DECLS

#endif /* __MARRAY_uchar_H__ */
