/* marray/marray_char.h
 *
 * Copyright (C) 2003, 2004, 2007 Jordi Burguet-Castell
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   Free Software Foundation, Inc.
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */

/*
 * This file contains the basic declarations for a marray.
 */
#ifndef __MARRAY_char_H__
#define __MARRAY_char_H__

#include <stdlib.h>


#include <gsl/gsl_types.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_check_range.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_vector.h>

#include "marray_utilities.h"

#undef __BEGIN_DECLS
#undef __END_DECLS
#ifdef __cplusplus
# define __BEGIN_DECLS extern "C" {
# define __END_DECLS }
#else
# define __BEGIN_DECLS /* empty */
# define __END_DECLS /* empty */
#endif

__BEGIN_DECLS


/*
 * A marray is a struct with the rank of the marray (number of
 * indices it has), an array with the number of elements available for
 * each index, and an array to store the n1*n2*...*nm values.
 *
 * For the moment, there is no tda, as opossed to matrices, because it
 * would complicate quite a bit the algorithms and probably it is not
 * worth it.
 */
typedef struct
{
  unsigned int rank;
  size_t * dimension;
  size_t size;
  char * data;
} marray_char;


/*
 * There is not such a thing as "marray views", in contrast with the
 * case for gsl_matrix.
 */

/* Allocation */

marray_char *
marray_char_alloc(const unsigned int rank, const size_t * dimension);

marray_char *
marray_char_calloc(const unsigned int rank, const size_t * dimension);

marray_char *
marray_char_copy(marray_char * t);

void marray_char_free(marray_char * t);


/* Views */

/*
 * There are no views.
 */


/* Conversions */

gsl_matrix_char * marray_char_2matrix(marray_char * t);
gsl_vector_char * marray_char_2vector(marray_char * t);


/* Operations */

char marray_char_get(const marray_char * t, const size_t * indices);
void marray_char_set(marray_char * t, const size_t * indices, const char x);


char * marray_char_ptr(marray_char * t, const size_t * indices);
const char * marray_char_const_ptr(const marray_char * t, const size_t * indices);

void marray_char_set_zero(marray_char * t);
void marray_char_set_all(marray_char * t, char x);

int marray_char_fread(FILE * stream, marray_char * t);
int marray_char_fwrite(FILE * stream, const marray_char * t);
int marray_char_fscanf(FILE * stream, marray_char * t);
int marray_char_fprintf(FILE * stream, const marray_char * t, const char * format);

int marray_char_memcpy(marray_char * dest, const marray_char * src);
int marray_char_swap(marray_char * t1, marray_char * t2);

char marray_char_max(const marray_char * t);
char marray_char_min(const marray_char * t);
void marray_char_minmax(const marray_char * t, char * min_out, char * max_out);

void marray_char_max_index(const marray_char * t, size_t * indices);
void marray_char_min_index(const marray_char * t, size_t * indices);
void marray_char_minmax_index(const marray_char * t, size_t * imin, size_t * imax);

int marray_char_isnull(const marray_char * t);

int marray_char_add(marray_char * a, const marray_char * b);
int marray_char_sub(marray_char * a, const marray_char * b);
int marray_char_mul_elements(marray_char * a, const marray_char * b);
int marray_char_div_elements(marray_char * a, const marray_char * b);
int marray_char_scale(marray_char * a, const double x);
int marray_char_add_constant(marray_char * a, const double x);


/* inline functions if you are using GCC */

#ifdef HAVE_INLINE
extern inline
size_t
marray_char_position(const size_t * indices, const marray_char * t)
{
  size_t shift, position;
  unsigned int i;

  shift = t->size;
  position = 0;
  for (i = 0; i < t->rank; i++)
    {
#if GSL_RANGE_CHECK
      if (indices[i] >= t->dimension[i])
        return t->size;
#endif
      shift /= t->dimension[i];
      position += shift * indices[i];
    }

  return position;
}


extern inline 
char
marray_char_get(const marray_char * t, const size_t * indices)
{
  size_t position;

  position = marray_char_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VAL("index out of range", GSL_EINVAL, 0);
#endif

  return t->data[position];
}


extern inline
void
marray_char_set(marray_char * t, const size_t * indices, const char x)
{
  size_t position;
  
  position = marray_char_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VOID("index out of range", GSL_EINVAL);
#endif

  t->data[position] = x;
}


extern inline 
char *
marray_char_ptr(marray_char * t, const size_t * indices)
{
  size_t position;

  position = marray_char_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (char *) (t->data + position);
}


extern inline 
const char *
marray_char_const_ptr(const marray_char * t, const size_t * indices)
{
  size_t position;

  position = marray_char_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (const char *) (t->data + position);
} 

#endif

__END_DECLS

#endif /* __MARRAY_char_H__ */
