###############################################################################
# Copyright (C) 2006, 2007 Mariano Spadaccini	mariano@marianospadaccini.it  #
#									      #
# This file is part of MacAddressLocator.                                     #
#                                                                             #
#  MacAddressLocator is free software; you can redistribute it and/or modify  #
#  it under the terms of the GNU General Public License as published by       #
#  the Free Software Foundation; either version 2 of the License, or          #
#  (at your option) any later version.                                        #
#                                                                             #
#  MacAddressLocator is distributed in the hope that it will be useful,       #
#  but WITHOUT ANY WARRANTY; without even the implied warranty of             #
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              #
#  GNU General Public License for more details.                               #
#                                                                             #
#  You should have received a copy of the GNU General Public License          #
#  along with Foobar; if not, write to the Free Software                      #
#  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA #
###############################################################################

package myFunctionDB;

use strict;
use warnings;
use DBI;

use ParamDB qw(:config);
use Param qw(:config);

# This function return connection DB handle
sub connectToDB {
    #### Connect to database
    my $dbh =
      DBI->connect( "DBI:$sql_server:$sql_db:$sql_host", $sql_user, $sql_pwd )
      or die $DBI::errstr;
    return $dbh;
}

# Switch list from DB
sub listSwitch {
    my ($dbh) = @_;

    # Get all (id, ip, ...) from $DBlistSwitch
    my $query = "SELECT id, name, ip, disable FROM $DBlistSwitch";
    my $sth   = $dbh->prepare($query);
    $sth->execute();
    return $sth;
}

# Insert switch into DB (ip and SNMP version)
sub insertSwitch {
    my ( $dbh, $IP, $SNMPVersion ) = @_;

    # Get last id from DB
    my $query = "SELECT max(id) as max FROM $DBlistSwitch";
    my $sth   = $dbh->prepare($query);
    $sth->execute();
    my ($maxID) = $sth->fetchrow_array;

    # Eval max id
    if ( defined($maxID) ) {
        $maxID += 1;
    }
    else {
        $maxID = 1;
    }

    # Insert the switch
    $query =
"INSERT INTO $DBlistSwitch (id, ip, snmp_version, disable, inserted) VALUES ($maxID, ?, ?, 1, now())";
    $sth = $dbh->prepare($query);
    $sth->execute( $IP, $SNMPVersion );

    return $maxID;
}

# Gets switches parameters from DB (ip address, mac address and SNMP version)
sub getSwitchParametersFromDB {
    my ( $dbh, $id ) = @_;

    #### Create query
    my $query =
"SELECT id, ip, PhysAddress, snmp_version FROM $DBlistSwitch WHERE disable=0";
    if ( defined($id) ) {
        $query .= " AND id=?";
    }
    else {
        $query .= ' ORDER BY id';
    }

    my $sth = $dbh->prepare($query);

    if ( defined($id) ) {
        $sth->execute($id);
    }
    else {
        $sth->execute();
    }

    my ( @idSwitch, @IPSwitch, @MACSwitch, @SNMPversion );
    while ( my $hash_ref = $sth->fetchrow_hashref ) {
        push @idSwitch,    $hash_ref->{'id'};
        push @IPSwitch,    $hash_ref->{'ip'};
        push @MACSwitch,   $hash_ref->{'PhysAddress'};
        push @SNMPversion, $hash_ref->{'snmp_version'};
    }

    # return id, IP, mac and SNMP version of the switch
    return ( \@idSwitch, \@IPSwitch, \@MACSwitch, \@SNMPversion );
}

# Get switch id from ip
sub getSwitchIDFromIP {
    my ( $IP, $dbh ) = @_;

    my $query = "SELECT id FROM $DBlistSwitch WHERE ip=? AND disable=0";
    my $sth   = $dbh->prepare($query);
    $sth->execute($IP);
    my ($idSwitch) = $sth->fetchrow_array;

    return $idSwitch;
}

# Get switch name from DB
sub getSwitchOtherParamsFromID {
    my ( $id, $dbh ) = @_;
    #### Create query
    my $query =
"SELECT name, description, comment, snmp_version FROM $DBlistSwitch WHERE id=?";
    my $sth = $dbh->prepare($query);
    $sth->execute($id);
    my ( $name, $description, $comment, $snmp_version ) = $sth->fetchrow_array;

    return ( $name, $description, $comment, $snmp_version );
}

# Update switch parameters into DB
sub setSwitchParamsFromID {
    my ( $dbh, $id, $params, $query_where ) = @_;

    my @values;
    my $query = "UPDATE $DBlistSwitch SET ";
    while ( my ( $key, $value ) = each %$params ) {
        $query .= $key . "=?, ";
        push @values, $value;
    }
    $query = substr $query, 0, length($query) - 2;
    $query .= " WHERE " . $query_where;
    my $sth = $dbh->prepare($query);
    $sth->execute(@values);

    return 0;
}

# Insert/Update location in db
sub updateLocation {
    my ( $mac, $idSwitch, $port, $status, $idTimeShoot, $dbh ) = @_;

    my $idMAC = getMacIDFromDB( $mac, $dbh );

    # Get last ID (last entry code) for mac from the same switch in db
    my ($lastIDLoc) =
      myFunctionDB::getLastIDFromDB( $idMAC, $idSwitch, $port, $dbh );

    # If there isn't entry in db
    if ( not defined($lastIDLoc) ) {

        # Insert entry in db
        myFunctionDB::insertMACLocInDB( $idMAC, $idSwitch, $port, $status,
            $idTimeShoot, $dbh );
    }
    else {

        # Update time in db
        myFunctionDB::updateMACLocInDB( $lastIDLoc, $status, $idTimeShoot,
            $dbh );
    }
}

# Get last mac's idSwitch from db
sub getLastIDFromDB {
    my ( $idMAC, $idSwitch, $port, $dbh ) = @_;

    my $query =
"SELECT id FROM $DBlistLocation WHERE idMAC=? AND idSwitch=? AND port=? ORDER BY idTimeShoot DESC LIMIT 1";
    myFunctionUtils::traceLog( $query, 3 );

    #### query to database
    my $sth = $dbh->prepare($query);
    $sth->execute( $idMAC, $idSwitch, $port );

    return $sth->fetchrow_array;
}

# get mac's id from db
sub getMacIDFromDB {
    my ( $mac, $dbh ) = @_;

    my $query = "SELECT id FROM $DBlistMAC WHERE macaddress=?";
    myFunctionUtils::traceLog( $query, 3 );

    my $sth = $dbh->prepare($query);
    $sth->execute($mac);
    my ($idMAC) = $sth->fetchrow_array;

    if ( not defined $idMAC ) {
        myFunctionDB::insertMACinDB( $mac, $dbh );

        # Re-Fetch mac in db
        myFunctionUtils::traceLog( $query, 3 );
        $sth = $dbh->prepare($query);
        $sth->execute($mac);
        ($idMAC) = $sth->fetchrow_array;
    }
    return $idMAC;
}

# Insert macaddress in db
sub insertMACinDB {
    my ( $mac, $dbh ) = @_;

    my $timeNow = myFunctionUtils::timeNow();

    my $query = "INSERT INTO $DBlistMAC (macaddress, inserted) VALUES (?, ?)";
    myFunctionUtils::traceLog( $query, 3 );

    my $sth = $dbh->prepare($query);
    $sth->execute( $mac, $timeNow );
}

# Insert mac localization in db
sub insertMACLocInDB {
    my ( $idMAC, $idSwitch, $port, $status, $idTimeShoot, $dbh ) = @_;

    # Evaluate nowTime
    my $nowTime = myFunctionUtils::timeNow();

    my $query =
"INSERT INTO $DBlistLocation (idMAC, idSwitch, port, status, time, idTimeShoot) VALUES (?,?,?,?,?,?)";
    myFunctionUtils::traceLog( $query, 3 );

    my $sth = $dbh->prepare($query);
    $sth->execute( $idMAC, $idSwitch, $port, $status, $nowTime, $idTimeShoot );
}

# Update last mac localization in db
sub updateMACLocInDB {
    my ( $lastIDLoc, $status, $idTimeShoot, $dbh ) = @_;

    # Evaluate nowTime
    my $nowTime = myFunctionUtils::timeNow();

    # Create query for update
    my $query = "UPDATE $DBlistLocation SET ";
    if ( defined($status) ) {
        $query .= "status=?, ";
    }
    $query .= "time=?, idTimeShoot=? WHERE id=?";
    myFunctionUtils::traceLog( $query, 3 );

    # query to database
    my $sth = $dbh->prepare($query);
    if ( defined($status) ) {
        $sth->execute( $status, $nowTime, $idTimeShoot, $lastIDLoc );
    }
    else {
        $sth->execute( $nowTime, $idTimeShoot, $lastIDLoc );
    }
}

# Insert timeShoot in db
sub insertTimeShoot {
    my ( $timeShoot, $dbh ) = @_;

    # Create query for insert
    my $query = "INSERT INTO $DBmalTimeShoot (timeShoot) VALUES ( ? )";
    myFunctionUtils::traceLog( $query, 3 );

    # execute query
    my $sth = $dbh->prepare($query);
    $sth->execute($timeShoot);
}

# get idTimeShoot from db
sub getIDTimeShoot {
    my ( $timeShoot, $dbh ) = @_;

    # Create query for insert
    my $query = "SELECT id FROM $DBmalTimeShoot WHERE timeShoot=?";
    myFunctionUtils::traceLog( $query, 3 );

    my $sth = $dbh->prepare($query);
    $sth->execute($timeShoot);

    my ($idTimeShoot) = $sth->fetchrow_array;
    return $idTimeShoot;
}

# Count how many entry for raw mac (family mac?)
sub countEntryForMAC {
    my ( $cleanedUpMAC, $dbh ) = @_;
    my $query = "SELECT count(*) FROM $DBlistMAC WHERE macaddress LIKE ?";
    myFunctionUtils::traceLog( $query, 5 );

    my $sth = $dbh->prepare($query);
    $sth->execute("%$cleanedUpMAC%");

    my ($countEntryForMAC) = $sth->fetchrow_array;
    return $countEntryForMAC;
}

# Count how many entry for switch's port
sub countEntryForSwitchPort {
    my ( $IP, $Port, $dbh ) = @_;
    my $query =
"SELECT count(*) FROM $DBlistLocation AS l, $DBlistSwitch AS s WHERE l.idSwitch = s.id AND s.ip=? AND l.port=?";
    myFunctionUtils::traceLog( $query, 5 );

    my $sth = $dbh->prepare($query);
    $sth->execute( $IP, $Port );

    my ($countEntryForMAC) = $sth->fetchrow_array;
    return $countEntryForMAC;
}

# Get older entry (get older inserted datetime in db) for raw mac
sub getOlderTimeEntryForMAC {
    my ( $cleanedUpMAC, $dbh ) = @_;
    my $query =
"SELECT inserted FROM $DBlistMAC WHERE macaddress LIKE ? ORDER BY inserted ASC LIMIT 1";
    myFunctionUtils::traceLog( $query, 5 );

    my $sth = $dbh->prepare($query);
    $sth->execute("%$cleanedUpMAC%");

    my ($OlderTimeEntryForMAC) = $sth->fetchrow_array;
    return myFunctionUtils::transformDatetimeInHumanDate($OlderTimeEntryForMAC);
}

# Get older/last entry (get older inserted datetime in db) for switch's IP and Port
sub getTimeEntryForSwitchPort {
    my ( $IP, $Port, $dbh, $type ) = @_;
    my $query =
"SELECT m.time FROM $DBlistLocation AS m, $DBlistSwitch AS s WHERE m.idSwitch = s.id AND s.ip=? AND m.port=? ORDER BY m.time";
    if ( $type eq 'last' ) {
        $query .= ' DESC';
    }
    else {
        $query .= ' ASC';
    }
    $query .= ' LIMIT 1';
    myFunctionUtils::traceLog( $query, 5 );

    my $sth = $dbh->prepare($query);
    $sth->execute( $IP, $Port );

    my ($TimeEntryForSwitchPort) = $sth->fetchrow_array;
    return myFunctionUtils::transformDatetimeInHumanDate(
        $TimeEntryForSwitchPort);
}

# Get last entry (get last inserted datetime in db) for raw mac
sub getLastTimeEntryForMAC {
    my ( $cleanedUpMAC, $dbh ) = @_;
    my $query =
"SELECT loc.time FROM $DBlistLocation AS loc, $DBlistMAC AS mac WHERE mac.macaddress LIKE ? and mac.id=loc.idMAC ORDER BY loc.time DESC LIMIT 1";
    myFunctionUtils::traceLog( $query, 5 );

    my $sth = $dbh->prepare($query);
    $sth->execute("%$cleanedUpMAC%");

    my ($LastTimeEntryForMAC) = $sth->fetchrow_array;
    return myFunctionUtils::transformDatetimeInHumanDate($LastTimeEntryForMAC);
}

# get current MAC from raw MAC (remember: raw MAC can be a fragment mac!!)
sub getMACFromRawMAC {
    my ( $cleanedUpMAC, $dbh ) = @_;
    my $query = "SELECT macaddress FROM $DBlistMAC WHERE macaddress LIKE ?";
    myFunctionUtils::traceLog( $query, 5 );

    my $sth = $dbh->prepare($query);
    $sth->execute("%$cleanedUpMAC%");

    my ($correctMAC) = $sth->fetchrow_array;
    return $correctMAC;
}

# get story of mac address
sub getMACstory {
    my ( $cleanedUpMAC, $dbh ) = @_;
    my $query =
"SELECT sw.ip AS ip, sw.description AS description, loc.port AS port, loc.time AS time FROM $DBlistLocation AS loc, $DBlistMAC AS mac, $DBlistSwitch AS sw WHERE mac.macaddress LIKE ? AND mac.id=loc.idMAC AND loc.idSwitch=sw.id ORDER BY loc.time DESC";
    myFunctionUtils::traceLog( $query, 5 );

    my $sth = $dbh->prepare($query);
    $sth->execute("%$cleanedUpMAC%");

    return $sth;
}

# Get list of macaddress from switch's port
sub getListMACFromSwitchPort {
    my ( $IP, $Port, $dbh ) = @_;
    my $query =
"SELECT m.macaddress AS mac, l.time AS time FROM $DBlistLocation AS l, $DBlistSwitch AS s, $DBlistMAC AS m WHERE l.idMAC=m.id AND l.idSwitch = s.id AND s.ip=? AND l.port=? ORDER BY l.time DESC";
    myFunctionUtils::traceLog( $query, 5 );

    my $sth = $dbh->prepare($query);
    $sth->execute( $IP, $Port );

    return $sth;
}

1;
