# -*- coding: utf-8 -*-
# Author:   $Author: merkosh $
# Revision: $Rev: 128 $
############################################################################
#    Copyright (C) 2005 by Uwe Mayer                                       #
#    merkosh@hadiko.de                                                     #
#                                                                          #
#    This program is free software; you can redistribute it and/or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             #
############################################################################

#-- imports --------------------------------------------------------------------
#-- Qt imports
from qt import SIGNAL, QListView, QListViewItem, QGridLayout, QSize, QTimer
from qt import PYSIGNAL, QImage, QPixmap, QApplication


#-- Python imports
import logging
import sys

from os.path import exists as fexists


#-- own imports
from RecordListBase import RecordListBase
from misc import correctSlice
from AMCForm import AMCForm
from PicturePreview import PicturePreview
from Settings import getPreferences
from ListViewRecord import ListViewRecord

# TODO: use C++ compiled version
#from Qt.QListViewItemNumeric.QListViewItemNumeric import QListViewItemNumeric


#-- preferences ----------------------------------------------------------------
pref = getPreferences(__name__)



#-- About ListViews and RecordList --
#
# ListViews are strange objects. Originally RecordList should have
# supported the list-interface. However, there is no real order on
# ListView objects. With auto-sorting turned on a ListView is more
# an iterable, set-like object. Element access occurs by object
# reference.
#
class RecordList(RecordListBase):
    """initialises the central ListView

    RecordList displays a list of records in a graphical interface.
    """
    def __init__(self, dirname, parent=None, name=None, fl=0):
        RecordListBase.__init__(self, parent, name, fl)

        # create logging
        self.log = logging.getLogger( self.__class__.__name__ )
        self.log.setLevel(pref['logLevel'])

        # embedd form on right hand side
        self.form = AMCForm( self.frameRecord )
        self.form.hide()              # show only if records are available
        formLayout = QGridLayout( self.frameRecord, 1,1 )
        formLayout.addWidget( self.form, 0,0 )

        # list view
        self.listViewRecord = ListViewRecord(self.frameListView)
        listViewLayout = QGridLayout(self.frameListView, 1,1)
        listViewLayout.addWidget( self.listViewRecord, 0,0 )

        # image on left bottom
        self.image = PicturePreview( dirname, self.frameImage )
        imageLayout = QGridLayout( self.frameImage, 1,1 )
        imageLayout.addWidget( self.image, 0,0 )

        # grouping
        self.currentGroup = None

        # connect signals
        self.connect( self.listViewRecord, SIGNAL("currentChanged(QListViewItem*)"), self.onCurrentChanged )
        self.connect( self.listViewRecord, PYSIGNAL("searchTextChanged"), PYSIGNAL("searchTextChanged"))
        self.connect( self.form, PYSIGNAL("modified()"), self.listViewRecord.onModified )    
        # currently not needed: 
        # self.connect( self.image, PYSIGNAL("modified()"), self.onModified )

    #-- list interface ---------------------------------------------------------
    def add(self, rec):
        """W.add(rec) -- append record(s)"""
        self.listViewRecord.add(rec)               
        # this also shows the main form when the first record becomes
        # available
        if (len(self) != 0): self.form.show()


    def values(self):
        """returns an unsorted list of all records"""
        return self.listViewRecord.values()


    def __delitem__(self, obj):
        """removes ListViewItem(s) by reference"""
        del self.listViewRecord[obj]
        if (len(self) == 0): self.form.hide()


    def __len__(self):
        """returns the number of elements in the list view"""
        return len(self.listViewRecord)


    def getCurrentRecord(self):
        # here it is important that all instances: listViewRecord,
        # form, image work on the same object;
        return self.listViewRecord.getCurrentRecord()


    def setCurrentRecord(self, record):
        self.listViewRecord.setCurrentRecord(record)
        self.form.setRecord(record)
        self.image.setRecord(record)


    def findNext(self, text, field, options):
        self.listViewRecord.findNext(text, field, options)


    #-- slots ------------------------------------------------------------------
    def onGroup(self, group):
        """called when an online re-grouping is necessary"""
        if (group != self.currentGroup):
            self.currentGroup = group
            self.disconnect(self.listViewRecord, SIGNAL("currentChanged(QListViewItem*)"), self.onCurrentChanged)
            self.listViewRecord.onGroup(group)
            self.connect(self.listViewRecord, SIGNAL("currentChanged(QListViewItem*)"), self.onCurrentChanged)
            self.focus(self.listViewRecord.currentItem())
        

    def onCurrentChanged(self, obj):
        """called when the \"current\" item changes"""
        if (obj != None):
            record = self.listViewRecord.getCurrentRecord()
            if (record == None):
                if (not self.form.isHidden()): self.form.hide()
                if (not self.image.isHidden()): self.image.hide()
            else:
                if (self.form.isHidden()): self.form.show()
                self.form.setRecord( record )
                if (self.image.isHidden()): self.image.show()
                self.image.setRecord( record )
        else:
            self.form.setRecord(None)
            self.image.setRecord(None)


    #-- utility functions ------------------------------------------------------
    def focusFirst(self):
        """displays the first item in the ListView"""
        self.listViewRecord.focusFirst()


    def focusNew(self):
        """displays the last item added"""
        self.listViewRecord.focusNew()


    def focus(self, obj):
        """displays the selected object"""
        self.listViewRecord.focus(obj)
    

#-- testroutine ----------------------------------------------------------------
if (__name__ == "__main__"):
    from qt import QApplication, QObject, SLOT

    a = QApplication(sys.argv)
    QObject.connect(a,SIGNAL("lastWindowClosed()"),a,SLOT("quit()"))

    # need logging handlers for testing
    log = logging.getLogger("")
    hdlr = logging.StreamHandler()
    formatter = logging.Formatter('[%(name)s] %(levelname)s: %(message)s')
    hdlr.setFormatter(formatter)
    log.addHandler(hdlr) 
    log.setLevel(pref['logLevel'])

    w = RecordList(".")

    from AMCFile33 import AMCFile33
    f = AMCFile33("Filme.amc", "rb")
    
    rec = f.read()
#    w.add( rec )
#    w.focusFirst()

    a.setMainWidget(w)
    w.show()

    a.exec_loop()

#-- EOF --
