# -*- coding: utf-8 -*-
# Author:   $Author: merkosh $
# Revision: $Rev: 105 $
############################################################################
#    Copyright (C) 2005 by Uwe Mayer                                       #
#    merkosh@hadiko.de                                                     #
#                                                                          #
#    This program is free software; you can redistribute it and/or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             #
############################################################################
"""Provides classes which scan multimedia files for file properties"""
#-- python imports --
import os
import re
import logging

from os.path import exists as fexists, getsize as fsize, basename


#-- own imports
from Settings import getPreferences
from Language import TR
from MediaInfoDialog import MediaInfoDialog
from StdIOProcess import StdIOProcess


#-- constants ------------------------------------------------------------------
AUTO  = 0
KB    = 1
MB    = 2
GB    = 3


#-- preferences ----------------------------------------------------------------
pref = getPreferences(__name__)


audioCodecMap = {
    '0x50': 'MPEG Layer-2',
    '0x55': 'MPEG Layer-3',
    '0xfffe': 'Ogg Vorbis',
    '0x2000': 'AC3',
    }


# create proxy format for script environment
scriptProxy = None
if (pref['proxy'] != None):
    scriptProxy = {}
    for (k,v) in pref['proxy'].iteritems():
        scriptProxy["%s_proxy"%k] = v

    

class MediaInfo(object):
    """root class providing access to fields"""
    def __init__(self, filename):
        """provides interface to access a file's properties"""
        self.log = logging.getLogger("MediaInfo")
        self.log.setLevel(pref['logLevel'])

        if (not fexists(filename)):
            raise IOError(TR("file not found ")+filename)

        self.filename = filename
        self.originalTitle = filename   # original title
        self.videoFormat = ""           # video codec
        self.audioFormat = ""           # audio codec
        self.resolution = ""            # video resolution
        self.languages = ""             # list of languages (as strings)
        self.subtitles = ""             # list of subtitles (as strings)
        self.videoBitrate = 0           # video bitrate (in KBPS)
        self.audioBitrate = 0           # audio bitrate (in KBPS)
        self.framerate = ""             # video framerate (in fps)
        self.fileSize = fsize(filename) # file size (in bytes)
        self.length = 0                 # length (in min)

        # do a set of simplifications on the filename to retrieve the original title
        self.originalTitle = basename(self.originalTitle)    
        self.originalTitle = re.sub('(?i)xvid.*|dvdrip.*|divx.*|ripped.*|cd ?\d.*|[ms]vcd.*|mpe?g.*|\.avi|\.ogm', '', self.originalTitle)
        self.originalTitle = re.sub('(?i)german\s*|english\s*$', '', self.originalTitle)
        self.originalTitle = re.sub('[\(\[].*', '', self.originalTitle)
        self.originalTitle = re.sub('[\._]', ' ', self.originalTitle)
        self.originalTitle = re.sub(' {2,}', ' ', self.originalTitle)
        self.originalTitle.strip()


    def scan(self):
        """starts the scanning process"""
        pass


    def normalizeFileSize(cls, size, format=AUTO):
        """converts a filesize in bytes to human readable format"""
        if (format == AUTO):
            sizes = ["Bytes", "kB", "MB", "GB"]
            index = 0
            while ((size >= 1024) and (index < len(sizes))):
                size /= 1024.0
                index += 1
            return "%.2f %s"%(size, sizes[index])
        else:
            return "%.2f"%(size /(1024**format))
    normalizeFileSize = classmethod(normalizeFileSize)
        

    def getFileSize(self, format=AUTO):
        """returns file size, formatted as bytes, kB, MB or GB"""
        return self.normalizeFileSize(self.fileSize, format)


    def regexSearch(self, expr, string, errorMsg):
        """searches self.text for an expression in a string"""
        m = re.search(expr, string, re.DOTALL)
        if (m == None):
            if (errorMsg != None):
                self.log.warning(errorMsg)
        else:
            return m.groups()


    def __str__(self):
        return """\
filename:         %s
  original title: %s
  video format:   %s
  audio format:   %s
  resolution:     %s 
  languages:      %s
  subtitles:      %s
  video bitrate:  %d kbps
  audio bitrate:  %d kbps
  framerate:      %s fps
  file size:      %s MB
  length:         %d min"""%\
    (self.filename, self.originalTitle, self.videoFormat, self.audioFormat,
     self.resolution, self.languages, self.subtitles, self.videoBitrate,
     self.audioBitrate, self.framerate, self.getFileSize(), self.length)
                                     



class Tcprobe(MediaInfo):
    """get file properties by calling tcprobe utility

    The following information is gathered:
    - video format:  only 4-letter fourcc can be retrieved
    - audio format:  tcprobe only ouputs hex code; which is looked
                     up in an internal dictionary
    - resolution:    detected
    - video bitrate: average calculated from file size, audio
                     bitrate and play time
    - audio bitrate: usually detected
    - framerate:     detected
    - file size:     detected
    - length:        usually detected
    """
    def __init__(self, filename):
        MediaInfo.__init__(self, filename)
        self.log = logging.getLogger("MediaInfo.Tcprobe")
        self.log.setLevel(pref['logLevel'])

        self.process = StdIOProcess("tcprobe", ['-i', self.filename.encode(pref['systemEncoding'])],
                                    scriptProxy)
        


    def scan(self):
        """starts scanning process"""
        # show dialog and exec tcprobe
        dlg = MediaInfoDialog(self.process, fl=MediaInfoDialog.WStyle_NoBorder|MediaInfoDialog.WX11BypassWM)
        if (dlg.exec_loop() == dlg.Rejected):
            self.log.info("Scanning aborted")
            return

        # get text
        text = self.process.stdout()

        # start processing
        tmp = self.regexSearch('V: .+ codec=(.+?), ', text, "no match for video codec")
        if (tmp): self.videoFormat = tmp[0]

        tmp = self.regexSearch('-n (.+?) \[', text, "no match for audio codec")
        if (tmp): self.audioFormat = tmp[0]
        if (audioCodecMap.has_key(self.audioFormat)):
            self.audioFormat = audioCodecMap[self.audioFormat]

        tmp = self.regexSearch('VBR', text, None)
        if ((tmp != None) and (self.audioFormat != "")):
            self.audioFormat += " VBR"

        tmp = self.regexSearch('-e \d+,\d+,\d+ \[(\d+),(\d+),(\d+)\] ', text, None)
        if (tmp and (len(tmp) == 3) and (self.audioFormat != "")):
            self.audioFormat += " (freq=%s, bits=%s, ch=%s)"%tmp
            
        tmp = self.regexSearch('-g (\d+x\d+)', text, "no match for resolution")
        if (tmp): self.resolution = tmp[0]

        tmp = self.regexSearch('audio track: .*bitrate=(\d+) kbps', text, "no match for audio bitrate")
        if (tmp): self.audioBitrate = int(tmp[0])

        tmp = self.regexSearch('-f ([\d\.]+) ', text, "no match for framerate")
        if (tmp): self.framerate = tmp[0]

        tmp = self.regexSearch(' duration=(\d+):(\d+):', text, "no match for length")
        if (tmp and (len(tmp) == 2)):
            self.length = int(tmp[0])*60 +int(tmp[1])
        else:
            tmp = self.regexSearch('V\: \d+ frames\, (\d+) sec', text, None)
            if (tmp): self.length = int(tmp[0])/60.0
            
        # calculate avg video bitrate
        if ((self.length != 0) and (self.audioBitrate != 0)):
            size = self.fileSize        # size in bytes
            # minus bytes used for audio
            size -= (self.length*60)*(1024*self.audioBitrate /8.0)

            self.videoBitrate = (8*size/1024.0)/(self.length*60)




#-- test area ------------------------------------------------------------------
if (__name__ == "__main__"):
    import sys
    from qt import QApplication, QObject, SIGNAL, SLOT

    a = QApplication(sys.argv)
    QObject.connect(a,SIGNAL("lastWindowClosed()"),a,SLOT("quit()"))

    filename1 = u"/home/merkosh/Movies/Man.On.Fire.2004.DVDRip.XviD-DvP.CD2.avi"
    filename2 = u"/home/merkosh/Movies/Last Action Hero German XVID DVDrip.avi"
    filename3 = u"/home/merkosh/Movies/12:01 (g0.5, de, 1o1).ogm"
    filename4 = u"/home/merkosh/Movies/topsecret_mvcd.mpg"
    filename5 = u"/home/merkosh/Movies/Team.America.World.Police.(2004).iNTERNAL.TS.XviD-WESTSIDE.ShareTheFiles.com.avi"
    filename6 = u"/home/merkosh/Movies/13 Going On 30  [en].avi"
    filename7 = u"/dev/hdb"
    filename8 = u"/mnt/hdb/Black Beauty (en,de 2o2 g0.4).ogm"

#    QMessageBox.information(None, "scanning file...", "Scanning file...")
    
    query = Tcprobe(filename4)
    query.scan()
    print query

