# -*- coding: utf-8 -*-
# Author:   $Author: merkosh $
# Revision: $Rev: 100 $
############################################################################
#    Copyright (C) 2005 by Uwe Mayer                                       #
#    merkosh@hadiko.de                                                     #
#                                                                          #
#    This program is free software; you can redistribute it and/or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             #
############################################################################

#-- imports --------------------------------------------------------------------
import logging
import os

from copy import deepcopy
from os.path import join as fjoin, exists as fexists
from pprint import pformat



#-- defaults -------------------------------------------------------------------
CONFIG_TEXT = """\
# -*- coding: utf-8 -*-
# Author:   $Author: merkosh $
# Revision: $Rev: 100 $
############################################################################
#    Copyright (C) 2005 by Uwe Mayer                                       #
#    merkosh@hadiko.de                                                     #
#                                                                          #
#    This program is free software; you can redistribute it and/or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             #
############################################################################

# This is an automatically generated text by Settings.py, which is part of
# lmc, the LMC Movie Catalogue program. This file contains the per user
# preferences and is executed as part of lmc during the initialisation phase
# by the python interpreter.
#
# If you have problems with this configuration file at any time you can
# restore a clean copy of the global defaults by executing Settings.py
# in the lmc src folder:
#
# $ python Settings.py
# [root] INFO: Resoring default configuration
# [_Settings] INFO: Saving current configuration
#
# You can test the configuration file by executing it in python:
#
# $ python ~/.lmc/lmc.conf
# $
#
# If no error occurs the configuration file was executed without
# problems.


#-- python imports needed for default settings
%(import)s


#-- default definitions

# logLevel is a dictionary, mapping class- and module names to
# a constant which defines the minimum type of log-message that
# will pass to log-output. Valid values in ascending order are:
# DEBUG      10
# INFO       20
# WARNING    30       
# ERROR      40
# FATAL      50
#
# These values are defined in the Python logging module included
# above. Setting a value to \"logging.WARN\" would cause all
# DEBUG and INFO messages to be swallowed and not to apprear on
# the log output.
#
# Output to the default logger should not appear and if you find
# output anyways, please submit a bug-report.

logLevel = %(logLevel)s


# defaultPath defines the path which lmc uses as its first entry
# point in the file open dialog
# The default is the current working directory.

defaultPath = %(defaultPath)s


# scriptPath defines the location of the auxilary scripts which
# lmc uses to gather information on movie titles from the internet.
# Default value is the \"scripts\" subdirectory, relative to lmc's
# source directory.

scriptPath = %(scriptPath)s


# browserPath is the path to the executable of your prefered internet
# browser. This is the program executed to load external HTML pages.
# The default is '/usr/bin/mozilla'.

browserPath = %(browserPath)s


# The values in defaultHistory are inserted in the comboboxes of the
# record view form. The structure is a dictionary with the keys being the
# original field names and the values being a list of strings to insert.

defaultHistory = %(defaultHistory)s


# If addHistoryFromRecords is set to True new values from comboboxes
# encountered when browsing the record list are inserted into the combo-
# box history.
# Default: True

addHistoryFromRecords = %(addHistoryFromRecords)s


# recordDefault specifies the default values used to initialise a new
# record field. The structure is again a dictionary, the keys being
# the original field names and the values being stings or numbers
# to be used as default values.

recordDefault = %(recordDefault)s


# proxy defines your proxy settings, in case you need one to access
# web pages on the internet.
#
# If proxy is set to 'None', either no proxy is used, or, if defined,
# your environment variables http_proxy, https_proxy, etc are used.
# The format for these environment variables is i.e.:
# http_proxy=http://proxy.somehost.de:3128
#
# You can set your proxy servers here explicitly, by assigning proxy a
# dictionary which keys define the type of proxy and setting the values
# to the URL of your proxy server, i.e.:
# proxy = {'http': 'http://proxy.somehost.de:3128'}
#
# None: this does not influence the proxy settings of the auxilary scripts.
#       The lmc default scripts currently still rely on your http_proxy
#       environment variable.
# 
# Default: use environment variable

proxy = %(proxy)s


# systemEncoding defines the encoding used by lmc to communicate with
# everything outside of lmc, i.e. when saving files to disc or passing
# command line arguments to auxilary scripts.
#
# Valid values include:
# latin1 (=iso-8859-1), iso-8859-15, utf8, utf16, ascii, and many more.
#
# For a complete list see Python library reference manual on
# \"standart-encodings\".
# 
# Default: latin1

systemEncoding = %(systemEncoding)s


# recordStoreOrderRandom is a boolean value which specifies wether the
# movie records should be stored in ascending sequence or if the order
# may remain random.
# Default: False   (= sort records on saving)

recordStoreOrderRandom = %(recordStoreOrderRandom)s


# Pictures may be saved embedded or linked to files on the filesystem.
# Setting deepcopyOnCopy to True will cause copy&paste to create a
# duplicate picture in the filesystem. The value False will just copy
# the link and reference the picture of the source record.
# Default: True

deepcopyOnCopy = %(deepcopyOnCopy)s

# mediaInfo is the class name of the library to use when scanning 
# a movie for its properties, such as video bitrate, length, size, etc.
# To set this value correctly you need to import the module, as in
# MediaInfo at the top.
# Currently there is only one default, using transcode's tcprobe command.
# Default: Tcprobe

mediaInfo = %(mediaInfo)s


# autoMountPoint, if not set to None, specifies the mount-point of a
# removeable device which will be automatically mounted and unmounted
# using the system mount command when a specific action from LMC menu
# bar is invoked.
# This option supposedly simplifies scanning movie files from removeable
# media which usually requires tedious mounting and unmounting before
# the medium can be removed.
# Default: None

autoMountPoint = %(autoMountPoint)s


# ejectOnUnmount is a modification on the autoMountPoint configuration
# which causes the removeable media to be ejected when unmounting.
# This option has no effect when autoMountPoint is set to None.
# Default: True

ejectOnUnmount = %(ejectOnUnmount)s


# autoScanAutoMountOnNew is a boolean value which decides on wether
# information is to be gathered on files from the autoMountPoint above when
# a new record is added to lmc. For this option to work the autoMountPoint
# must be set to non-None.
# Default: True

autoScanAutoMountOnNew = %(autoScanAutoMountOnNew)s


# audoScanOnNew is a boolean value which decides on wether information
# is to be gathered on files from disc when a new record is added to lmc.
# Default: False

autoScanOnNew = %(autoScanOnNew)s


# autoScriptOnNew is a boolean value which decides on wether the get-
# information-from-script process should be started after a new record
# has been added to lmc.
# This process happens after auto-scanning a file, if autoScan*OnNew is
# set to True.
# Default: True

autoScriptOnNew = %(autoScriptOnNew)s


# embeddedImage defines wether lmc will store image data internally
# or wether to save it as file to under a given path on the filesystem.
# The default is to save it externally.

embeddedImage = %(embeddedImage)s


# imageFormat defines the format which is used to save both internal and
# external images to. Supported output formats include:
# JPEG, BMP, PBM, PGM, PNG, PPM, XBM, XPM
#
# The default is: JPEG

imageFormat = %(imageFormat)s


# When images are saved externally, the variable filenamePrefix defines
# the path and prefix used when storing images. Path and filename may be
# set to any valid values, but a '%%d' format specifier must be contained
# somewhere in the filename. This is replaced with the record ID as the
# picture is saved.
# If the path does not exist, it will be created recursively. You can
# specify an absolute path here.
# The default value will put the files in the directory \"pictures\"
# from the current working directory and let the record ID be zero-prefixed
# to four decimal places.
# The file extention is added automatically according to the file format
# specified above.

filenamePrefix = %(filenamePrefix)s


# sortColumn defines the column number in the main list view, according
# to which the records are sorted ascendingly. The columns are indexed
# starting from zero.
# Default: 0

sortColumn = %(sortColumn)s


# zeroPrefix_patch defines wether the IDs first column in the main list
# view are zero-prefixed to (currently 4) digits.
# This is a patch for a deficiency in Qt /PyQt: The columns are sorted
# alphanumerically by Qt. This is not settable by an option. Instead you
# can subclass and overwrite the sort method. However, implementing the
# comparison function in Python introduces a nagging delay each time the
# display is sorted, which was unacceptable.
# If you'd rather remove the leading zeros, set this to False.
# Default: True

zeroPrefix_patch = %(zeroPrefix_patch)s


# scripts lists all available scripts which can be used to download
# information on a movie from the internet.
# scripts itself is a dictionary, the keys denoting a unique text
# which identifies the entry. Its value is a 2-tuple:
# The first field in the 2-tuple is a text string which is displayed
# as a description in the script selection dialog.
# The second field is a list of arbitrary length. Each list element
# is a n-tuple, the first element being the executable name of the script
# to execute, relative to the scripts subdirectory defined above.
# The rest of the fields are field values which should be retrieved from
# the scripts result. This allows you to specifically remove a field
# from a scripts result (i.e. if you find its result should not ever be
# included).
# The list of scripts is processed in-order they appear. Some field values
# are accumulative, some are overwriting. Accumulative fields, such as
# \"desciption\" and \"comments\" are appended after eachother, separated
# by empty lines. Overwriting fields replace a previously specified value
# by a new one. This is typically the case for fields which do cannot
# be displayed by the main record form as alternatives, i.e. rating,
# picture, etc.
# In combination, the order of the script-tuples allows you to specify
# preferences, i.e. in the example 'Uwe Special' below: pictures from
# filmposterarchiv-de.pl script are prefered. However, if the script
# does not return a result the picture from amazon-en.pl is used.
# Last but not least the failsafe IMDB picture is used.

scripts = %(scripts)s


# defaultScript holds the key to the script which should be selected by default
# Default is: 'Uwe Special'

defaultScript = %(defaultScript)s


#-- EOF --
"""

#-- first config format --------------------------------------------------------
class _Settings0(dict):
    """defines first element in configuration chain"""
    default = {
        'logLevel': {'<none>'                : logging.DEBUG,
                     'default'               : logging.DEBUG,
                     '__main__'              : logging.DEBUG,
                     'AbstractFile'          : logging.DEBUG,
                     'AMCFile33'             : logging.DEBUG,
                     'AMCForm'               : logging.INFO,
                     'DragObject'            : logging.DEBUG,
                     'Interface'             : logging.DEBUG,
                     'IODeviceBuffer'        : logging.DEBUG,
                     'Language'              : logging.DEBUG,
                     'MediaInfo'             : logging.DEBUG,
                     'MediaInfoDialog'       : logging.DEBUG,
                     'misc'                  : logging.DEBUG,
                     'PictureDialog'         : logging.DEBUG,
                     'PicturePreview'        : logging.DEBUG,
                     'RecordList'            : logging.DEBUG,
                     'ListViewRecord'        : logging.DEBUG,
                     'RecordPreview'         : logging.DEBUG,
                     'ScriptExecDialog'      : logging.DEBUG,
                     'StdIOProcess'          : logging.DEBUG,
                     'ScriptSelection'       : logging.DEBUG,
                     'Settings'              : logging.DEBUG,
                     'TitleMatchList'        : logging.DEBUG,
                     },

        'defaultPath': os.environ['PWD'],
        'scriptPath': 'scripts/',
        'browserPath': '/usr/bin/mozilla',

        'defaultHistory': {'subtitles'       : ('English', 'German'),
                           'languages'       : ('English', 'German', 'English, German'),
                           'framerate'       : (),
                           'videoFormat'     : (),
                           'audioFormat'     : (),
                           'category'        : (),
                           'country'         : (),
                           'mediaType'       : ('CD-Rom', 'DVD'),
                           'source'          : (),
                           'borrower'        : ()                          
                           },
        'addHistoryFromRecords': True,
        'recordDefault': {'mediaLabel'      : u"",
                          'originalTitle'   : u"",
                          'translatedTitle' : u"",
                          'director'        : u"",
                          'producer'        : u"",
                          'url'             : u"",
                          'resolution'      : u"",
                          'fileSize'        : u"",
                          'actors'          : u"",
                          'description'     : u"",
                          'comments'        : u"",
                          'subtitles'       : u"",
                          'languages'       : u"",
                          'framerate'       : u"",
                          'videoFormat'     : u"",
                          'audioFormat'     : u"",
                          'category'        : u"",
                          'country'         : u"",
                          'mediaType'       : u"CD-Rom",
                          'source'          : u"",
                          'borrower'        : u"",
                          'discs'           : 0,
                          'videoBitrate'    : 0,
                          'audioBitrate'    : 0,
                          'length'          : 0,
                          'rating'          : 0,
                          'year'            : 1900,
                          'date'            : 0,
                          },
        
        'proxy': None,
        
        'systemEncoding': 'latin1',
        
        'recordStoreOrderRandom': False,

        'deepcopyOnCopy': True,
        
        'mediaInfo': None,
        'autoMountPoint': None,
        'ejectOnUnmount': True,
        'autoScanAutoMountOnNew': True,
        'autoScanOnNew': False,
        'autoScriptOnNew': False,
        
        'embeddedImage': False,
        'imageFormat': 'JPEG',
        'filenamePrefix': u"pictures/movie_%04d",
        
        'sortColumn': 0,
        'zeroPrefix_patch': True,
        
        'scripts':{'[Merkosh Special]':
                   (u"all available fields from IMDB (US)\n+ medium sized picture from Amazon\n+ big picture from filmposter-archiv.de",
                    [('IMDB-en.pl', 'originalTitle', 'year', 'picture', 'category', 'actors', 'rating', 'url', 'length', 'country', 'languages', 'description', 'comments', 'translatedTitle', 'producer', 'director'),
                     ('amazon-en.pl', 'picture'),
                     ('filmposterarchiv-de.pl', 'picture'), ]),

                   '[picture]':
                    (u"any picture, preferably large",
                    [('IMDB-en.pl', 'picture'),
                     ('amazon-en.pl', 'picture'),
                     ('filmposterarchiv-de.pl', 'picture'), ]),
                   
                   'IMDB':
                   (u"download all available fields from IMDB (US)",
                    [('IMDB-en.pl', 'originalTitle', 'year', 'picture', 'category', 'actors', 'rating', 'url', 'length', 'country', 'languages', 'description', 'comments', 'translatedTitle', 'producer', 'director'),]
                    ),
                   
                   'Amazon.com':
                   (u"download medium sized picture from Amazon.com",
                    [('amazon-en.pl', 'picture')]
                    ),
                   
                   'Filmposter Archiv':
                   (u"download big picture from filmposter-archiv.de",
                    [('filmposterarchiv-de.pl', 'picture')]
                    ),
                   },
        
        'defaultScript': '[Merkosh Special]',       
        }
    
    def __init__(self, *args, **kwargs):
        # me is a dictionarry with all settings
        dict.__init__(self, *args, **kwargs)
        dict.update(self, _Settings0.default)


    def asText(self):
        """returns a dict which interpolates configText"""
        # whats needed to be imported
        defs = {'import': "\n".join(["import os, logging",
                                     "import MediaInfo"])}

        # general fields
        for key in ['logLevel', 'defaultPath', 'scriptPath', 'browserPath',
                    'proxy', 'systemEncoding', 'recordStoreOrderRandom',
                    'embeddedImage', 'imageFormat', 'filenamePrefix', 'sortColumn',
                    'zeroPrefix_patch', 'scripts', 'defaultScript', 'autoMountPoint',
                    'ejectOnUnmount', 'defaultHistory', 'addHistoryFromRecords',
                    'recordDefault', 'autoScanAutoMountOnNew', 'autoScanOnNew',
                    'autoScriptOnNew', 'deepcopyOnCopy']:
            defs[key] = pformat(self[key])

        # class instance 
        defs['mediaInfo'] = 'MediaInfo.Tcprobe'

        # return results
        return defs



#-- main class for config format -----------------------------------------------
# If the configuration format needs to be changed and in order to keep
# a transparent layer of compatibility for seemless upgrade the following
# strategy is to be implemented:
#
# A new subclass of _SettingsX is created, subclassing _Settings(X-1).
# In there the methods asText and update() are overwritten to in such
# a way that when calling the parent's asText() or update() method the
# interface to the old-style configuration is maintained and the return
# value reflects the current state of the configuration.
class _Settings(_Settings0):
    """encapsulates loading, updating and saving of global program defaults

    Compared to the _Pref class, this contains the global, cross-class
    program settings. Class-based settings are available via the __Pref
    class.
    """
    singleton = None
    
    def __init__(self):
        """returns settings object object"""
        # run parent constructor
        super(_Settings, self).__init__()
            
        # runtime variables
        self.userdir = fjoin(os.environ['HOME'], '.lmc')
        self.configFile = fjoin(self.userdir, "lmc.conf")
        
        self.log = logging.getLogger(self.__class__.__name__)
        self.log.setLevel(self['logLevel']['Settings'])


    def save(self):
        """saves the current version of user settings"""
        # target folder does not exist: create
        if (not fexists(self.userdir)):
            self.log.info("Creating config directory %s", self.userdir)
            os.mkdir(self.userdir)

        # write configuration
        self.log.info("Saving current configuration")
        try:
            f = open(self.configFile, "w")
            f.write( CONFIG_TEXT%self.asText() )
        except IOError, e:
            self.log.error("Could not save configuration settings to %s: %s", configFile, unicode(e))
        else:
            f.close()
        

    def load(self):
        """loads user settings"""
        # if no config exists: create one
        if (not fexists(self.userdir)):
            self.log.info("Creating default configuration file")
            self.save()

        # execute configuration file
        self.log.info("Loading program configuration")
        config = dict()
        try:
            execfile(self.configFile, {}, config)
        except StandardError, e:
            self.log.error("Cannot load configuration file: %s", unicode(e))
            return
        except TypeError, e:
            self.log.error("Cannot unpack configuration settings: %s", unicode(e))
            return

        # add missing from default
        self.update(config)
        


#-- preferences class ----------------------------------------------------------
class _Pref(dict):
    """defines dictionary access to global program settings

    Compared to the _Settings class, this provides class-based
    preferences in form of a dictionary.
    """
    preferences = []
    
    def __init__(self, name):
        """stores preferences for module /class <name>"""
        self.name = name
        _Pref.preferences.append(self)

        # create logging
        self.log = logging.getLogger(self.__class__.__name__)
        self.log.setLevel(logging.DEBUG)


    def update(self, pref):
        """updates the own contents, given a template"""
        self.clear()
        # update simple values
        for key in ['defaultPath', 'scriptPath', 'browserPath', 'proxy',
                    'systemEncoding', 'recordStoreOrderRandom', 'mediaInfo',
                    'embeddedImage', 'imageFormat', 'filenamePrefix',
                    'sortColumn', 'zeroPrefix_patch', 'scripts',
                    'defaultScript', 'autoMountPoint', 'ejectOnUnmount',
                    'defaultHistory', 'addHistoryFromRecords', 'recordDefault',
                    'autoScanAutoMountOnNew', 'autoScanOnNew', 'autoScriptOnNew',
                    'deepcopyOnCopy']:
            self[key] = deepcopy(pref[key])

        # debug info on a per-class basis
        if (not pref['logLevel'].has_key(self.name)):
            self.log.error("No specialised settings available for %s; using default", self.name)
            self.name = 'default'
        self['logLevel'] = deepcopy(pref['logLevel'][self.name])
        

#-- user functions -------------------------------------------------------------
def Settings():
    """helper function to create singleton Settings object"""
    if (_Settings.singleton == None):
        _Settings.singleton = _Settings()

    return _Settings.singleton


def getPreferences(name='<none>'):
    """returns a per-class definition of the current preferences

    The name parameter specifies which per-class preferences should be
    returned. If no class preferences exist for this name then a default
    valued instance will be returned.
    Two invocations with the same name do not yield the identical object,
    just copies with the same values.
    """
    tmp = _Pref(name)
    tmp.update(Settings())
    return tmp


def updatePreferences():
    """updates all preferences accoring to the current global settings"""
    s = Settings()
    for pref in _Pref.preferences:
        pref.update(s)



#-- testroutine ----------------------------------------------------------------
if (__name__ == '__main__'):
    log = logging.getLogger("")
    hdlr = logging.StreamHandler()
    formatter = logging.Formatter('[%(name)s] %(levelname)s: %(message)s')
    hdlr.setFormatter(formatter)
    log.addHandler(hdlr)
    log.setLevel(logging.DEBUG)

    log.info("Resoring default configuration")
    a = Settings()
    a.save()
    b = getPreferences()
