# -*- coding: utf-8 -*-
# Author:   $Author: merkosh $
# Revision: $Rev: 73 $
############################################################################
#    Copyright (C) 2005 by Uwe Mayer                                       #
#    merkosh@hadiko.de                                                     #
#                                                                          #
#    This program is free software; you can redistribute it and/or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             #
############################################################################

#-- imports --------------------------------------------------------------------
#-- Qt imports
from qt import SIGNAL, QListView, QListViewItem, QGridLayout
from qt import PYSIGNAL, QImage, QPixmap


#-- Python imports
import logging
import sys

from types import ListType
from os.path import exists as fexists


#-- own imports
from RecordListBase import RecordListBase
from misc import correctSlice
from AMCForm import AMCForm
from AbstractFile import AbstractFileRecord
from AMCFile33 import AMCRecord33
from PicturePreview import PicturePreview
from Settings import getPreferences

# TODO: use C++ compiled version
#from Qt.QListViewItemNumeric.QListViewItemNumeric import QListViewItemNumeric


#-- preferences ----------------------------------------------------------------
pref = getPreferences(__name__)


#-- About ListViews and RecordList --
#
# ListViews are strange objects. Originally RecordList should have
# supported the list-interface. However, there is no real order on
# ListView objects. With auto-sorting turned on a ListView is more
# an iterable, set-like object. Element access occurs by object
# reference.
#
class RecordList(RecordListBase):
    """initialises the central ListView

    RecordList displays a list of records in a graphical interface.
    """
    def __init__(self, dirname, parent=None, name=None, fl=0):
        RecordListBase.__init__(self, parent, name, fl)

        # create logging
        self.log = logging.getLogger( self.__class__.__name__ )
        self.log.setLevel(pref['logLevel'])

        # embedd form on right hand side
        self.form = AMCForm( self.frameRecord )
        self.form.hide()              # show only if records are available
        formLayout = QGridLayout( self.frameRecord, 1,1 )
        formLayout.addWidget( self.form, 0,0 )

        self.image = PicturePreview( dirname, self.frameImage )
        imageLayout = QGridLayout( self.frameImage, 1,1 )
        imageLayout.addWidget( self.image, 0,0 )

        self.listViewRecord.setAllColumnsShowFocus( True )
        self.listViewRecord.setSorting( pref['sortColumn'] )
        self.listViewRecord.setShowSortIndicator( True )
        self.listViewRecord.setColumnWidthMode( 0, QListView.Manual )
        self.listViewRecord.setColumnWidthMode( 1, QListView.Maximum )
        self.listViewRecord.setResizeMode( QListView.LastColumn )
        self.listViewRecord.setSelectionMode( QListView.Single )
        self.listViewRecord.setDefaultRenameAction( QListView.Accept )

        self.__obj = None               # keeps track of the last item added
        self.__items = {}               # maps ListViewItem -> record

        self.connect( self.listViewRecord, SIGNAL("currentChanged(QListViewItem*)"), self.onCurrentChanged )
        self.connect( self.listViewRecord, SIGNAL("itemRenamed(QListViewItem*,int,const QString&)"), self.onItemRenamed )
        self.connect( self.form, PYSIGNAL("modified()"), self.onModified )
        # currently not needed: 
        # self.connect( self.image, PYSIGNAL("modified()"), self.onModified )
        

    #-- list interface ---------------------------------------------------------
    def add(self, rec):
        """W.add(rec) -- append record(s)"""
        # add all new records
        if (not type(rec) is ListType): rec = [rec]
        obj = None
        for r in rec:
            if (not isinstance(r, AbstractFileRecord)):
                r = AMCRecord33(r)
            obj = QListViewItem( self.listViewRecord, *self.__compileTitles(r) )
            obj.setRenameEnabled(0, True)
            self.__items[obj] = r
        # keep track of last item added
        self.__obj = obj
        
        # this also shows the main form when the first record becomes
        # available
        if (obj != None): self.form.show()


    def iterkeys(self):
        """returns iterator over all ListViewItems"""
        return iter(self.__items.keys())

    def itervalues(self):
        """returns iterator over all records"""
        return iter(self.__items.values())

    def keys(self):
        """returns an unsorted list of all keys"""
        return self.__items.keys()

    def values(self):
        """returns an unsorted list of all records"""
        return self.__items.values()

    def __delitem__(self, y):
        """removes ListViewItem(s) by reference"""
        self.log.debug("deleting record %s", y)
        if (not type(y) is ListType): y = [y]
        for obj in y:
            self.listViewRecord.takeItem(obj)
            del self.__items[obj]

    def __len__(self):
        """returns the number of elements in the list view"""
        return len( self.__items.keys() )

    def getCurrentRecord(self):
        return self.__items[self.listViewRecord.currentItem()]

    def setCurrentRecord(self, record):
        currentObj = self.listViewRecord.currentItem()
        self.__items[currentObj] = record
        self.onCurrentChanged(currentObj)
        self.onModified()


    #-- observer ---------------------------------------------------------------
    def onCurrentChanged(self, obj):
        """called when the \"current\" item changes"""
        if (obj != None):
            self.form.setRecord( self.__items[obj] )
            self.image.setRecord( self.__items[obj] )
        else:
            self.form.setRecord(None)
            self.image.setRecord(None)


    def onModified(self):
        """called when a record is modified in the form"""
        # update ListViewItem title
        index = 0
        record = self.__items[self.listViewRecord.currentItem()]
        for text in self.__compileTitles(record):
            self.listViewRecord.currentItem().setText(index, text)
            index += 1


    def onItemRenamed(self, obj, col, text):
        """called when record number is modified in the list view"""
        try:
            number = int(unicode(text))
        except ValueError:
            pass
        else:
            self.__items[obj]['number'] = number
        # update list view
        self.onModified()
        self.listViewRecord.sort()

    
    #-- utility functions ------------------------------------------------------
    def focus(self, obj):
        """displays the object"""
        if (obj != None):
            self.listViewRecord.ensureItemVisible(obj)
            self.listViewRecord.setCurrentItem(obj)
            self.onCurrentChanged(obj)
    
    def focusFirst(self):
        """displays the first item in the ListView"""
        self.focus(self.listViewRecord.firstChild())

    def focusNew(self):
        """displays the last item added"""
        self.focus(self.__obj)

    def __compileTitles(self, record):
        """return a text-tuple what forms the ListViewItem entries """
        num = unicode(record['number'])
        if (pref['zeroPrefix_patch']):
            num = u'0'*(4-len(num))+num
        return (num,
                record['mediaLabel'] or record['originalTitle'] or record['translatedTitle'])
        
    

#-- testroutine ----------------------------------------------------------------
if (__name__ == "__main__"):
    from qt import QApplication, QObject, SLOT

    a = QApplication(sys.argv)
    QObject.connect(a,SIGNAL("lastWindowClosed()"),a,SLOT("quit()"))

    # need logging handlers for testing
    log = logging.getLogger("")
    hdlr = logging.StreamHandler()
    formatter = logging.Formatter('[%(name)s] %(levelname)s: %(message)s')
    hdlr.setFormatter(formatter)
    log.addHandler(hdlr) 
    log.setLevel(pref['logLevel'])

    w = RecordList()

    from AMCFile33 import AMCFile33
    f = AMCFile33("Filme.amc", "rb")
    
    rec = f.read()
    w.add( rec )
    w.focusFirst()

    a.setMainWidget(w)
    w.show()

    a.exec_loop()

#-- EOF --
