# -*- coding: utf-8 -*-
# Author:   $Author: merkosh $
# Revision: $Rev: 81 $
############################################################################
#    Copyright (C) 2005 by Uwe Mayer                                       #
#    merkosh@hadiko.de                                                     #
#                                                                          #
#    This program is free software; you can redistribute it and/or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             #
############################################################################

#-- imports --------------------------------------------------------------------
#-- Qt imports
from qt import SIGNAL, PYSIGNAL, QDate, QStringList


#-- Python imports
import logging
import sys
import os

from types import IntType


#-- own imports
from AMCFormBase import AMCFormBase
from AbstractFile import AbstractFileRecord
from AMCFile33 import AMCRecord33
from misc import Date2YMD, YMD2Date
from Settings import getPreferences


#-- preferences ----------------------------------------------------------------
pref = getPreferences(__name__)


#-- mappings -------------------------------------------------------------------
field2lineEdit = {
    'mediaLabel'      : 'lineEditMediaLabel',
    'originalTitle'   : 'lineEditOriginalTitle',
    'translatedTitle' : 'lineEditTranslatedTitle',
    'director'        : 'lineEditDirector',
    'producer'        : 'lineEditProducer',
    'url'             : 'lineEditURL',
    'resolution'      : 'lineEditResolution',
    'fileSize'        : 'lineEditFileSize'
    }
field2textEdit = {
    'actors'          : 'textEditActors',
    'description'     : 'textEditDescription',
    'comments'        : 'textEditComment'
    }
field2comboBox = {
    'subtitles'       : 'comboBoxSubtitle',
    'languages'       : 'comboBoxLanguage',
    'framerate'       : 'comboBoxFramerate',
    'videoFormat'     : 'comboBoxVideoFormat',
    'audioFormat'     : 'comboBoxAudioFormat',
    'category'        : 'comboBoxCategory',
    'country'         : 'comboBoxCountry',
    'mediaType'       : 'comboBoxMediaType',
    'source'          : 'comboBoxSource',
    'borrower'        : 'comboBoxBorrower'
    }
field2spinBox = {
    'discs'           : 'spinBoxDiscs',
    'videoBitrate'    : 'spinBoxVideoKBPS',
    'audioBitrate'    : 'spinBoxAudioKBPS',
    'length'          : 'spinBoxLength',
    'rating'          : 'spinBoxRating',
    'year'            : 'spinBoxYear'
    }
field2dateEdit = {
    'date'            : 'dateEditDateAdded'
    }



#-- history --------------------------------------------------------------------
comboHistory = {
    'subtitles'       : {},
    'languages'       : {},
    'framerate'       : {},
    'videoFormat'     : {},
    'audioFormat'     : {},
    'category'        : {},
    'country'         : {},
    'mediaType'       : {},
    'source'          : {},
    'borrower'        : {}
    }



class AMCForm(AMCFormBase):
    def __init__(self, parent=None, name=None, fl=0):
        """initialises the AMCForm and connects signals

        AMCForm displays record data in a graphical interface.
        Read- /write access of certain fields can be achieved using
        the dictionary interface. Invalid field names will be ignored
        but create a debug message.

        Complete records can be read /set using the getRecord() and
        setRecord() methods.

        AMCForm keeps a modified flag for the currently displayed
        record, which can be read as attribute \"modified\".

        Signals:
          modified()    called each time user modifies a field
        """
        AMCFormBase.__init__(self, parent, name, fl)

        # create logging
        self.log = logging.getLogger("AMCForm")
        self.log.setLevel( pref['logLevel'] )
        
        self.record = None            # currently displayed record

        # True <=> change-handler has been installed
        # the signal handler is called each time a modification on the
        # GUI fields is made; setting a complete record with setRecord()
        # does not trigger the signal handler
        self.__installedChangeHandler = False

        # customisations
        for fname in field2comboBox.values():
            field = getattr(self, fname)
            # maximum 10 elements
            field.setMaxCount(10)
            
        # set default history
        for fname in pref['defaultHistory']:
            # create QStringList with elements
            history = QStringList()
            for element in pref['defaultHistory'][fname]:
                history.append(element)
                # remember in runtime-history
                comboHistory[fname][element] = 1
            # set history
            getattr(self, field2comboBox[fname]).insertStringList(history)

        # connect signals
        self.connect(self.pushButtonURL, SIGNAL("pressed()"), self.onPushButtonURL)

        
    #-- slots ------------------------------------------------------------------
    def onPushButtonURL(self):
        """executes browser with URL in lineEditURL"""
        if (self['url']):
            if (pref['browserPath'] == ""):
                self.log.warning(u"No browser specified")
                return
            os.spawnlp(os.P_NOWAIT, pref['browserPath'], pref['browserPath'], self['url'])
            # no error handling to be done: cannot intercept return code with
            # P_NOWAIT


    #-- signal handler ---------------------------------------------------------
    def onModified(self):
        """sets the modified flag of the currently displayed record and emits the modified() signal"""
        if (self.record != None): self.record.modified(True)
        self.getRecord()                # updated record from form
        self.emit(PYSIGNAL("modified()"), ())



    #-- dictionary interface ---------------------------------------------------
    def keys(self):
        """returns a list of all supported fields"""
        return field2lineEdit.keys()+\
               field2textEdit.keys() +\
               field2comboBox.keys() +\
               field2spinBox.keys() +\
               field2dateEdit.keys()


    def has_key(self, k):
        """returns wether key <k> exists"""
        return (k in self.keys())

    
    def get(self, k,d=None):
        """D.get(k[,d]) -> D[k] if k in D, else d.  d defaults to None"""
        if (self.has_key(k)): return self[k]
        return d


    def __setitem__(self, name, value):
        """loads a value from a record into the fields of the user interface"""        
        if (field2lineEdit.has_key(name)):
            getattr(self, field2lineEdit[name]).setText(unicode(value).strip())
        elif (field2textEdit.has_key(name)):
            getattr(self, field2textEdit[name]).setText(unicode(value).strip())
        elif (field2comboBox.has_key(name)):
            # simplify value
            value = unicode(value).strip()
            # check if element is already in history
            if (pref['addHistoryFromRecords'] and (value != "") and
                (not comboHistory[name].has_key(value))):
                # no: add to history
                getattr(self, field2comboBox[name]).insertItem(value, 0)
                comboHistory[name][value] = 1
            else:
                # yes: only display element
                getattr(self, field2comboBox[name]).setEditText(value)
        elif (field2spinBox.has_key(name)):
            # debugging
            try:
                value = int(round(float(value)))
                self.record[name] = value
                getattr(self, field2spinBox[name]).setValue(value)
            except TypeError:
                self.log.error(u"Error setting %s to %s; changing to 0", name, value)
                self.record[name] = 0
                self[name] = 0
                self.modified()
            except ValueError:
                self.log.error(u"Cannont convert from string to integer: %s=%s", name, value)
                self.record[name] = 0
                self[name] = 0
                self.modified()
        elif (field2dateEdit.has_key(name)):
            y,m,d = Date2YMD( value )
            getattr(self, field2dateEdit[name]).setDate( QDate(y,m,d) )
        else:
            self.log.debug("Cannot set unknown option %s=\"%s\"", name, unicode(value)[:20])


    def __getitem__(self, name):
        """returns the value of a field"""
        if (field2lineEdit.has_key(name)):
            return unicode(getattr(self, field2lineEdit[name]).text())
        elif (field2textEdit.has_key(name)):
            return unicode(getattr(self, field2textEdit[name]).text())
        elif (field2comboBox.has_key(name)):
            return unicode(getattr(self, field2comboBox[name]).currentText())
        elif (field2spinBox.has_key(name)):
            return int( getattr(self, field2spinBox[name]).value() )
        elif (field2dateEdit.has_key(name)):
            obj = getattr(self, field2dateEdit[name]).date()
            return YMD2Date( obj.year(), obj.month(), obj.day() )
        else:
            self.log.warning("Cannot get unknown option %s", name)


    #-- extended class interface -----------------------------------------------
    def setRecord(self, rec):
        """sets a record and displays it on the screen"""
        # set failsafe default for log output
        if (not rec.has_key('number')): rec['number'] = 0
        if (not rec.has_key('mediaLabel')): rec['mediaLabel'] = ''
        self.log.debug("Setting record: %d, %s", rec['number'], rec['mediaLabel'])

        if (self.__installedChangeHandler): self.__removeChangeHandler()

        # remember original record
        self.record = rec

        # set field to new values
        for field in self.keys():
            if (rec.has_key(field)):
                self[field] = rec[field]
            else:
                # field not available: set default
                self.log.warning("Missing field value %s; using default", field)
                self[field] = pref['recordDefault'][field]

        self.__installChangeHandler()


    def getRecord(self):
        """returns the currently displayed record"""
        self.log.debug("Returning record: %d, %s", self.record['number'], self.record['mediaLabel'])

        # create empty record if none set
        if (self.record == None): return None

        # get values from fields
        for field in self.keys():
            self.record[field] = self[field]

        # return record
        return self.record


    def focus(self, field):
        """sets keyboard focus on field"""
        if (field2lineEdit.has_key(field)):
            getattr(self, field2lineEdit[field]).setFocus()        
        elif (field2textEdit.has_key(field)):
            getattr(self, field2textEdit[field]).setFocus()
        elif (field2comboBox.has_key(field)):
            getattr(self, field2comboBox[field]).setFocus()
        elif (field2spinBox.has_key(field)):
            getattr(self, field2spinBox[field]).setFocus()
        elif (field2dateEdit.has_key(field)):
            getattr(self, field2dateEdit[field]).setFocus()
        else:
            self.log.warning("Cannot focus on unknown field %s", field)
        



              
    #-- miscellaneous methods -------------------------------------------------- 
    def __installChangeHandler(self):
        """installes a change-handler that is called when a field changes"""
        self.log.debug(u"Connecting change-handler")

        if (self.__installedChangeHandler):
            self.log.warning(u"Change-handler already installed")
            return
        for val in field2lineEdit.values() +field2comboBox.values():
            self.connect(getattr(self, val), SIGNAL("textChanged(const QString&)"), self.onModified)
        for val in field2textEdit.values():
            self.connect(getattr(self, val), SIGNAL("textChanged()"), self.onModified)
        for val in field2spinBox.values():
            self.connect(getattr(self, val), SIGNAL("valueChanged(int)"), self.onModified)
        for val in field2dateEdit.values():
            self.connect(getattr(self, val), SIGNAL("valueChanged(const QDate&)"), self.onModified)
        self.__installedChangeHandler = True


    def __removeChangeHandler(self):
        """removes the change-handler"""
        self.log.debug("disconnecting change-handler")

        if (not self.__installedChangeHandler):
            self.log.warning("change-handler not installed")
            return
        for val in field2lineEdit.values() +field2comboBox.values():
            self.disconnect(getattr(self, val), SIGNAL("textChanged(const QString&)"), self.onModified)
        for val in field2textEdit.values():
            self.disconnect(getattr(self, val), SIGNAL("textChanged()"), self.onModified)
        for val in field2spinBox.values():
            self.disconnect(getattr(self, val), SIGNAL("valueChanged(int)"), self.onModified)
        for val in field2dateEdit.values():
            self.disconnect(getattr(self, val), SIGNAL("valueChanged(const QDate&)"), self.onModified)
        self.__installedChangeHandler = False
        


#-- test main program ----------------------------------------------------------
if __name__ == "__main__":
    from qt import SLOT, QApplication, QObject
    import Settings

    Settings.Settings().load()

    a = QApplication(sys.argv)
    QObject.connect(a,SIGNAL("lastWindowClosed()"), a,SLOT("quit()"))
    w = AMCForm()

    from AMCFile33 import AMCFile33
    f = AMCFile33("Filme.amc", "rb")
    rec = f.read(1)[0]
    w.setRecord( {} )

    a.setMainWidget(w)
    w.show()
    a.exec_loop()

#-- EOF --
