# -*- coding: utf-8 -*-
# Author:   $Author: merkosh $
# Revision: $Rev: 154 $
############################################################################
#    Copyright (C) 2005 by Uwe Mayer                                       #
#    merkosh@hadiko.de                                                     #
#                                                                          #
#    This program is free software; you can redistribute it and/or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             #
############################################################################
"""Implementation of the Ant Movie Catalogue file specification v3.3"""

#-- imports --------------------------------------------------------------------
#-- Python imports
import logging

from struct import pack, unpack, calcsize, error as StructError
from os import fstat
from stat import ST_SIZE
from cStringIO import StringIO
from types import TupleType, DictType


#-- own imports 
from AbstractFile import AbstractFile, AbstractFileRecord
from Language import TR
from Settings import getPreferences


#-- constants & preferences ---------------------------------------------------
REC_POS = 0
REC_FMT = 1

pref = getPreferences(__name__)

    
#-------------------------------------------------------------------------------
class AMCRecord33(AbstractFileRecord):
    """Implementation of the AMC Record format, version 3.3

    Available fields are:
    #-- 32 bit unsigned integers
    number, date, rating, year, length, videoBitrate, audioBitrate,
    discs
    #-- 8 bit boolean 
    checked
    #-- Delphi long strings: <32-bit-unsigned-int><length terminated data>
    mediaLabel, mediaType, source, borrower, originalTitle, translatedTitle,
    director, producer, country, category, actors, url, description,
    comments, videoFormat, audioFormat, resolution, framerate, languages,
    subtitles, fileSize, picture, embeddedImage

    The record structure internally stores a version number (currently unused)
    and updates the modified-flag.

    Additionally to the AbstractFileRecord interface this class introduces the
    method genReadFMT() which generates a struct-like format that enables the
    creation of an index file for faster read/write access by AMCFile33. This
    method is compulsory.
    Also two methods load() and dump() are provided which initialise the
    record from a tupple or dumps a binary representation of it, respectively.

    The fields can be accessed via the dictionary interface. All operations
    are supported, except for deleting elements, which raises an
    AttributeError.
    """
    version = "3.3"                     # internal version number

    __default = {'number': 0, 'date': 1, 'rating': 0, 'year': 0,
                 'length': 0, 'discs': 0, 'checked': 0,
                 'videoBitrate': 0, 'audioBitrate': 0,
                 'mediaLabel': "", 'mediaType': "", 'source': "",
                 'borrower': "", 'originalTitle': "",
                 'translatedTitle': "", 'director': "", 'producer': "",
                 'country': "", 'category': "", 'actors': "",
                 'url': "", 'description': "", 'comments': "",
                 'videoFormat': "", 'audioFormat': "",
                 'resolution': "", 'framerate': "", 'languages': "",
                 'subtitles': "", 'fileSize': "", 'picture': "",
                 'embeddedImage': ""}

    log = logging.getLogger("AMCRecord33")

    def __init__(self, arg=__default):
        """creates a new object

        When initialising an instance upon creation the <modified>
        flag is cleared.

        The copy-constructor from the base class may be used, but
        will be overwritten when the first parameter is a 32-tuple.
        """
        AbstractFileRecord.__init__(self, arg)

        if (type(arg) is TupleType):
            self.load(arg)
            self.modified(False)
            
        # make sure partially incomplete dict is completed
        if (type(arg) is DictType):
            for k in AMCRecord33.__default.keys():
                self[k] = AMCRecord33.__default[k]
            for k in arg:
                self[k] = arg[k]
            self.modified(False)


    #-- additional methods
    def load(self, *arg):
        """initialises the object from a string and a given format

        Param: <arg> is a 32-tupel

        Strings are decoded as utf-8 strings by default (with fallback)
        except for the embedded image, which is kept in ascii encoding.
        """
        self['number'],       self['date'],   self['rating'], \
        self['year'],         self['length'], self['videoBitrate'], \
        self['audioBitrate'], self['discs'],  self['checked'] = arg[0][:9]

        try:
            self['mediaLabel'],  self['mediaType'],     self['source'], \
            self['borrower'],    self['originalTitle'], self['translatedTitle'], \
            self['director'],    self['producer'],      self['country'], \
            self['category'],    self['actors'],        self['url'], \
            self['description'], self['comments'],      self['videoFormat'], \
            self['audioFormat'], self['resolution'],    self['framerate'], \
            self['languages'],   self['subtitles'],     self['fileSize'], \
            self['picture'] = map(lambda a: unicode(a, 'utf8'), arg[0][9:-1])
        except UnicodeDecodeError, e:
            self.log.error(u"Cannot decode input string: %s, trying %s", unicode(e), pref['systemEncoding'])
            self['mediaLabel'],  self['mediaType'],     self['source'], \
            self['borrower'],    self['originalTitle'], self['translatedTitle'], \
            self['director'],    self['producer'],      self['country'], \
            self['category'],    self['actors'],        self['url'], \
            self['description'], self['comments'],      self['videoFormat'], \
            self['audioFormat'], self['resolution'],    self['framerate'], \
            self['languages'],   self['subtitles'],     self['fileSize'], \
            self['picture'] = map(lambda a: unicode(a, pref['systemEncoding']), arg[0][9:-1])
            self.modified(True)
        self['embeddedImage'] = arg[0][-1]


    def packPasString(self, text, encodeUTF8=True):
        """pack a Python string to a 32 bit LTS
        
        Return: <32 bit LTS>
        """
        if (encodeUTF8):
            try:
                tmp = unicode(text).encode('utf8')
            except UnicodeDecodeError, e:
                self.log.error(u"Cannot encode text to utf8: %s; trying %s", unicode(e), pref['systemEncoding'])
                tmp = unicode(text, pref['systemEncoding']).encode("utf8")
        else:
            tmp = text
        return pack("L",len(tmp))+tmp


    def dump(self):
        """returns a binary representation of the record

        String contents is wrapped into utf-8 except for the embedded image
        which is stored as ascii.
        Integers are stored little endian.
        """
        try:
            text = pack("8LB", self['number'], self['date'], self['rating'], \
                        self['year'], self['length'], self['videoBitrate'], \
                        self['audioBitrate'], self['discs'], self['checked'])
        except StructError, e:
            numFields = "  number:       "+unicode(self['number'])+"\n"+\
                        "  date:         "+unicode(self['data'])+"\n"+\
                        "  rating:       "+unicode(self['rating'])+"\n"+\
                        "  year:         "+unicode(self['year'])+"\n"+\
                        "  length:       "+unicode(self['length'])+"\n"+\
                        "  videoBitrate: "+unicode(self['videoBitrate'])+"\n"+\
                        "  audioBitrate: "+unicode(self['audioBitrate'])+"\n"+\
                        "  discs:        "+unicode(self['discs'])+"\n"+\
                        "  checked:      "+unicode(self['checked'])+"\n"
            self.log.error("Cannot convert number to binary: %s; setting to zero\nOriginal numerical fields:\n%s", unicode(e), numFields)
            text = pack("8LB", 0, 0, 0, 0, 0, 0, 0, 0, 0)

        text += self.packPasString(self['mediaLabel'])
        text += self.packPasString(self['mediaType'])
        text += self.packPasString(self['source'])
        text += self.packPasString(self['borrower'])
        text += self.packPasString(self['originalTitle'])
        text += self.packPasString(self['translatedTitle'])
        text += self.packPasString(self['director'])
        text += self.packPasString(self['producer'])
        text += self.packPasString(self['country'])
        text += self.packPasString(self['category'])
        text += self.packPasString(self['actors'])
        text += self.packPasString(self['url'])
        text += self.packPasString(self['description'])
        text += self.packPasString(self['comments'])
        text += self.packPasString(self['videoFormat'])
        text += self.packPasString(self['audioFormat'])
        text += self.packPasString(self['resolution'])
        text += self.packPasString(self['framerate'])
        text += self.packPasString(self['languages'])
        text += self.packPasString(self['subtitles'])
        text += self.packPasString(self['fileSize'])
        text += self.packPasString(self['picture'])
        text += self.packPasString(self['embeddedImage'], False)
        return text

       
    def genReadFMT(cls, f):
        """generates a read format from file f

        Param: <f> must support the file interface 

        This is essential for updating the recordMap for read access
        on AMC files.

        To generate a record format from a string use:
          genReadFMT( StringIO(<string>) )

        To generate a record format from an initialised record:
          genReadFMT( StringIO(<record>.dump()) )
        """
        format = "8LB"
        f.seek(33, 1)
        for i in range(23):
            size = unpack("L", f.read(4))[0]
            f.seek(size, 1)
            format += "4x"+str(size)+"s"
        return format
    genReadFMT = classmethod(genReadFMT)



#-------------------------------------------------------------------------------
class AMCFile33(AbstractFile):
    """Implementation of the Ant Movie Catalogue file specification v3.3

    Upon opening a file, the file header is checked. On ID mismatch a TypeError
    is raised. Otherwhise the 5 global strings (name, mail, icq, site and description)
    are read from the beginning, which are accessible via the dictionary interface.
    """
    __fileVersionID = " AMC_3.3 Ant Movie Catalog 3.3.x   www.buypin.com  www.ant.be.tf "


    def __init__(self, filename, mode="rb", bufsize=-1):
        """opens a new file in mode <mode>

        raises TypeError if file is no AMC file v3.3
        """
        AbstractFile.__init__(self)

        self.log = logging.getLogger( self.__class__.__name__ )

        self.name = filename
        self.mode = mode
        
        self.__file = file(filename, mode, bufsize)
        self.closed = self.__file.closed
        self.encoding = self.__file.encoding
        self.newlines = self.__file.newlines
        self.softspace = self.__file.softspace
        self.__fsize = fstat(self.__file.fileno())[ST_SIZE]
        self.__recordMap = []           # indicates start and format of records
        self.__recordIndex = 0          # current logical file pointer

        # test if it is a new file, create header
        if (self.__fsize == 0):
            self.__writeHeader()
            self.__fsize = self.__file.tell()
        else:
            #-- test for file version
            text = self.__file.read(len(self.__fileVersionID))
            if (text != self.__fileVersionID):
                self.__file.close()
                raise TypeError(TR(u"File ID mismatch: this is no valid AMC file v3.3"))

            #-- load: name, mail, icq, site, description
            self['name'] = self.__readPasString()
            self['mail'] = self.__readPasString()
            self['icq'] = self.__readPasString()
            self['site'] = self.__readPasString()
            self['description'] = self.__readPasString()
            self.__modified = False
        
        #-- load file structure
        self.__loadStructure()


    #-- track write access to header fields    
    def __setitem__(self, name, value):
        AbstractFile.__setitem__(self, name, value)
        if (name in ['name', 'mail', 'icq', 'site', 'description']):
            self.__modified = True 


    #-- some helper functions
    def __writeHeader(self):
        """writes the file header

        The file pointer must already point to the correct location
        and is not restored to its original position.
        """
        self.__file.write(AMCFile33.__fileVersionID)        
        for data in ['name', 'mail', 'icq', 'site', 'description']:           
            raw = self[data] or ""
            try:
                value = unicode(raw).encode('utf8')
            except UnicodeDecodeError, e:
                self.log.error(u"Cannot encode text to utf8: %s; trying %s", unicode(e), pref['systemEncoding'])
                value = unicode(raw, pref['systemEncoding']).encode('utf8')
            self.__file.write( pack("L", len(value)) )
            self.__file.write( value )
        self.__modified = False


    def __loadStructure(self):
        """loads the record structure from an opend file

        The file pointer must be positioned at the first record. The
        result is stored in self.__recordMap.
        The file pointer and record index are not moved.
        """
        startPos = self.__file.tell()
        self.__recordMap = []
        while (not self.isEOF()):
            position = self.__file.tell()
            format = AMCRecord33.genReadFMT(self.__file)
            self.__recordMap.append( (position, format) )

        self.__recordMap.append( (self.__fsize, "") ) # == EOF
        self.__file.seek(startPos)


    def __shiftStructure(self, offset, recRange=None):
        """shifts start adresses of the record format map by <offset>

        Param: <offset> will be *added* to the current address
               <start> all records from this starting index will be
                       processed
                       default: process all records

        After write accesses the start offsets of the records need
        to be shifted. Modifying the offsets is probably faster than
        re-reading the format structure.
        """
        if (recRange == None): recRange = range(0, len(self.__recordMap))
        for i in recRange:
            x = self.__recordMap[i]
            self.__recordMap[i] = (x[REC_POS]+offset, x[REC_FMT])
        

    def __readFMT(self, format):
        """basic read-format method: reads structured data from file

        Param: <fmt> = format as in struct
        Return: read format, wrapped in a tupple

        raises StructError if format cannot be parsed, except on EOF
          (AMC seems to be sloppy if the last record in the file
          has emty tailing fields)
        """
        try:
            return unpack(format, self.__file.read(calcsize(format)) )
        except StructError:
            # if EOF and text is smaller than the format they're probbably
            # just missing; AMC seems to ignore this
            # fix: ignore if at end-of-file 
            if (not self.isEOF()): raise
    

    def __readPasString(self):
        """reads a 32 bit LTS from file"""
        size = self.__readFMT("L")[0]
        raw = self.__readFMT(str(size)+"s")[0]
        try:
            text = unicode(raw, 'utf8')
        except UnicodeDecodeError,e :
            self.log.error(u"Cannot decode input string: %s, trying %s", unicode(e), pref['systemEncoding'])
            text = unicode(raw, pref['systemEncoding'])
        return text


    #-- these are not supported by <file> ---------------
    def isEOF(self):
        """returns True if at end-of-file"""
        return (self.__file.tell() == self.__fsize)


    def filesize(self):
        """returns the filesize in bytes"""
        return self.__fsize

    
    def filelength(self):
        """returns the number of records in the file"""
        return len(self.__recordMap)-1

    def getRecordFactory(cls):
        """returns an AMCRecord33 factory"""
        return AMCRecord33
    getRecordFactory = classmethod(getRecordFactory)
    

    #-----------------------------------------------------


    def close(self):
        """flushes and closes the file"""
        # only direct flushes should store header data
#        self.flush()
        self.__file.close()
        self.closed = self.__file.closed

        
    def flush(self):
        """flushes the file buffers and writes any changes to disc

        Only this causes modifications to the file header to be
        saved permanently. Its somehow inefficient since all of
        the file has to be read into memory and saved to disk shortly
        afterwards.
        """
        if (self.__modified):
            # save to file
            self.__file.seek( self.__recordMap[0][REC_POS] )
            buf = self.__file.read()
            self.__file.seek(0)
            self.__file.truncate()
            self.__writeHeader()
            headerPos = self.__file.tell()
            self.__file.write( buf )
            self.__file.seek( headerPos )
            # adjust internal structures
            buf = headerPos -self.__recordMap[0][REC_POS] # difference in filesize
            self.__fsize += buf
            self.__shiftStructure(buf)
            # return to previous position
            self.__file.seek( self.__recordMap[self.__recordIndex][REC_POS] )            

        # what we came here for:
        self.__file.flush()


    def read(self, count=-1):
        """returns at max the next <count> records

        Return: [<record>, ...]

        raises StructError when record does not match expected size

        Records are returned as instances of <AMCRecord33>.
        Trying to read from end-of-file results in an empty list returned.
        """
        assert (self.__file.tell() == self.__recordMap[self.__recordIndex][REC_POS]), \
               u"internal record pointer does not match actual file pointer: %d vs. %d (idx=%d)"%\
               (self.__file.tell(), self.__recordMap[self.__recordIndex][REC_POS], self.__recordIndex)
        result = []
        while ((count != 0) and (not self.isEOF())):
            data = self.__readFMT( self.__recordMap[self.__recordIndex][REC_FMT] )
            record = AMCRecord33( data )
            result.append( record )
            self.__recordIndex += 1
            count -= 1
        return result


    def seek(self, offset, whence=0):
        """skips to record number <offset>

        Records are labeled increasingly, starting from 0.

        'whence' determines how the offset ist interpreted:
        0 = absolute from start of file
        1 = relative to current position
        2 = absolute from end of file

        raises IndexError if <offset> exceeds filelength()
        """
        if (whence == 0):
            self.__recordIndex = offset
        elif (whence == 1):
            self.__recordIndex += offset
        elif (whence == 2):
            self.__recordIndex = len(self.__recordMap) -(offset +1)

        self.__file.seek( self.__recordMap[self.__recordIndex][REC_POS] )


    def tell(self):
        """returns the record number of the internal file pointer

        Records are labeled increasingly, starting from 0.
        """
        return self.__recordIndex
    

    def truncate(self, size=-1):
        """truncates the file to (at most) that record

        <size> == -1: <current position>
        <size> != -1: <record number from start of file>

        Afterwards the filepointer will be positioned at end of file.
        """
        if (size == -1): size = self.__recordIndex
        self.seek(size)

        self.__file.truncate()
        self.__fsize = fstat(self.__file.fileno())[ST_SIZE]

        del self.__recordMap[size:]
        self.__recordMap.append( (self.__fsize, "") )


    def write(self, record):
        """writes <record> to the current position of the internal file pointer

        Records must be of class AMCRecord33 or a subclass thereof, otherwise
        TypeError ist raised.
        Supports middle-of-file overwrite record. The <modified>-flag is
        cleared when the record has been written to disc.
        """
        assert (self.__file.tell() == self.__recordMap[self.__recordIndex][REC_POS]), \
               u"internal record pointer does not match actual file pointer: %d vs. %d (idx=%d)"%\
               (self.__file.tell(), self.__recordMap[self.__recordIndex][REC_POS], self.__recordIndex)

        if (not isinstance(record, AMCRecord33)):
            record = AMCRecord33(record)  # try to convert record to AMCRecord33
        
        recDump = record.dump()
        newFMT = record.genReadFMT( StringIO(recDump) )
        oldPos = self.__recordMap[self.__recordIndex][REC_POS]        

        if (not self.isEOF()):
            #-- overwrite record
            self.__file.seek( self.__recordMap[self.__recordIndex+1][REC_POS] )
            buf = self.__file.read()          # remember rest-of-file
            self.__file.seek( oldPos )
            self.__file.truncate()
            self.__file.write(recDump)        # write record
            newEnd = self.__file.tell()       # remember end of record
            self.__file.write(buf)            # write rest of original file
            self.__file.seek(newEnd)          # return to end of record

            #-- install new read format
            sizeDiff = calcsize(newFMT) -calcsize(self.__recordMap[self.__recordIndex][REC_FMT])
            self.__recordMap[self.__recordIndex] = (oldPos, newFMT)
            self.__shiftStructure(sizeDiff, range(self.__recordIndex+1, len(self.__recordMap)))
        else:
            self.__file.write(recDump)
            self.__recordMap.pop()
            self.__recordMap.append( (oldPos, newFMT) )
            self.__recordMap.append( (self.__file.tell(), "") )

        record.modified(False)
        self.__recordIndex += 1
        self.__fsize = fstat(self.__file.fileno())[ST_SIZE]


    def writelines(self, sequence):
        """writes a sequence of records to the current position

        Records must be of class AMCRecord, otherwise TypeError is raised.
        """
        for rec in sequence:
            self.write(rec)



#-- testsuite ------------------------------------------------------------------
if (__name__ == "__main__"):
    f = AMCFile33("Filme.amc", "r+b")

    r = f.read(1)[0]


    
#-- EOF --
