# -*- coding: utf-8 -*-
# Author:   $Author: merkosh $
# Revision: $Rev: 407 $
############################################################################
#    Copyright (C) 2005 by Uwe Mayer                                       #
#    merkosh@hadiko.de                                                     #
#                                                                          #
#    This program is free software; you can redistribute it and/or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             #
############################################################################

#-- imports --------------------------------------------------------------------
#-- Qt imports --
from qt import QApplication, QObject, SLOT, SIGNAL, QProcess, QDir, PYSIGNAL
from qt import QFileDialog, QMessageBox, QInputDialog, QLineEdit, QPopupMenu
from qt import QIconSet

#-- Python imports --
import sys
import os
import logging
import time

from types import ListType
from os.path import dirname, basename, exists as fexists, samefile, \
     getsize as fsize, splitext, join as fjoin
from copy import deepcopy


#-- create logging and load preferences ----------------------------------------
log = logging.getLogger("")
hdlr = logging.StreamHandler()
formatter = logging.Formatter('[%(name)s] %(levelname)s: %(message)s')
hdlr.setFormatter(formatter)
log.addHandler(hdlr) 
log.setLevel(logging.DEBUG)             # default log level

# output program text-logo
log.info("moviefly  $Rev: 407 $ (c) by Uwe Mayer")


#-- own imports --
import Settings
import MediaInfo

from defaults import scriptdir, version

from LMCWidgetBase import LMCWidgetBase
from AMCFile33 import AMCFile33, AMCRecord33
from RecordList import RecordList
from misc import YMD2Date
from TitleMatchList import TitleMatchList
from RecordPreview import RecordPreview
from ScriptExecDialog import ScriptExecDialog
from ScriptSelection import ScriptSelection
from StdIOProcess import StdIOProcess
from FindDialog import FindDialog
from CatalogueProperties import CatalogueProperties
from ChangeNumberDialog import ChangeNumberDialog


# load settings
settings = Settings.Settings()
settings.load()
Settings.updatePreferences()

# load preferences
pref = Settings.getPreferences('__main__')

# adjust log-level
log.setLevel(pref['logLevel'])


#-- create proxy format for script environment
scriptProxy = None
if (pref['proxy'] != None):
    scriptProxy = {}
    for (k,v) in pref['proxy'].iteritems():
        scriptProxy["%s_proxy"%k] = v        


#-- used as filter in select movie to scan dialog
movieFileFilter = [u'Movies (*.avi *.ogm *.mpg *.mpeg)',
                   u'All (*.*)']


#-- available file format (with autoprobing)

ALL_FILES_FILTER = ";;All Files (*)"

# list of (<class>, <file filter> <prefered extension>)
# empty file filter will be ignored
fileFormats = [(AMCFile33, 'AMC v3.3 (*.amc)', '.amc'),
               ]

# convienience list constructed from the above (do not modify)
# list of all file filter which are not empty
fileFilterList = filter(lambda a: a != "", map(lambda a: a[1], fileFormats))        


# defines the start index in the file menu where the list of recent
# files is inserted
RECENT_FILES_START_INDEX = 13
MAX_COUNT_RECENT_FILES = 4


#-- utility functions ----------------------------------------------------------
defaultCaption = "Moviefly  v%s"%version
filenameCaption = lambda a: defaultCaption+" - %s"%basename(a)


#-- main class -----------------------------------------------------------------
class LMCMain(LMCWidgetBase):
    """main program class"""
    def __init__(self, input_file):
        LMCWidgetBase.__init__(self)

        # read in config file
        try:
            execfile("lmc.conf", pref)
        except IOError, e:
            pass                        # TODO: do abort

        # create logging
        self.log = logging.getLogger("LMCMain")

        # scan from file popup menu
        self.movieGetLocalPopup = QPopupMenu(self)
        self.movieGetLocalPopup.insertItem(QIconSet(self.image13), self.tr("from files..."), self.movieGetLocalFromFiles)
        self.movieGetLocalPopup.insertItem(QIconSet(self.image30), self.tr("from files on CD..."), self.movieGetLocalCDFromFiles)
        # disable actions
        self.movieGetLocalPopup.setItemEnabled(self.movieGetLocalPopup.idAt(0), True)
        self.movieGetLocalPopup.setItemEnabled(self.movieGetLocalPopup.idAt(1), False)
        # add popup menu
        self.toolGetFromFilesAction.setPopup(self.movieGetLocalPopup)

        # grouping popup menu
        self.groupPopup = QPopupMenu(self)
        self.groupPopup.insertItem(self.tr("no grouping"), self.movieGroup)
        self.groupPopup.insertSeparator()
        self.groupPopup.insertItem(self.tr("by rating"), self.movieGroup)
        self.groupPopup.insertItem(self.tr("by year"), self.movieGroup)
        self.groupPopup.insertItem(self.tr("by source"), self.movieGroup)
        self.groupPopup.insertItem(self.tr("by borrower"), self.movieGroup)
        self.groupPopup.insertItem(self.tr("by country"), self.movieGroup)
        self.groupPopup.insertItem(self.tr("by category"), self.movieGroup)
        self.groupPopup.insertItem(self.tr("by length"), self.movieGroup)
        self.groupPopup.insertItem(self.tr("by languages"), self.movieGroup)
        self.groupPopup.insertItem(self.tr("by date added"), self.movieGroup)
        
        # remember id->type
        self.groupType = dict()
        self.groupType[self.groupPopup.idAt(0)] = None
        self.groupType[self.groupPopup.idAt(2)] = 'rating'
        self.groupType[self.groupPopup.idAt(3)] = 'year'
        self.groupType[self.groupPopup.idAt(4)] = 'source'
        self.groupType[self.groupPopup.idAt(5)] = 'borrower'
        self.groupType[self.groupPopup.idAt(6)] = 'country'
        self.groupType[self.groupPopup.idAt(7)] = 'category'
        self.groupType[self.groupPopup.idAt(8)] = 'length'
        self.groupType[self.groupPopup.idAt(9)] = 'languages'
        self.groupType[self.groupPopup.idAt(10)] = 'date'
        # add popup menu
        self.toolGroupAction.setPopup(self.groupPopup)

        # populate recent-files-list
        self.recentFilesIndex = RECENT_FILES_START_INDEX
        self._updateRecentFiles()

        # runtime variables
        self.__file = None              # no file opened so far
        self.lastFolder = pref['defaultPath']
        self.lastMovieFolder = pref['defaultMoviePath']
        self.recordCopy = None
        self.modified = False           # modified metadata or deleted record
        self.recordActionsEnabled = False
        self.pictureActionsEnabled = False

        # set default caption
        self.setCaption(defaultCaption)

        # assign slots
        self.connect(self.fileOpenAction, SIGNAL("activated()"), self.fileOpen)
        self.fileOpenAction.setEnabled(True)
        self.connect(self.toolOpenAction, SIGNAL("pressed()"), self.fileOpen)
        self.toolOpenAction.setEnabled(True)

        self.connect(self.fileNewAction, SIGNAL("activated()"), self.fileNew)
        self.fileNewAction.setEnabled(True)
        self.connect(self.toolNewAction, SIGNAL("pressed()"), self.fileNew)
        self.toolNewAction.setEnabled(True)

        # auto-open
        infile = input_file or pref['defaultFile']
        if (infile != None):
            self.log.info("Opening default file %s", basename(infile))
            if (not fexists(infile)):
                self.log.warning("Cannot open file %s; file does not exist", basename(infile))
            else:
                self.fileOpen(infile)



    #-- slot implementations ---------------------------------------------------
    # wrapper with unified maximum delay
    def updateStatusBar(self, message, delay=5000):
        """updates status bar"""
        self.statusBar().message(message, delay)


    def infoStatusBar(self, message, delay=1000):
        """displays short message in status bar"""
        self.statusBar().message(message, delay)


    def currentChanged(self, obj):
        """called when the current item is changed"""
        # this is used to connect and disconnect special actions which
        # are only available when a record is selected        
        # disconnect record-signals
        if (obj == None):
            if (self.recordActionsEnabled): 
                self._disableRecordActions()
            if (self.pictureActionsEnabled):
                self._disablePictureActions()
            
        # connect record-signals
        else:
            if (not self.recordActionsEnabled):
                self._enableRecordActions()

            # check wether to enable picture actions
            rec = self.centralWidget().getCurrentRecord()
            if ((rec['embeddedImage'] or rec['picture']) and not self.pictureActionsEnabled):
                self._enablePictureActions()

            elif (not (rec['embeddedImage'] or rec['picture']) and self.pictureActionsEnabled):
                self._disablePictureActions()


    def onModified(self):
        """called when the record was modified; currently: only picture"""
        # check wether to enable picture actions
        rec = self.centralWidget().getCurrentRecord()
        if ((rec['embeddedImage'] or rec['picture']) and not self.pictureActionsEnabled):
            self._enablePictureActions()

        elif (not (rec['embeddedImage'] or rec['picture']) and self.pictureActionsEnabled):
            self._disablePictureActions()
        

    def closeEvent(self, e):
        """called when application is closed: checks if file needs saving"""
        if (self.__file != None):
            self.fileClose()

        else:
            # keep track of lastFileOpened configuration setting
            if (pref['reopenLastFile']):
                settings['defaultFile'] = None
                Settings.updatePreferences()

        settings.save()
        self.log.info("Quitting")
        e.accept()
        

    def fileOpenRecent(self, id=None):
        """called when a recently opened file should be opened"""
        # update global settings object
        filename = settings['recentFiles'].pop(id -RECENT_FILES_START_INDEX)
        settings['recentFiles'].insert(0, filename)
        Settings.updatePreferences()
        # update file menu
        self._updateRecentFiles()
        # open file
        self.fileOpen(filename)


    def fileClose(self):
        """closes a file and removes the central widget"""
        self.log.info("Closing file %s", self.__file.name)
        
        # remove slots
        self.disconnect(self.centralWidget(), PYSIGNAL("currentChanged"), self.currentChanged)
        if (self.pictureActionsEnabled):
            self._disablePictureActions()
        if (self.recordActionsEnabled):
            self._disableRecordActions()
        self._disableFileActions()

        self._querySaveOnDirty()

        self.infoStatusBar("Closing file %s"%basename(self.__file.name))
        self.centralWidget().hide()
        self.setCentralWidget(None)
        self.__file.close()
        self.__file = None

        # set caption
        self.setCaption(defaultCaption)


    def fileSave(self):
        """saves modifications to the opened file"""
        # check for modifications
        modified = self.modified
        if (not modified):
            for rec in self.centralWidget().values():
                modified = rec.modified()
                if (modified): break    # early exit
            
        # skip saving if not modified
        if (not modified):
            self.log.info("File not modified; not saving")
            return

        self.log.info("Saving file %s", basename(self.__file.name))
        self.infoStatusBar("Saving file %s"%(basename(self.__file.name)))
        self.__file.truncate(0)
        # sort records ascendingly on 'number'
        records = self.centralWidget().values()
        if (not pref['recordStoreOrderRandom']):
            records.sort( lambda a,b: int(a['number']) -int(b['number']) )
        self.__file.writelines(records)

        # clear internal flag
        self.modified = False

        
    def fileSaveAs(self):
        """handles save-as (new file) process"""
        # create file dialog and get filename
        dlg = QFileDialog(self.lastFolder, ";;".join(fileFilterList), self)
        dlg.setCaption(self.tr("Save as new file"))
        dlg.setViewMode(QFileDialog.Detail)
        dlg.setMode(QFileDialog.AnyFile)
        
        if (dlg.exec_loop() == QFileDialog.Rejected): return
        filename = unicode(dlg.selectedFile())

        # get format class and filename
        (cls, filename) = self._getFormatByFilename(filename, dlg.selectedFilter())

        # test if saving as already opened file
        if (fexists(filename) and samefile(filename, self.__file.name)):
            self.log.debug("Save-as to already opened file; using fileSave()")
            self.fileSave()
            return

        # test if file already exists
        if (fexists(filename)):
            ans = QMessageBox.question(self, self.tr("File already exists"),
                                       self.tr("File already exists. Overwrite?"),
                                       QMessageBox.Yes, QMessageBox.No)
            if (ans == QMessageBox.No): return           

        # finally save file
        self.infoStatusBar("Saving as new file %s"%basename(filename))
        try:
            # create new file
            f = cls(filename, "wb")
            # copy header fields
            for k in self.__file.keys():
                f[k] = self.__file[k]
            # save body
            records = self.centralWidget().values()
            f.writelines(records)
        except IOError, e:
            QMessageBox.warning(self, self.tr("Error saving file"),
                                self.tr("Could not save file: ")+unicode(e),
                                self.tr("Ok"))
            return

        # swap currently opened file
        self.__file.close()
        self.lastFolder = dirname(filename) # remember last folder
        self.log.info("Saving to file %s", basename(filename))
        self.__file = f

        # set caption
        self.setCaption(filenameCaption(self.__file.name))

        # clear internal flag
        self.modified = False


    def fileNew(self):
        """handles the create-new-file process"""
        self._querySaveOnDirty()
        
        # create file dialog and get filename
        dlg = QFileDialog(self.lastFolder, ";;".join(fileFilterList), self)
        dlg.setCaption(self.tr("Create new file"))
        dlg.setViewMode(QFileDialog.Detail)
        dlg.setMode(QFileDialog.AnyFile)
        
        if (dlg.exec_loop() == QFileDialog.Rejected): return
        filename = unicode(dlg.selectedFile())

        # get format class and filename
        (cls, filename) = self._getFormatByFilename(filename, dlg.selectedFilter())

        # test wether file already exists
        if (fexists(filename)):
            ans = QMessageBox.question(self, self.tr("File already exists"),
                                       self.tr("File already exis. Overwrite?"),
                                       QMessageBox.Yes, QMessageBox.No)
            if (ans == QMessageBox.No): return

        # try opening read-write access
        self.infoStatusBar("Creating new file %s"%basename(filename))
        try: 
            f = cls(filename, "w+b")
        except IOError, e:
            QMessageBox.warning(self, self.tr("Error opening file"),
                                self.tr("Could not open file: ")+unicode(e),
                                self.tr("Ok"))
            return

        # close currently opened file
        if (self.__file != None):
            self.fileClose()

        # now remember newly opened file
        self.lastFolder = dirname(filename) # remember last folder
        self.log.info("Creating new file %s", basename(filename))
        self.__file = f

        # keep track of lastFileOpened configuration setting
        if (pref['reopenLastFile']):
            settings['defaultFile'] = filename
            Settings.updatePreferences()

        # keep track of recent files opened
        # update global settings for recentFiles
        settings['recentFiles'] = filter(lambda a: a != filename, pref['recentFiles'])
        if (len(settings['recentFiles']) >= MAX_COUNT_RECENT_FILES):
            settings['recentFiles'].pop()
        settings['recentFiles'].insert(0, filename)
        Settings.updatePreferences()
        # update file menu
        self._updateRecentFiles()

        # set caption
        self.setCaption(filenameCaption(self.__file.name))

        # instanciate and new, empty center widget
        self.setCentralWidget( RecordList(dirname(filename), self) )
        self.centralWidget().show()

        # connect slots
        self._enableFileActions()
        self.connect(self.centralWidget(), PYSIGNAL("currentChanged"), self.currentChanged)
        


    def fileOpen(self, defaultFile=None):
        """handles the open-existing-file process"""
        # query save when records dirty
        self._querySaveOnDirty()

        # no default open given: load from dialog
        if (defaultFile == None):
            # create file dialog and get filename
            dlg = QFileDialog(self.lastFolder, u";;".join(fileFilterList)+ALL_FILES_FILTER, self)
            dlg.setCaption(self.tr("Open a new file"))
            dlg.setViewMode(QFileDialog.Detail)
        
            if (dlg.exec_loop() == QFileDialog.Rejected): return
            filename = unicode(dlg.selectedFile())
        else:
            filename = defaultFile

        # autoprobe file formats (they raise TypeError on mismatch)
        self.infoStatusBar("Opening file %s"%basename(filename))
        f = None
        for (cls, _, _) in fileFormats:
            self.log.debug("Autoprobing file format")
            # try opening read-write access
            try: 
                f = cls(filename, "a+b")
            except IOError, e:
                # try opening read-only access
                try:
                    f = cls(filename, "rb")
                except IOError, e:
                    QMessageBox.warning(self, self.tr("Error opening file"),
                                        self.tr("Could not open file: ")+unicode(e),
                                        self.tr("Ok"))
                    return
                else:
                    QMessageBox.warning(self, self.tr("Error opening file"),
                                        self.tr("Could not open file for write access.\n Opening read-only."),
                                        self.tr("Ok"))
            # class does not support this file format
            except TypeError: pass

        # test if file format was found
        if (f == None):
            self.log.error("Cannot read file format: %s", filename)
            return
        else:
            self.log.info("Detected file format: %s", cls)

        # close currently opened file
        if (self.__file != None):
            self.fileClose()
            
        # now remember newly opened file
        self.lastFolder = dirname(filename) # remember last folder
        self.log.info("Opening file %s (size: %d)", basename(filename), fsize(filename))
        self.__file = f

        # keep track of lastFileOpened configuration setting
        if (pref['reopenLastFile']):
            settings['defaultFile'] = filename
            Settings.updatePreferences()

        # keep track of recent files opened
        # update global settings for recentFiles
        settings['recentFiles'] = filter(lambda a: a != filename, pref['recentFiles'])
        if (len(settings['recentFiles']) >= MAX_COUNT_RECENT_FILES):
            settings['recentFiles'].pop()
        settings['recentFiles'].insert(0, filename)
        Settings.updatePreferences()
        # update file menu
        self._updateRecentFiles()

        # set caption
        self.setCaption(filenameCaption(self.__file.name))

        # instanciate and load new center widget
        self.setCentralWidget( RecordList(self.lastFolder, self) )
        self.centralWidget().add( self.__file.read() )

        # connect currentChanged signal now so that focusFirst causes
        # picture and record actions to be enabled
        self.connect(self.centralWidget(), PYSIGNAL("currentChanged"), self.currentChanged)

        self.centralWidget().focusFirst()
        self.centralWidget().show()

        # enable rest of the actions
        self._enableFileActions()


    def fileProperties(self):
        """display catalogue properties"""
        dlg = CatalogueProperties(self.__file)
        if ((dlg.exec_loop() == dlg.Accepted) and dlg.modified):
            self.log.info("Saving file meta-data")
            self.infoStatusBar("Saving file meta-data")
            self.__file.flush()
    

    def movieAdd(self):
        """adds a new movie record"""
        # scan for the first available index number
        indexList = []
        for record in self.centralWidget().values():
            indexList.append(int(record['number']))

        index = 1
        indexList.sort()
        for i in indexList:
            if (i < index): continue
            elif (i == index): index += 1
            else: break

        # append empty record
        self.infoStatusBar("Adding new record %d"%index)
        dateVal = YMD2Date(*time.localtime()[0:3])
        self.log.debug("adding new record at index %d", index)
        new = deepcopy(pref['recordDefault'])
        new.update({'number': index, 'date': dateVal})
        # create AMCRecord
        record = AMCRecord33( new )
        record.modified(True)
        self.centralWidget().add(record)
        self.centralWidget().focusNew()

        # info: connecting record-actions not necessary; they will be
        # connected as part of the "current-changed" signal
            
        # do autoScanOnNew
        if (pref['autoScanOnNew']):
            self.movieGetLocalFromFiles()
        # do autoScanAutoMountOnNew
        if (pref['autoScanAutoMountOnNew']):
            if (pref['autoMountPoint'] == None):
                self.log.warning("Cannot auto-mount removeable device: none specified")
            else:
                self.movieGetLocalCDFromFiles()
        # do autoScriptOnNew
        if (pref['autoScriptOnNew']):
            self.movieGetLocalFromScript()


    def movieDelete(self):
        """deletes a record"""
        record = self.centralWidget().getCurrentRecord()
        self.infoStatusBar("Deleting record %d"%record['number'])
        del self.centralWidget()[self.centralWidget().listViewRecord.currentItem()]

        self.modified = True


    def moviePictureDelete(self):
        """delete picture of current record"""
        # delegate to central widget
        self.centralWidget().deletePicture()


    def moviePictureCopy(self):
        """copies current picture"""
        self.centralWidget().copyPicture()
        

    def moviePictureSaveAs(self):
        """copies current picture"""
        self.centralWidget().saveAsPicture()
        

    def moviePictureLink(self):
        """called when linking a picture (vs. embedding it)"""
        self.centralWidget().linkPicture()


    def moviePictureEmbed(self):
        """called when embedding a picture (vs. linking it)"""
        self.centralWidget().embedPicture()
    

    def movieGetLocalFromFiles(self):
        """get information from a file"""
        # select filenames
        filenames = QFileDialog.getOpenFileNames(
            ";;".join(movieFileFilter), self.lastMovieFolder, self,
            "scan file dialog", "Select file to scan")
        if (filenames == None): return
        filenames = map(lambda a: unicode(a.ascii(), pref['systemEncoding']), filenames)

        # quit if nothing selected
        if (len(filenames) == 0): return

        # remember last directory
        self.lastMovieFolder = dirname(filenames[0])

        # get media info
        self.log.info("Getting file information for %s using %s", basename(filenames[0]), pref['mediaInfo'])
        self.infoStatusBar("Getting file information from %s"%basename(filenames[0]))
        info = pref['mediaInfo'](filenames[0])
        info.scan()

        record = self.centralWidget().getCurrentRecord()
        record['originalTitle'] = info.originalTitle
        record['videoFormat'] = info.videoFormat
        record['audioFormat'] = info.audioFormat
        record['resolution'] = info.resolution
        record['subtitles'] = info.subtitles
        record['videoBitrate'] = info.videoBitrate
        record['audioBitrate'] = info.audioBitrate
        record['framerate'] = info.framerate
        if (record['fileSize'] != ""):
            record['fileSize'] = "+".join([record['fileSize'], info.getFileSize(MediaInfo.MB)])
        else:
            record['fileSize'] = info.getFileSize(MediaInfo.MB)
        record['length'] = info.length
        record['discs'] += 1

        # get detailed info on first file; for all others just adjust
        # filesize
        for filename in filenames[1:]:
            record['fileSize'] += "+%s"%info.normalizeFileSize(fsize(filename), MediaInfo.MB)
        
        self.centralWidget().setCurrentRecord(record)

        # focus first field on AMCForm
        self.centralWidget().form.focus("mediaLabel")


    def movieGetLocalCDFromFiles(self):
        """mount /unmount when getting information from a file"""
        # mount target volume
        self.log.info("Automounting %s", pref['autoMountPoint'])
        ret = os.system("mount %s"%pref['autoMountPoint'])
        
        # remember usual default target path
        tmp = self.lastMovieFolder
        self.lastMovieFolder = pref['autoMountPoint']
        # get info
        self.movieGetLocalFromFiles()
               
        # restore target path
        self.lastMovieFolder = tmp

        # unmount or eject
        if (pref['ejectOnUnmount']):
            if (ret != 0):
                self.log.info("mount exited with non-zero exit code; not ejecting")
            else:
                self.log.info("Ejecting %s", pref['autoMountPoint'])
                os.system("eject %s &"%pref['autoMountPoint'])
        else:
            if (ret != 0):
                self.log.info("mount exited with non-zero exit code; not unmounting")
            else:
                self.log.info("Unmounting %s", pref['autoMountPoint'])
                os.system("umount %s &"%pref['autoMountPoint'])
       

    def movieGetLocalFromScript(self):
        """get information from a script"""
        # get script to execute
        scriptDlg = ScriptSelection(self, modal=True)
        # if process was aborted, stop
        if (scriptDlg.exec_loop() == scriptDlg.Rejected):
            self.log.info("Aborting search")
            return

        # get selected script
        scriptList = scriptDlg.selectedItem()
        # if nothing selected
        if (scriptList == None):
            self.log.info("No script selected; aborting")
            return

        # get title or current record          
        record = self.centralWidget().getCurrentRecord()
        (searchTitle, ok) = QInputDialog.getText(
            self.tr("search for title"),
            self.tr("Please enter the title to search for:"),
            QLineEdit.Normal,
            record['originalTitle'] or record['translatedTitle'] or record['mediaLabel'],
            self)
        if (not ok): return
        searchTitle = unicode(searchTitle)

        # empty search title
        if (not searchTitle):
            self.log.warning("Input dialog returned empty search string; aborting")
            return
        # shell quote title
        searchTitle = searchTitle.replace('"', '\\"')

        # get details on all scripts
        summary = ""
        self.infoStatusBar("Getting information from script on %s"%searchTitle)
        for script in scriptList:
            self.log.info("Getting title list from %s", script[0])

            # execute script for title matches
            titleSearch = StdIOProcess(fjoin(".",script[0]), [searchTitle.encode(pref['systemEncoding'])], scriptProxy, scriptdir)
            execDlg = ScriptExecDialog(titleSearch, modal=True)

            # if process was aborted, stop
            if (execDlg.exec_loop() == execDlg.Rejected):
                self.log.info("Aborting search for title matches")
                continue
        
            # process child error output
            if (titleSearch.stderr() != ""):
                log = logging.getLogger(script[0])
                log.error(titleSearch.stderr())
                return
            
            # remember list of title matches or details
            # (whatever was returned)
            stdoutText = titleSearch.stdout()

            # parse output from search for matched titles
            titleMatchDlg = TitleMatchList(self)
            status = titleMatchDlg.parseScriptInput(stdoutText)
            if (status == 'failed'): return
            if (status == 'empty'):
                self.log.info("No titles found; skipping")
                continue

            # set URL to default - in case no url needed (direct hit)
            url = None                  # url to get details from

            # check return status - wether to display title matches
            if (status == 'list'):
                # display title matches
                if (titleMatchDlg.exec_loop() == titleMatchDlg.Rejected):
                    self.log.info("Abort choosing title match")
                    continue

                # get url for details
                url = titleMatchDlg.selectedItem()

                # fetch details
                detailSearch = StdIOProcess(fjoin(".",script[0]), [url.encode(pref['systemEncoding'])], scriptProxy, scriptdir)
                execDlg2 = ScriptExecDialog(detailSearch, modal=True)
                
                # if process was aborted, stop
                if (execDlg2.exec_loop() == execDlg2.Rejected):
                    self.log.info("Abort search for title details")
                    continue

                # process child error output
                if (detailSearch.stderr() != ""):
                    log = logging.getLogger(script[0])
                    log.error(detailSearch.stderr())
                    return
                
                # remember details
                stdoutText = detailSearch.stdout()

            # exclude invalid fields
            stdoutLines = filter(lambda a: a[:a.find(':')] in script[1:], stdoutText.split("\n"))

            # gather details from all scripts
            summary += "\n"+"\n".join(stdoutLines)
            
        # display preliminary details
        previewDlg = RecordPreview(self, modal=True)
        status = previewDlg.parseScriptInput(summary)

        if (status == 'empty'):
            self.log.info("Empty details dialog; skipping")
            return
        if ((status == "failed") or (previewDlg.exec_loop() == previewDlg.Rejected)):
            self.log.info("Abort setting title details")
            return

        # set selected record details
        new = previewDlg.getSelected()
        for f in new.keys():
            record[f] = new[f]
        self.centralWidget().setCurrentRecord(record) # display new values
        if (new.has_key('picture')): 
            self.centralWidget().defaultImageDecode()
                    
        # focus first field on AMCForm
        self.centralWidget().form.focus("mediaLabel")
        

    def movieCopy(self):
        """copy current record"""
        record = self.centralWidget().getCurrentRecord()
        self.log.info("Copying record %s to clipboard", unicode(record['number']))
        self.infoStatusBar("Copying record %s to clipboard"%unicode(record['number']))
        self.recordCopy = record        # copy reference to original

        if (not self.moviePasteAction.isEnabled()):
            self.connect(self.moviePasteAction, SIGNAL("activated()"), self.moviePaste)
            self.moviePasteAction.setEnabled(True)

            self.connect(self.toolPasteAction, SIGNAL("pressed()"), self.moviePaste)
            self.toolPasteAction.setEnabled(True)
        

    def moviePaste(self):
        """paste over current record"""
        record = self.centralWidget().getCurrentRecord()

        # if copied record == current record: paste as new
        if (record == self.recordCopy):
            self.log.info("Pasting to new record")
            # prevent autoScanOnNew, autoScanAutoMountOnNew and autoScriptOnNew
            a, b, c = pref['autoScanOnNew'], pref['autoScanAutoMountOnNew'], pref['autoScriptOnNew']
            pref['autoScanOnNew'], pref['autoScanAutoMountOnNew'], pref['autoScriptOnNew'] = False, False, False
            self.movieAdd()
            record = self.centralWidget().getCurrentRecord()
            pref['autoScanOnNew'], pref['autoScanAutoMountOnNew'], pref['autoScriptOnNew'] = a, b, c
        
        self.log.info("Pasting to record %s from clipboard", unicode(record['number']))
        self.infoStatusBar("Pasting to record %s from clipboard"%unicode(record['number']), 2000)

        originalNumber = record['number']
        record.update(deepcopy(self.recordCopy))
        record['number'] = originalNumber

        self.centralWidget().setCurrentRecord(record)
        
        if (pref['deepcopyOnCopy']):
            self.log.debug("Creating deepcopy of image")
            self.centralWidget().defaultImageDecode()
            

    def movieFind(self):
        """find a movie according to field value"""

        def findNext(text, field, options):
            self.centralWidget().findNext(text, field, options)
            
        findDlg = FindDialog(self)
        self.connect(findDlg, PYSIGNAL("findValue"), findNext)
        findDlg.exec_loop()
        findNext(None, None, None)
        self.disconnect(findDlg, PYSIGNAL("findValue"), findNext)


    def movieGroup(self, groupId=None):
        """called when an online re-grouping is necessary"""
        # get *what* to group by
        if (groupId != None):
            groupId = self.groupType[groupId]

        # call responsible function
        self.centralWidget().onGroup(groupId)
        

    def movieChangeNumber(self):
        """called when the number of a record is to be changed"""
        # display change number dialog
        dlg = ChangeNumberDialog(modal=True)
        if (dlg.exec_loop() == dlg.Rejected):
            return

        # get result
        mode = dlg.getState()
        number = dlg.getNumber()
        # change number
        self.centralWidget().changeNumber(mode, number)
        


    #-- helper methods ---------------------------------------------------------
    def _updateRecentFiles(self):
        # remove recent files in the menu
        while (self.recentFilesIndex > RECENT_FILES_START_INDEX):
            self.recentFilesIndex -= 1
            self.fileMenu.removeItem(self.recentFilesIndex)

        # remove recent files which don't exist
        for i in range(len(settings['recentFiles'])-1, -1, -1):
            if (not fexists(settings['recentFiles'][i])):
                self.log.info("Removing recent file %s: file does not exist", basename(settings['recentFiles'][i]))
                del settings['recentFiles'][i]

        # add to menu
        for f in settings['recentFiles']:
            if (f[:len(os.environ['HOME'])] == os.environ['HOME']):
                f = f.replace(os.environ['HOME'], '~')
            if (len(f) > 15+3+15):
                f = f[:15]+"..."+f[-15:]
            self.fileMenu.insertItem(f, self.fileOpenRecent, 0, self.recentFilesIndex, self.recentFilesIndex)
            self.recentFilesIndex += 1
    

    def _querySaveOnDirty(self):
        """asks user wether to save file when there are dirty records"""
        # check if currently opened records were modified
        if (self.__file != None):
            # test for modified records
            modified = self.modified
            if (not modified):
                for rec in self.centralWidget().values():                
                    modified = rec.modified()
                    if (modified): break    # early exit

            # offer to save changes
            if (modified):
                ans = QMessageBox.question(self, self.tr("Save changes?"),
                                           self.tr("File was modified - save changes?"),
                                           QMessageBox.Yes, QMessageBox.No)
                # save changes
                if (ans == QMessageBox.Yes):
                    self.fileSave()
        # nothing to do


    # used by SaveAs and New to decide on final file format and filename
    def _getFormatByFilename(self, filename, selectedFilter):
        """returns file format class and final filename from filename and selectedFilter"""
        cls = None                      # takes up target class
        # 1. extension
        (base, ext) = splitext(filename)
        if (ext != ""):
            # search for possible extensions
            possible = filter(lambda a: a[2] == ext, fileFormats)
            if (len(possible) == 0):
                self.log.warning("File extension was specified, but no fitting file format found")
                # continue with selected filter
            else:
                cls = possible[0][0]
                self.log.info("Saving file in %s format", cls)
                
        # 2. used file filter
        if (cls == None):
            # get list of possible classes
            possible = filter(lambda a: a[1] == selectedFilter, fileFormats)
            if (len(possible) == 0):
                self.log.warning("Cannot determine file format by extension or file filter. Using default: %s", str(fileFormats[0][0]))
                cls = fileFormats[0][0]
                filename = base+fileFormats[0][2] # set prefered extension
            else:
                cls = possible[0][0]
                filename = base+possible[0][2] # append prefered extenstion

        self.log.info("Using file format: %s", cls)
        return (cls, filename)
        
        

    def _enableFileActions(self):
        """enables Save, SaveAs and Close actions"""
        self.log.debug("Enabling file actions")
        
        self.connect(self.fileCloseAction, SIGNAL("activated()"), self.fileClose)
        self.fileCloseAction.setEnabled(True)
        
        self.connect(self.fileSaveAction, SIGNAL("activated()"), self.fileSave)
        self.fileSaveAction.setEnabled(True)

        self.connect(self.toolSaveAction, SIGNAL("pressed()"), self.fileSave)
        self.toolSaveAction.setEnabled(True)

        self.connect(self.fileSaveAsAction, SIGNAL("activated()"), self.fileSaveAs)
        self.fileSaveAsAction.setEnabled(True)

        self.connect(self.movieNewAction, SIGNAL("activated()"), self.movieAdd)
        self.movieNewAction.setEnabled(True)

        self.connect(self.toolAddAction, SIGNAL("pressed()"), self.movieAdd)
        self.toolAddAction.setEnabled(True)

        self.connect(self.filePropertiesAction, SIGNAL("activated()"), self.fileProperties)
        self.filePropertiesAction.setEnabled(True)

        self.connect(self.movieFindAction, SIGNAL("activated()"), self.movieFind)
        self.movieFindAction.setEnabled(True)

        self.connect(self.toolFindAction, SIGNAL("pressed()"), self.movieFind)
        self.toolFindAction.setEnabled(True)

        self.connect(self.toolGroupAction, SIGNAL("clicked()"), self.movieGroup)
        self.toolGroupAction.setEnabled(True)


    def _disableFileActions(self):
        """disables Save, SaveAs and Close actions"""
        self.log.debug("Disabling file actions")
        
        self.fileCloseAction.setEnabled(False)
        self.disconnect(self.fileCloseAction, SIGNAL("activated()"), self.fileClose)

        self.fileSaveAction.setEnabled(False)
        self.disconnect(self.fileSaveAction, SIGNAL("activated()"), self.fileSave)

        self.toolSaveAction.setEnabled(False)
        self.disconnect(self.toolSaveAction, SIGNAL("pressed()"), self.fileSave)

        self.fileSaveAsAction.setEnabled(False)
        self.disconnect(self.fileSaveAsAction, SIGNAL("activated()"), self.fileSaveAs)
        
        self.movieNewAction.setEnabled(False)
        self.disconnect(self.movieNewAction, SIGNAL("activated()"), self.movieAdd)

        self.toolAddAction.setEnabled(False)
        self.disconnect(self.toolAddAction, SIGNAL("pressed()"), self.movieAdd)

        if (self.moviePasteAction.isEnabled()):
            self.recordCopy = None
            
            self.moviePasteAction.setEnabled(False)
            self.disconnect(self.moviePasteAction, SIGNAL("activated()"), self.moviePaste)

            self.toolPasteAction.setEnabled(False)
            self.disconnect(self.toolPasteAction, SIGNAL("pressed()"), self.moviePaste)

        self.filePropertiesAction.setEnabled(False)
        self.disconnect(self.filePropertiesAction, SIGNAL("activated()"), self.fileProperties)

        self.movieFindAction.setEnabled(False)
        self.disconnect(self.movieFindAction, SIGNAL("activated()"), self.movieFind)

        self.toolFindAction.setEnabled(False)
        self.disconnect(self.toolFindAction, SIGNAL("pressed()"), self.movieFind)

        self.toolGroupAction.setEnabled(False)
        self.disconnect(self.toolGroupAction, SIGNAL("clicked()"), self.movieGroup)


    def _enableRecordActions(self):
        """enables actions related to deleting a record"""
        self.log.debug("Enabling record actions")
        assert(self.recordActionsEnabled == False)

        self.connect(self.movieDeleteAction, SIGNAL("activated()"), self.movieDelete)
        self.movieDeleteAction.setEnabled(True)

        self.connect(self.toolDeleteAction, SIGNAL("pressed()"), self.movieDelete)
        self.toolDeleteAction.setEnabled(True)

        self.connect(self.movieGetLocalFromFilesAction, SIGNAL("activated()"), self.movieGetLocalFromFiles)
        self.movieGetLocalFromFilesAction.setEnabled(True)

        self.connect(self.toolGetFromFilesAction, SIGNAL("clicked()"), self.movieGetLocalFromFiles)
        self.toolGetFromFilesAction.setEnabled(True)

        if (pref['autoMountPoint'] != None):
            self.connect(self.movieGetLocalCDFromFilesAction, SIGNAL("activated()"), self.movieGetLocalCDFromFiles)
            self.movieGetLocalCDFromFilesAction.setEnabled(True)

            self.movieGetLocalPopup.setItemEnabled(self.movieGetLocalPopup.idAt(1), True)


        self.connect(self.movieGetLocalFromScriptAction, SIGNAL("activated()"), self.movieGetLocalFromScript)
        self.movieGetLocalFromScriptAction.setEnabled(True)

        self.connect(self.toolGetFromScriptAction, SIGNAL("pressed()"), self.movieGetLocalFromScript)
        self.toolGetFromScriptAction.setEnabled(True)

        self.connect(self.centralWidget(), PYSIGNAL("searchTextChanged"), self.updateStatusBar)
        self.connect(self.centralWidget(), PYSIGNAL("modified"), self.onModified)

        self.connect(self.movieCopyAction, SIGNAL("activated()"), self.movieCopy)
        self.movieCopyAction.setEnabled(True)

        self.connect(self.toolCopyAction, SIGNAL("pressed()"), self.movieCopy)
        self.toolCopyAction.setEnabled(True)

        self.connect(self.movieChangeNumberAction, SIGNAL("activated()"), self.movieChangeNumber)
        self.movieChangeNumberAction.setEnabled(True)

        self.connect(self.moviePictureLinkAction, SIGNAL("activated()"), self.moviePictureLink)
        self.moviePictureLinkAction.setEnabled(True)
        
        self.connect(self.moviePictureEmbedAction, SIGNAL("activated()"), self.moviePictureEmbed)
        self.moviePictureEmbedAction.setEnabled(True)
        
        self.recordActionsEnabled = True
        


    def _disableRecordActions(self):
        """disables actions related to deleting a record"""
        self.log.debug("Disabling record actions")
        assert(self.recordActionsEnabled == True)

        self.movieDeleteAction.setEnabled(False)
        self.disconnect(self.movieDeleteAction, SIGNAL("activated()"), self.movieDelete)

        self.toolDeleteAction.setEnabled(False)
        self.disconnect(self.toolDeleteAction, SIGNAL("pressed()"), self.movieDelete)

        self.movieGetLocalFromFilesAction.setEnabled(False)
        self.disconnect(self.movieGetLocalFromFilesAction, SIGNAL("activated()"), self.movieGetLocalFromFiles)

        self.toolGetFromFilesAction.setEnabled(False)
        self.disconnect(self.toolGetFromFilesAction, SIGNAL("clicked()"), self.movieGetLocalFromFiles)

        if (pref['autoMountPoint'] != None):
            self.movieGetLocalCDFromFilesAction.setEnabled(False)
            self.disconnect(self.movieGetLocalCDFromFilesAction, SIGNAL("activated()"), self.movieGetLocalCDFromFiles)
            
            self.movieGetLocalPopup.setItemEnabled(self.movieGetLocalPopup.idAt(1), False)

        self.movieGetLocalFromScriptAction.setEnabled(False)
        self.disconnect(self.movieGetLocalFromScriptAction, SIGNAL("activated()"), self.movieGetLocalFromScript)

        self.toolGetFromScriptAction.setEnabled(False)
        self.disconnect(self.toolGetFromScriptAction, SIGNAL("pressed()"), self.movieGetLocalFromScript)

        self.disconnect(self.centralWidget(), PYSIGNAL("searchTextChanged"), self.updateStatusBar)
        self.disconnect(self.centralWidget(), PYSIGNAL("modified"), self.onModified)

        self.movieCopyAction.setEnabled(False)
        self.disconnect(self.movieCopyAction, SIGNAL("activated()"), self.movieCopy)

        self.toolCopyAction.setEnabled(False)
        self.disconnect(self.toolCopyAction, SIGNAL("pressed()"), self.movieCopy)

        self.movieChangeNumberAction.setEnabled(False)
        self.disconnect(self.movieChangeNumberAction, SIGNAL("activated()"), self.movieChangeNumber)

        self.moviePictureLinkAction.setEnabled(False)     
        self.disconnect(self.moviePictureLinkAction, SIGNAL("activated()"), self.moviePictureLink)
        
        self.moviePictureEmbedAction.setEnabled(False)
        self.disconnect(self.moviePictureEmbedAction, SIGNAL("activated()"), self.moviePictureEmbed)

        self.recordActionsEnabled = False


    def _enablePictureActions(self):
        """enables actions on a picture"""
        self.log.debug("Enabling picture actions")
        assert(self.pictureActionsEnabled == False)

        self.connect(self.moviePictureDeleteAction, SIGNAL("activated()"), self.moviePictureDelete)
        self.moviePictureDeleteAction.setEnabled(True)

        self.connect(self.moviePictureCopyAction, SIGNAL("activated()"), self.moviePictureCopy)
        self.moviePictureCopyAction.setEnabled(True)

        self.connect(self.moviePictureSaveAsAction, SIGNAL("activated()"), self.moviePictureSaveAs)
        self.moviePictureSaveAsAction.setEnabled(True)

        self.pictureActionsEnabled = True


    def _disablePictureActions(self):
        """disables actions on a picture"""
        self.log.debug("Disabling picture actions")
        assert(self.pictureActionsEnabled == True)

        self.moviePictureDeleteAction.setEnabled(False)
        self.disconnect(self.moviePictureDeleteAction, SIGNAL("activated()"), self.moviePictureDelete)

        self.moviePictureCopyAction.setEnabled(False)
        self.disconnect(self.moviePictureCopyAction, SIGNAL("activated()"), self.moviePictureCopy)

        self.moviePictureSaveAsAction.setEnabled(False)
        self.disconnect(self.moviePictureSaveAsAction, SIGNAL("activated()"), self.moviePictureSaveAs)

        self.pictureActionsEnabled = False


#-- main program ---------------------------------------------------------------
app = QApplication(sys.argv)

#-- parse command line arguments
# help text
if (('-h' in app.argv()[1:]) or ('--help' in app.argv()[1:])):
    print """
Synopsis: lmc.py [-h|--help] [<input-file>]

   -h       --help       this screen
   -v       --version    show version and revision number and exit
   <input-file>          open this file on startup
                         This option overwrites the default startup file from the
                         program configuration.
"""
    sys.exit()

# version
elif (('-v' in app.argv()[1:]) or ('--version' in app.argv()[1:])):
    print "Moviefly  (c) by Uwe Mayer"
    print "Version:  %s"%version
    print "Revision: $Rev: 407 $"
    print 
    sys.exit()
    

# input file
infile = None
if (len(app.argv()) > 1):
    infile = app.argv()[1]


QObject.connect(app, SIGNAL("lastWindowClosed()"), app, SLOT("quit()"))

main = LMCMain(infile)

app.setMainWidget(main)
main.show()
app.exec_loop()

