/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: $

   Created 07/19/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/window/glx/glxtext.h
  \brief gl text output for glx.
*/

#include "leg/support/window/glx/glxtext.h"
#include "leg/support/utils/errors.h"

namespace leg
{
namespace support
{
namespace window
{
   internal::Display*		 GlxText::disp = 0;
   GLuint			 GlxText::list_start_id = 0;
   std::list<GlxText::FontPair*> GlxText::loaded_fonts;
   std::list<std::string*>	 GlxText::patterned_fonts;
   unsigned int			 GlxText::max_elements = 20;

   using support::utils::Error;
   using support::utils::Info;

   GlxText::GlxText()
   {
   }

   GlxText::~GlxText()
   {
      DeleteList();
   }

   void
   GlxText::Connect (internal::Display *d)
   {
      disp = d;
      if (!disp)
	 Error (  "Connection to a null display.",
		  "support::window::glx::GlxText::Connect (Display*)");
   }

   void
   GlxText::LoadFont (std::string font_name)
   {
      if (FontAlreadyLoaded (font_name)){
	 SelectFont (font_name);
	 return;
      }

      int 	  first = 0;
      int 	  last = 0;
      XFontStruct *font_info;

      font_info = XLoadQueryFont (disp, font_name.c_str());
      if (!font_info)
	 Error (  "Font is unknown.",
		  "support::window::glx::GlxText::LoadFont (string)");
      
      Font font = font_info->fid;
      first = font_info->min_char_or_byte2;
      last = font_info->max_char_or_byte2;
      
      list_start_id = glGenLists ((GLuint)last + 1);
      if (!list_start_id)
	 Error (  "Unable to allocate font lists, you may call it too earlier.",
		  "support::window::glx::GlxText::Connect (Display*)");

      int list_l = last - first + 1;      
      glXUseXFont (font, first, list_l, list_start_id + first);

      std::pair<std::string,XFontStruct*> *tmp1 = new std::pair<std::string,XFontStruct*> (font_name,font_info);
      std::pair<GLuint,GLuint> *tmp2 = new std::pair<GLuint,GLuint> (list_start_id,(GLuint)list_l);
      FontPair *tmpfont = new FontPair (tmp1,tmp2);
      loaded_fonts.push_back (tmpfont);
   }

   void
   GlxText::LoadMatchedFonts()
   {
      std::list<std::string*>::iterator i = patterned_fonts.begin();
      for (;i != patterned_fonts.end(); ++i)
	 LoadFont (**i);
   }

   void
   GlxText::UnloadFont (const std::string& font_name)
   {
      std::list<FontPair*>::iterator i = loaded_fonts.begin();

      for (;i != loaded_fonts.end(); ++i){
	 FontPair *tmp = *i;
	 if (tmp->first->first == font_name){
	    XUnloadFont (disp,tmp->first->second->fid);
	    loaded_fonts.remove (*i);
	 }
      }
   }

   void
   GlxText::Locate (int x, int y, int z)
   {
      glRasterPos3i (x,y,z);
   }

   void
   GlxText::Locate (float x, float y, float z)
   {
      glRasterPos3f (x,y,z);
   }

   void
   GlxText::Locate (double x, double y, double z)
   {
      glRasterPos3d (x,y,z);
   }

   void
   GlxText::Print (const std::string& str)
   {
      glListBase (list_start_id);
      glCallLists (str.length(), GL_UNSIGNED_BYTE, (GLubyte*) (str.c_str()));
   }

   bool
   GlxText::FontAlreadyLoaded (const std::string& str)
   {
      std::list<FontPair*>::iterator i = loaded_fonts.begin();

      for (;i != loaded_fonts.end(); ++i)
	 if ((*i)->first->first == str)
	    return true;

      return false;
   }

   void
   GlxText::DeleteList()
   {
      std::list<FontPair*>::iterator i = loaded_fonts.begin();

      for (;i != loaded_fonts.end(); ++i){
	 FontPair *tmp = *i;
	 glDeleteLists (tmp->second->first,tmp->second->second);
	 delete tmp->second;
	 tmp->second = 0;
	 XUnloadFont (disp,tmp->first->second->fid);
	 delete tmp;
	 tmp = 0;
	 loaded_fonts.remove (*i);
      }
   }

   void
   GlxText::SelectFont (const std::string& font_name)
   {
      std::list<FontPair*>::iterator i = loaded_fonts.begin();

      for (;i != loaded_fonts.end(); ++i)
	 if ((*i)->first->first == font_name)
	    list_start_id = (*i)->second->first;
   }

   bool
   GlxText::CheckFontsAvailability (const std::string& pattern)
   {
      int nb = 0;
      char** list = XListFonts (disp,pattern.c_str(),max_elements,&nb);

      if (!nb || !list){
	 std::list<std::string*>::iterator i = patterned_fonts.begin();
	 for (;i != patterned_fonts.end(); ++i){
	    std::string* tmp = *i;
	    delete tmp;
	    tmp = 0;
	    patterned_fonts.remove (*i);
	 }
	 return false;
      }

      unsigned int i = 0;
      for (;i < static_cast<unsigned int> (nb);++i,++*list)
	 patterned_fonts.push_back (new std::string (*list));

      // TODO: freeing memory issue.
      // This provokes a segmentation error, and I don't understand why it
      // seems too earlier to free it: after all, we have copied them into
      // dynamically allocated strings... But it appears X needs them after...
      // So we haven't got any means to free them at the end neither...
//      XFreeFontNames (&this->list[0]);
      
      return true;
   }

   GlxText::FontIterator
   GlxText::GetFontIterator()
   {
      return patterned_fonts.begin();
   }

   unsigned int
   GlxText::GetPatternedFontsSize()
   {
      return patterned_fonts.size();
   }

   void
   GlxText::SetMaxElementsInPatternedFonts (unsigned int max)
   {
      max_elements = max;
   }

   unsigned int
   GlxText::GetMaxElementsInPatternedFonts()
   {
      return max_elements;
   }
}
}
}

