/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: glx12window.cc,v 1.6 2004/06/15 12:57:37 jd Exp $

   Created 03/16/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/window/glx12window.cc
  \brief XFree glx 1.2 compatibility window impl.
*/

#include "leg/support/window/glx/glx12window.h"
#include "leg/support/utils/errors.h"

namespace leg
{
namespace support
{
namespace window
{

using leg::support::utils::Warning;
using leg::support::utils::Error;
using leg::support::utils::Info;

GLX12Window::GLX12Window (DisplayConnect& disp): BaseGLXWindow (disp)
{
   drawable_ptr = malloc (sizeof (DrawableType));
   drawable_ptr = reinterpret_cast<DrawableType*> (drawable_ptr);
   drawable = new (drawable_ptr) DrawableType;
   drawable->SetDisplay (disp.Get());
   attr.SetGLX12Attributes();
}

GLX12Window::~GLX12Window()
{
   attr.SetKeyboardAutoRepeatMode (display.Get(), true);
   Destroy();
   drawable->~DrawableType();
   free (drawable_ptr);
}

void
GLX12Window::DoCreate()
{
   XSetWindowAttributes& xattr= *attr.GetXAttributes();
   int MajorVersion, MinorVersion;

   internal::Display *disp= drawable->GetDisplay();
   internal::Window root = RootWindow (disp, scr);

   MajorVersion = MinorVersion = 0;
   XF86VidModeQueryVersion (disp, &MajorVersion, &MinorVersion);

   int nb = 0;
   visual_info = drawable->GetVisual (scr, attr.GetGLWMAttributes(nb));

   int mode_num = -1;
   XF86VidModeGetAllModeLines (disp, scr, &mode_num, &modes);

   if (!attr.GetDecorationUse() || attr.GetFullScreenUse())
      xattr.override_redirect = True;
   
   unsigned int width = 0, height = 0;   
   attr.GetSize (width,height);
   if (attr.GetFullScreenUse()){
      XSync (drawable->GetDisplay(), 0);
      xattr.backing_store = NotUseful;
      xattr.save_under = False;
      
      int dot;
      XF86VidModeModeLine actual_mode;      
      XF86VidModeGetModeLine (disp, scr, &dot, &actual_mode);

      int i,dw,dh;
      for (i=0; i< mode_num; i++){
	 if ((modes[i]->hdisplay==width) && (modes[i]->vdisplay==height)){
	    best_mode= i;
	    break;
	 }
      }
 
      XF86VidModeSwitchToMode (disp, scr, modes[best_mode]);
      XF86VidModeSetViewPort (disp, scr, 0, 0);
			
      dw= modes[best_mode]->hdisplay;
      dh= modes[best_mode]->vdisplay;
   }

   xattr.background_pixel = 0;
   xattr.border_pixel = 0;
   xattr.colormap = XCreateColormap (disp, root, visual_info->visual, AllocNone);
   xattr.event_mask = KeyPressMask | KeyReleaseMask | ButtonPressMask | ButtonReleaseMask | 
		     PointerMotionMask | ButtonMotionMask | VisibilityChangeMask | StructureNotifyMask;

   drawable->GetWindow() = CreateXWindow();
   XMapWindow (disp, drawable->GetWindow());
   
   if (!attr.GetFullScreenUse())
      XSetStandardProperties (disp, 
			      drawable->GetWindow(), 
			      attr.GetName().c_str(), 
			      attr.GetName().c_str(), 
			      None,
			      0,
			      0,
			      0);
   else{
      XMoveWindow (disp, drawable->GetWindow(), 0, 0);
      XRaiseWindow (disp, drawable->GetWindow());
      XWarpPointer (disp, None, drawable->GetWindow(), 0, 0, 0, 0, width/2, height/2);
      XGrabKeyboard (disp, drawable->GetWindow(), False, GrabModeAsync, GrabModeAsync, CurrentTime);
   }
 
   XFlush (disp);

   context = drawable->CreateContext (0);
   drawable->EnterContext (context);

   is_created = true;
}

void 
GLX12Window::Create()
{
   GetGlxVersion();
   CreateScreen();
   DoCreate();
   drawable->CheckDirectRendering (drawable->GetDisplay(), context);
}
	
void 
GLX12Window::Destroy()
{
   internal::Display *disp= drawable->GetDisplay();
   if (!disp)
      Error ("No display connexion is opened.","leg::support::window::GLX12Window::Destroy()");
      
   if (this->context){
      drawable->LeaveContext();
      glXDestroyContext (drawable->GetDisplay(), this->context);
      this->context=0;
   }
   
   XDestroyWindow (drawable->GetDisplay(), drawable->GetWindow());
   drawable->GetWindow() = 0;
   
   if (attr.GetFullScreenUse())
      XF86VidModeSwitchToMode (drawable->GetDisplay(), scr, modes[0]);

   XFree (modes);
}

void 
GLX12Window::SetAttributes (XAttributes& attr)
{
   attr= attr;
}

XAttributes& GLX12Window::GetAttributes()
{
   return attr;
}

BaseDrawable&
GLX12Window::GetDrawable()
{
   return *drawable;
}

void
GLX12Window::ShowCursor (bool show)
{
   if (!show)
      XDefineCursor (display.Get(),drawable->GetWindow(),None);
   else
      Error ("Cannot make cursor back!","BaseGLXWindow::ShowCursor");
}

}
}
}
