/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: $

   Created 03/27/05 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004-2005 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/model/formatleg.cc
  \brief leg model loader/saver.
*/

#include "leg/support/model/formatleg.h"
#include "leg/support/model/modelclass.h"
#include "leg/support/model/meshes.h"
#include <fstream>
#include <string>

namespace leg
{
namespace support
{
namespace model
{

const std::string leg_format_version	  = "0.0.1";
const std::string leg_format_begin	  = "LegMoDel";
const std::string leg_format_model_type	  = "ModelType";
const std::string leg_format_type_mesh	  = "ModelTypeMesh";
const std::string leg_format_new_mesh	  = "Mesh";
const std::string leg_format_face_type	  = "FaceType";
const std::string leg_format_tri	  = "Triangles";
const std::string leg_format_tri_strip	  = "TriStrip";
const std::string leg_format_tri_fan	  = "TriFan";
const std::string leg_format_vertices	  = "Vertices";
const std::string leg_format_normals	  = "Normals";
const std::string leg_format_colors	  = "Colors";
const std::string leg_format_texture	  = "Texture";
const std::string leg_format_end_texture  = "EndTexture";
const std::string leg_format_tex_id	  = "TexelId";
const std::string leg_format_texels	  = "Texels";
const std::string leg_format_material	  = "Material";
const std::string leg_format_mat_ambient  = "Ambient";
const std::string leg_format_mat_diffuse  = "Diffuse";
const std::string leg_format_mat_specular = "Specular";
const std::string leg_format_mat_shine	  = "Shininess";
const std::string leg_format_end	  = "leDoMgeL";
   
FormatLeg::FormatLeg()
{
}

FormatLeg::FormatLeg (const FormatLeg& v)
{
}

FormatLeg::~FormatLeg()
{
}

Model&
FormatLeg::Load (const std::string& filename)
{
   Model *mdl = new Model;
   char dump[128] = "\0";
   std::ifstream ifs (filename.c_str(), std::ios_base::in);
   std::string tmp;
   
   ifs >> dump;
   tmp = dump;
   if (tmp != leg_format_begin){
      std::cout << "Mismatch format." << std::endl;
      throw std::exception();
   }

   ifs >> dump;
   tmp = dump;
   if (tmp != "0.0.1"){
      std::cout << "Bad version." << std::endl;
      throw std::exception();
   }

   ifs >> dump;
   tmp = dump;
   if (tmp != leg_format_model_type){
      std::cout << "File corrupted: model type expected." << std::endl;
      throw std::exception();
   }

   ifs >> dump;
   tmp = dump;
   if (tmp != leg_format_type_mesh){
      std::cout << "Bad argument: only meshes are supported yet." << std::endl;
      throw std::exception();
   }
   
   Meshes *mesh = 0;
   uint i, v, t;
   i = v = t = 0;
   real d;
   Texture *texture = 0;
   
   while (1){
      ifs >> dump;
      tmp = dump;
      if (tmp == leg_format_end)
	 break;
      else if (tmp != leg_format_new_mesh){
	 std::cout << "File corrupted: new mesh was expected." << std::endl;
	 throw std::exception();
      }

      ifs >> v;
      mesh = new Meshes (v);
      
      ifs >> dump;
      tmp = dump;
      if (tmp != leg_format_face_type){
	 std::cout << "File corrupted: face type was expected." << std::endl;
	 throw std::exception();
      }
      
      ifs >> dump;
      tmp = dump;
      if (tmp == leg_format_tri)
	 mesh->tri_org = triangles;
      else if (tmp == leg_format_tri_strip)
	 mesh->tri_org = tri_strip;
      else if (tmp == leg_format_tri_fan)
	 mesh->tri_org = tri_fan;
      else
	 std::cout << "File corrupted: bad face format." << std::endl;

      ifs >> dump;
      tmp = dump;
      if (tmp != leg_format_vertices){
	 std::cout << "File corrupted: vertices was expected." << std::endl;
	 throw std::exception();
      }
      ifs >> v;
      for (i = 0; i < 9 * mesh->count; ++i){
	 ifs >> d;
	 mesh->vert.d[i] = d;
      }

      // Get the size here.
      for (i = 0; i < 9 * mesh->count; i+=3){
	 if (mesh->vert.d[i] < min_v[0])
	    min_v[0] = mesh->vert.d[i];
	 else if (mesh->vert.d[i] > max_v[0])
	    max_v[0] = mesh->vert.d[i];
	 if (mesh->vert.d[i+1] < min_v[1])
	    min_v[1] = mesh->vert.d[i+1];
	 else if (mesh->vert.d[i+1] > max_v[1])
	    max_v[1] = mesh->vert.d[i+1];
	 if (mesh->vert.d[i+2] < min_v[2])
	    min_v[2] = mesh->vert.d[i+2];
	 else if (mesh->vert.d[i+2] > max_v[2])
	    max_v[2] = mesh->vert.d[i+2];
      }

      ifs >> dump;
      tmp = dump;
      if (tmp != leg_format_normals){
	 std::cout << "File corrupted: normals was expected." << std::endl;
	 throw std::exception();
      }
      for (i = 0; i < 9 * mesh->count; ++i){
	 ifs >> d;
	 mesh->norm.d[i] = d;
      }

      ifs >> dump;
      tmp = dump;
      if (tmp != leg_format_colors){
	 std::cout << "File corrupted: colors was expected." << std::endl;
	 throw std::exception();
      }
      for (i = 0; i < 12 * mesh->count; ++i){
	 ifs >> d;
	 mesh->col.d[i] = d;
      }

      ifs >> dump;
      tmp = dump;
      if (tmp != leg_format_texels){
	 std::cout << "File corrupted: texels was expected." << std::endl;
	 throw std::exception();
      }
      for (i = 0; i < 6 * mesh->count; ++i){
	 ifs >> d;
	 mesh->tex.d[i] = d;
      }
      
      ifs >> dump;
      tmp = dump;
      if (tmp != leg_format_texture){
	 std::cout << "File corrupted: texture was expected." << std::endl;
	 throw std::exception();
      }
      
      while (1){
	 ifs >> dump;
	 tmp = dump;

	 if (tmp == "none")
	    continue;
	 else if (tmp == leg_format_end_texture)
	    break;

	 texture = new Texture;

	 texture->name = tmp;
	 ifs >> t;
	 texture->type = static_cast<TextureType> (t);
	 ifs >> texture->rot;
	 ifs >> texture->s[0] >> texture->s[1];
	 ifs >> texture->o[0] >> texture->o[1];
	 ifs >> texture->a >> texture->id;
	 ifs >> t;
	 texture->application = static_cast<Texture::Application> (t);

	 mesh->texture.push_back (*texture);
	 delete texture;
	 texture = 0;
      }
      
      ifs >> dump;
      tmp = dump;
      if (tmp != leg_format_material){
	 std::cout << "File corrupted: material was expected." << std::endl;
	 throw std::exception();
      }
      
      ifs >> dump;
      tmp = dump;
      if (tmp != leg_format_mat_ambient){
	 std::cout << "File corrupted: ambient was expected." << std::endl;
	 throw std::exception();
      }
      ifs >> mesh->material.a[0] >> mesh->material.a[1] >> mesh->material.a[2];
      
      ifs >> dump;
      tmp = dump;
      if (tmp != leg_format_mat_diffuse){
	 std::cout << "File corrupted: diffuse was expected." << std::endl;
	 throw std::exception();
      }
      ifs >> mesh->material.d[0] >> mesh->material.d[1] >> mesh->material.d[2];

      ifs >> dump;
      tmp = dump;
      if (tmp != leg_format_mat_specular){
	 std::cout << "File corrupted: specular was expected." << std::endl;
	 throw std::exception();
      }
      ifs >> mesh->material.s[0] >> mesh->material.s[1] >> mesh->material.s[2];

      ifs >> dump;
      tmp = dump;
      if (tmp != leg_format_mat_shine){
	 std::cout << "File corrupted: shininess was expected." << std::endl;
	 throw std::exception();
      }
      ifs >> mesh->material.h;
      
      mdl->AddMesh (*mesh);
      mesh = 0;
   }
   
   ifs.close();
   
   support::maths::Vector<3> s;
   s.x = max_v[0] - min_v[0];
   s.y = max_v[1] - min_v[1];
   s.z = max_v[2] - min_v[2];
   mdl->SetSize (s);
   
   support::maths::Vector<3> minv (min_v[0], min_v[1], min_v[2]);
   support::maths::Vector<3> maxv (max_v[0], max_v[1], max_v[2]);
   mdl->SetMinMax (minv,maxv);
   
   return *mdl;
}

void
FormatLeg::Save (const std::string& filename, Model& mdl)
{
   std::ofstream ofs (filename.c_str(), std::ios_base::out);
   ofs << leg_format_begin << " " << leg_format_version << std::endl;
   ofs << leg_format_model_type << " " << leg_format_type_mesh << std::endl;
   
   Model::MeshList::const_iterator i;
   i = mdl.BeginMesh();
   uint q;
   std::list<Texture>::const_iterator t;
   
   for (; i != mdl.EndMesh(); ++i){
      ofs << leg_format_new_mesh << " " << (*i)->count << std::endl;
      ofs << leg_format_face_type << " ";
      
      switch ((*i)->tri_org){
	 case triangles:
	    ofs << leg_format_tri << std::endl;
	    break;
	 case tri_strip:
	    ofs << leg_format_tri_strip << std::endl;
	    break;
	 case tri_fan:
	    ofs << leg_format_tri_fan << std::endl;
	    break;
	 default:
	    ofs << "unknown" << std::endl;
      }
      
      ofs << leg_format_vertices << " " << (*i)->vert.q << std::endl;
      for (q = 0; q < 9 * (*i)->count; ++q){
	 ofs << (*i)->vert.d[q];
	 if (!((q+1)%3))
	    ofs << std::endl;
	 else
	    ofs << " ";
      }
      
      ofs << leg_format_normals << std::endl;
      for (q = 0; q < 9 * (*i)->count; ++q){
	 ofs << (*i)->norm.d[q];
	 if (!((q+1)%3))
	    ofs << std::endl;
	 else
	    ofs << " ";
      }
      
      ofs << leg_format_colors << std::endl;
      for (q = 0; q < 12 * (*i)->count; ++q){
	 ofs << (*i)->col.d[q];
	 if (!((q+1)%4) && q)
	    ofs << std::endl;
	 else
	    ofs << " ";
      }
      
      ofs << leg_format_texels << std::endl;
      for (q = 0; q < 6 * (*i)->count; ++q){
	 ofs << (*i)->tex.d[q];
	 if (!((q+1)%2) && q)
	    ofs << std::endl;
	 else
	    ofs << " ";
      }
      
      ofs << leg_format_texture << " ";
      if (!((*i)->texture.size()))
	 ofs << "none" << std::endl;
      for (t = (*i)->texture.begin(); t != (*i)->texture.end(); ++t){
	 ofs << (*t).name << std::endl;
	 ofs << static_cast<int> ((*t).type) << " ";
	 ofs << (*t).rot << " ";
	 ofs << (*t).s[0] << " ";
	 ofs << (*t).s[1] << " ";
	 ofs << (*t).o[0] << " ";
	 ofs << (*t).o[1] << " ";
	 ofs << (*t).a << " ";
	 ofs << (*t).id << " ";
	 ofs << static_cast<int> ((*t).application) << std::endl;
      }
      ofs << leg_format_end_texture << std::endl;
      
      ofs << leg_format_material << std::endl;
      Material *mat = &(*i)->material;
      ofs << leg_format_mat_ambient << " ";
      ofs << mat->a[0] << " " << mat->a[1] << " " << mat->a[2] << std::endl;
      ofs << leg_format_mat_diffuse << " ";
      ofs << mat->d[0] << " " << mat->d[1] << " " << mat->d[2] << std::endl;
      ofs << leg_format_mat_specular << " ";
      ofs << mat->s[0] << " " << mat->s[1] << " " << mat->s[2] << std::endl;
      ofs << leg_format_mat_shine << " " << mat->h << std::endl;
   }
   
   ofs << leg_format_end;

   ofs.close();
}

BaseFormatLoader&
FormatLeg::Clone()
{
   return *new FormatLeg (*this);
}

void
FormatLeg::Copy (const FormatLeg& f)
{
}

}
}
}
