/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: $

   Created 07/28/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/model/format3ds.cc
  \brief 3ds model loader.
*/

#include "leg/support/model/format3ds.h"
#include "leg/support/model/model.h"
#include "leg/support/utils/errors.h"
#include "leg/support/maths/matrix.h"

#include <lib3ds/camera.h>
#include <lib3ds/mesh.h>
#include <lib3ds/material.h>
#include <lib3ds/matrix.h>
#include <lib3ds/vector.h>

#include <cstdlib>

#include <iostream>

using namespace std;

namespace leg
{
namespace support
{
namespace model
{
   maths::Matrix<4,4>	node_matrix;
   maths::Vector<3>	node_pivot;
   
   const std::string Format3ds::extension = ".3ds";

   Format3ds::Format3ds(): BaseFormatLoader(),
			   mdl (0)
   {
      min_v[0] = min_v[1] = min_v[2] = 0;
      max_v[0] = max_v[1] = max_v[2] = 0;
   }

   Format3ds::Format3ds (const Format3ds& v):
      BaseFormatLoader (static_cast<const BaseFormatLoader&> (v))
   {
      Copy (v);
   }

   Format3ds::~Format3ds()
   {
      if (mdl){
	 delete mdl;
	 mdl = 0;
      }
   }

   Model&
   Format3ds::Load (const std::string& f)
   {
      file_name = f;
      Lib3dsFile *file = 0;
      Lib3dsNode *node = 0;

      file = lib3ds_file_load (file_name.c_str());
      if (!file)
	 support::utils::Error ("Cannot load 3ds file, probably bad name.","support::model::Load()");
      
      lib3ds_file_eval (file,0);
      file_rescale *= file->master_scale;
      
      // should make a placement new instead
      if (mdl){
	 delete mdl;
	 mdl = 0;
      }
      
      mdl = new Model;
      node = file->nodes;
      Lib3dsMatrix M;
      
      if (!node){
	 node_matrix = maths::IdentityMatrix<4,4>();
	 node_pivot.x = 0;
	 node_pivot.y = 0;
	 node_pivot.z = 0;
	 LoadByMesh (file);
      }
      else{      
	 lib3ds_matrix_copy (M, node->matrix);
	 
	 maths::Matrix<4,4> tmp_matrix
	    (  M[0][0],M[1][0],M[2][0],M[3][0],
	       M[0][1],M[1][1],M[2][1],M[3][1],
	       M[0][2],M[1][2],M[2][2],M[3][2],
	       M[0][3],M[1][3],M[2][3],M[3][3]);
	 
	 node_matrix = tmp_matrix;

	 node_pivot.x = -node->data.object.pivot[0];
	 node_pivot.y = -node->data.object.pivot[1];
	 node_pivot.z = -node->data.object.pivot[2];
	 
	 for (; node != 0; node = node->next){
	    if (node->childs)
	       std::cout << "WARNING: Childs of node is not yet supported !" << std::endl;
	    ReadNodeInformation (node,file);
	 }
      }
      
      support::maths::Vector<3> s;
      s.x = max_v[0] - min_v[0];
      s.y = max_v[1] - min_v[1];
      s.z = max_v[2] - min_v[2];
      mdl->SetSize (s);
      
      support::maths::Vector<3> minv (min_v[0], min_v[1], min_v[2]);
      support::maths::Vector<3> maxv (max_v[0], max_v[1], max_v[2]);
      mdl->SetMinMax (minv,maxv);
      
      lib3ds_file_free (file);
      
      return *mdl;
   }

   void
   Format3ds::LoadByMesh (Lib3dsFile* file)
   {
      Lib3dsMesh *m = file->meshes;
      if (!m)
	 support::utils::Error ("Corrupted 3ds file","support::model::LoadByMesh");
      
      for (; m; m = m->next)
	 ReadMeshInformation (m,file);
   }

   BaseFormatLoader&
   Format3ds::Clone()
   {
      return * new Format3ds (*this);
   }

   void
   Format3ds::Copy (const Format3ds& f)
   {
      file_name = f.file_name;
      mdl = new Model (*f.mdl);
   }
   
   void
   Format3ds::ReadNodeInformation (Lib3dsNode *const node, Lib3dsFile *const file)
   {
      if (node->type == LIB3DS_OBJECT_NODE){
	 if (strcmp (node->name, "$$$DUMMY") == 0)
	    support::utils::Error ( "Bad information, cannot read file.",
				    "support::model::Format3ds::ReadNodeInformation");
	 
	 Lib3dsMesh *mesh = lib3ds_file_mesh_by_name (file,node->name);
	 
	 ReadMeshInformation (mesh, file);
      } //if (node->type==LIB3DS_OBJECT_NODE){
   }

   void
   Format3ds::ReadMeshInformation (Lib3dsMesh *mesh, Lib3dsFile *const file)
   {
      uint	t, i = 0, k = 0;
      GLuint	mat_match_color;
      
      ASSERT (mesh);
      
      Lib3dsVector *normalL = static_cast<float(*)[3]> (std::malloc (3*sizeof(Lib3dsVector)*mesh->faces));
      if (!normalL)
	 support::utils::Error ( "Memory is probably full.",
				 "support::model::Format3ds::ReadNodeInformation");

      // We should save the per-mesh matrix because some objects simply have be
      // transformed in the modeler. This is not a real problem for loading 
      // simple models but could be one if it does not match the wanted 
      // orientation or if it is not centered in the origin.
      
      Lib3dsMatrix M;
      lib3ds_matrix_copy (M, mesh->matrix);
      lib3ds_matrix_inv (M);

      /*
      for (i = 0; i < 4; ++i)
	 for (k = 0; k < 4; ++k)
	    if (M[i][k] <= 1E-7)
	       M[i][k] = 0.;
      */
      i = k = 0;

      maths::Matrix<4,4> active_matrix = maths::IdentityMatrix<4,4>();
      
      maths::Matrix<4,4> transl_matrix
	    (  1,0,0,node_pivot.x,
	       0,1,0,node_pivot.y,
	       0,0,1,node_pivot.z,
	       0,0,0,1);

      support::maths::Matrix<4,4> mesh_matrix
	       (M[0][0],M[1][0],M[2][0],M[3][0],
	       M[0][1],M[1][1],M[2][1],M[3][1],
	       M[0][2],M[1][2],M[2][2],M[3][2],
	       M[0][3],M[1][3],M[2][3],M[3][3]);
      
//      active_matrix = node_matrix;
//      active_matrix = active_matrix * transl_matrix;
//      active_matrix = active_matrix * mesh_matrix;

//      std::cout << active_matrix << std::endl << std::endl;

      support::maths::Vector<3> tmp_vertex;
      support::maths::Vector<3> tr_vertex;

      lib3ds_mesh_calculate_normals (mesh, normalL);
      
      // As we're now looking inside the mesh if it has different
      // textures, the mesh might not have mesh->faces number of faces
      // but lesser. We'll then need to change that if we need to make
      // a new mesh before assigning this one to the model.
      Meshes *tmp_mesh = new Meshes (mesh->faces);
			
      std::string    prev_text;
      Material	     prev_mat;
      Lib3dsMaterial *mat = 0;
      Material	     *last_mat = 0;
      Lib3dsFloat    scale[2] = {1,1};
      Lib3dsFloat    offset[2] = {0,0};
      GLfloat	     alpha = 1.;
      
      for (t = 0; t < mesh->faces; ++t){
	 Lib3dsFace *f = &mesh->faceL[t];
	 
	 if (f->material[0])
	    mat = lib3ds_file_material_by_name (file, f->material);
	 else
	    mat = 0;
	 
	 if (mat){
	    if (std::strlen (mat->texture1_map.name) > 0){
	       if ((prev_text != mat->texture1_map.name) && (t != 0)){
		  // WE NEED TO MAKE A NEW MESH AND ADD THE CURRENT TO THE MODEL !
		  std::cout << "Need to make a new mesh !" << std::endl;
	       }
	       else if (t == 0){
		  Texture tmp_text;
		  tmp_text.name = mat->texture1_map.name;
		  tmp_text.rot = mat->texture1_map.rotation;
		  tmp_text.type = texture;
		  tmp_text.s[0] = mat->texture1_map.scale[0];
		  tmp_text.s[1] = mat->texture1_map.scale[1];
		  tmp_text.o[0] = mat->texture1_map.offset[0];
		  tmp_text.o[1] = mat->texture1_map.offset[1];
		     
		  tmp_mesh->texture.push_back (tmp_text);
	       }

	       prev_text = mat->texture1_map.name;
	    }
	    else{
	       if ((prev_text != "") && (t != 0)){
		  // WE NEED TO MAKE A NEW MESH AND ADD THE CURRENT MODEL !
		  std::cout << "Need to make a new mesh !" << std::endl;
	       }
	       /*	no more need for that, we will be able to check the 
		  *	list size instead.
	       */
	       prev_text = "";
	    }
		  
	    if (std::strlen (mat->texture2_map.name) > 0){
	       // We only take the first 'second texture', just 
	       // forgetting the others. This is much easier for the
	       // moment.
	       if (t == 0){
		  Texture tmp_text;
		  tmp_text.name = mat->texture2_map.name;
		  tmp_text.rot = mat->texture2_map.rotation;
		  tmp_text.type = texture;
		  tmp_text.s[0] = mat->texture2_map.scale[0];
		  tmp_text.s[1] = mat->texture2_map.scale[1];
		  tmp_text.o[0] = mat->texture2_map.offset[0];
		  tmp_text.o[1] = mat->texture2_map.offset[1];

		  std::cout << "Adding 2nd texture to mesh" << std::endl;
		  tmp_mesh->texture.push_back (tmp_text);
	       }
	    }
		  
	    if (std::strlen (mat->reflection_map.name) > 0){
	       // We only take the first 'second texture', just 
	       // forgetting the others. This is much easier for the
	       // moment.
	       if (t == 0){
		  Texture tmp_text;
		  tmp_text.name = mat->reflection_map.name;
		  tmp_text.rot = mat->reflection_map.rotation;
		  tmp_text.type = texture;
		  tmp_text.s[0] = mat->reflection_map.scale[0];
		  tmp_text.s[1] = mat->reflection_map.scale[1];
		  tmp_text.o[0] = mat->reflection_map.offset[0];
		  tmp_text.o[1] = mat->reflection_map.offset[1];

		  std::cout << "Adding reflection texture to mesh" << std::endl;
		  tmp_mesh->texture.push_back (tmp_text);
	       }
	    }
		  
	    alpha = 1 - mat->transparency;
	    
	    tmp_mesh->material.a[0] = mat->ambient[0];
	    tmp_mesh->material.a[1] = mat->ambient[1];
	    tmp_mesh->material.a[2] = mat->ambient[2];
	    tmp_mesh->material.a[3] = alpha;
	    
	    tmp_mesh->material.d[0] = mat->diffuse[0];
	    tmp_mesh->material.d[1] = mat->diffuse[1];
	    tmp_mesh->material.d[2] = mat->diffuse[2];
	    tmp_mesh->material.d[3] = alpha;
	    
	    tmp_mesh->material.s[0] = mat->specular[0];
	    tmp_mesh->material.s[1] = mat->specular[1];
	    tmp_mesh->material.s[2] = mat->specular[2];
	    tmp_mesh->material.s[3] = alpha;
	    
	    tmp_mesh->material.h = 11-0.2*mat->shininess;
		  
	    last_mat = &tmp_mesh->material;
	 }
	 else{
	    Material *tmp_mat = new Material;
		  
	    if (!last_mat){
	       std::cout << "WARNING: Mesh has no material !" << std::endl;
	       Lib3dsRgba a = {0.2, 0.2, 0.2, 1.0};
	       Lib3dsRgba d = {0.8, 0.8, 0.8, 1.0};
	       Lib3dsRgba s = {0.0, 0.0, 0.0, 1.0};
	       
	       tmp_mesh->material.a[0] = a[0];
	       tmp_mesh->material.a[1] = a[1];
	       tmp_mesh->material.a[2] = a[2];
	       tmp_mesh->material.a[3] = a[3];
	       
	       tmp_mesh->material.d[0] = d[0];
	       tmp_mesh->material.d[1] = d[1];
	       tmp_mesh->material.d[2] = d[2];
	       tmp_mesh->material.d[3] = d[3];
	       
	       tmp_mesh->material.s[0] = s[0];
	       tmp_mesh->material.s[1] = s[1];
	       tmp_mesh->material.s[2] = s[2];
	       tmp_mesh->material.s[3] = s[3];
	       
	       tmp_mesh->material.h = 0;
	    }
	    else
	       *tmp_mat = *last_mat;   // we assume this material is the same as the previous one.

	    last_mat = tmp_mat;
	    alpha = 1.;
	 } //if (mat)
		  
	 mat_match_color= k;
	       
	 for (i=0; i<3; ++i){
	    k= 12*t+4*i;
	    tmp_mesh->col.d[k]= last_mat->d[0];
	    tmp_mesh->col.d[k+1]= last_mat->d[1];
	    tmp_mesh->col.d[k+2]= last_mat->d[2];
	    tmp_mesh->col.d[k+3]= alpha;
		  
	    k = 9*t+3*i;

	    tmp_vertex.x = mesh->pointL[f->points[i]].pos[0];
	    tmp_vertex.y = mesh->pointL[f->points[i]].pos[1];
	    tmp_vertex.z = mesh->pointL[f->points[i]].pos[2];
	    /*
	    tr_vertex = node_matrix.HomogeneousMultiply (tmp_vertex);
	    tmp_vertex = tr_vertex;
	    tr_vertex = transl_matrix.HomogeneousMultiply (tmp_vertex);
	    tmp_vertex = tr_vertex;
	    tr_vertex = mesh_matrix.HomogeneousMultiply (tmp_vertex);
	    */
	    tr_vertex = tmp_vertex;
	    tmp_mesh->vert.d[k]= tr_vertex.x*file_rescale;
	    tmp_mesh->vert.d[k+1]= tr_vertex.z*file_rescale;
	    tmp_mesh->vert.d[k+2]= -tr_vertex.y*file_rescale;
	    
	    if (tmp_mesh->vert.d[k] < min_v[0])
	       min_v[0] = tmp_mesh->vert.d[k];
	    else if (tmp_mesh->vert.d[k] > max_v[0])
	       max_v[0] = tmp_mesh->vert.d[k];
	    if (tmp_mesh->vert.d[k+1] < min_v[1])
	       min_v[1] = tmp_mesh->vert.d[k+1];
	    else if (tmp_mesh->vert.d[k+1] > max_v[1])
	       max_v[1] = tmp_mesh->vert.d[k+1];
	    if (tmp_mesh->vert.d[k+2] < min_v[2])
	       min_v[2] = tmp_mesh->vert.d[k+2];
	    else if (tmp_mesh->vert.d[k+2] > max_v[2])
	       max_v[2] = tmp_mesh->vert.d[k+2];
		  
	    tmp_mesh->norm.d[k]= normalL[3*t+i][0];
	    tmp_mesh->norm.d[k+1]= normalL[3*t+i][2];
	    tmp_mesh->norm.d[k+2]= -normalL[3*t+i][1];
		  
	    k= 6*t+2*i;
	    if (mesh->texels > 0){
	       scale[0] = mat->texture1_map.scale[0];
	       scale[1] = mat->texture1_map.scale[1];
		     
	       if (scale[0] == 0)
		  scale[0] = 1;
	       if (scale[1] == 0)
		  scale[1] = 1;
		     
	       offset[0] = mat->texture1_map.offset[0];
	       offset[1] = mat->texture1_map.offset[1];
		     
	       tmp_mesh->tex.d[k] = offset[0] + scale[0] * mesh->texelL[f->points[i]][0];
	       tmp_mesh->tex.d[k+1] = offset[1] + scale[1] * mesh->texelL[f->points[i]][1];
	    }
	    // if no texture, do as if there wasn't coordinated
	    else{
	       tmp_mesh->tex.d[k] = 0;
	       tmp_mesh->tex.d[k+1] = 0;
	    } 
	 } //for (i=0; i<3; ++i){

	 mat= 0;
	       
      } //for (t=0; t<mesh->faces; ++t){
	    
      mdl->AddMesh (*tmp_mesh);
      free (normalL);
      normalL = 0;
   }

}
}
}

