/* This file is part of GNU Libraries and Engines for Games.

   $Id: glconfig.cc,v 1.2 2004/04/30 20:15:54 jechk Exp $
   $Log: glconfig.cc,v $
   Revision 1.2  2004/04/30 20:15:54  jechk
   Big merge.  See ChangeLog for details.

   Revision 1.1  2004/03/28 21:30:07  jechk
   Added the support/graphics module.



   Created 3/24/04 by Jeff Binder <bindej@rpi.edu>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file glconfig.cc
  \brief OpenGL setting managment.
*/

#include "leg/support/graphics/glconfig.h"

namespace leg
{
namespace support
{
namespace graphics
{

static std::map<GLenum, bool> global_options;
static std::map<GLenum, bool> global_client_options;

void
SetGlobalOption (GLenum option, bool flag)
{
  if (global_options.find (option) == global_options.end ()
      || global_options[option] != flag)
    {
      global_options[option] = flag;
      if (flag)
	glEnable (option);
      else
	glDisable (option);
    }
}

// Doesn't actually set the option.  Used internally.
void
FakeSetGlobalOption (GLenum option, bool flag)
{
  if (global_options.find (option) == global_options.end ()
      || global_options[option] != flag)
    {
      global_options[option] = flag;
    }
}

bool
GetGlobalOption (GLenum option)
{
  if (global_options.find (option) == global_options.end ())
    {
      return global_options[option];
    }
  else
    {
      return glIsEnabled (option) == GL_TRUE;
    }
}

void
SetGlobalClientOption (GLenum option, bool flag)
{
  if (global_client_options.find (option) == global_client_options.end ()
      || global_client_options[option] != flag)
    {
      global_client_options[option] = flag;
      if (flag)
	glEnableClientState (option);
      else
	glDisableClientState (option);
    }
}

bool
GetGlobalClientOption (GLenum option)
{
  if (global_client_options.find (option) == global_client_options.end ())
    {
      return global_client_options[option];
    }
  else
    {
      return false;
    }
}

void
GLConfig::SetOption (GLenum option, bool flag)
{
  options[option] = flag;
}

bool
GLConfig::GetOption (GLenum option)
{
  if (options.find (option) == options.end ())
    {
      return false;
    }
  else
    {
      return options[option];
    }
}

bool
GLConfig::IsOptionStored (GLenum option)
{
  return options.find (option) != options.end ();
}

void
GLConfig::RemoveOption (GLenum option)
{
  client_options.erase (option);
}

void
GLConfig::SetClientOption (GLenum option, bool flag)
{
  client_options[option] = flag;
}

bool
GLConfig::GetClientOption (GLenum option)
{
  if (client_options.find (option) == client_options.end ())
    {
      return false;
    }
  else
    {
      return client_options[option];
    }
}

bool
GLConfig::IsClientOptionStored (GLenum option)
{
  return options.find (option) != options.end ();
}

void
GLConfig::RemoveClientOption (GLenum option)
{
  client_options.erase (option);
}

void
GLConfig::Push ()
{
  saved.push (Options ());
  if (use_display_list)
    {
      for (Options::iterator i = options.begin (); i != options.end (); ++i)
	{
	  saved.top ()[i->first] = GetGlobalOption (i->first);
	  FakeSetGlobalOption (i->first, i->second);
	}
      display_list ();
    }
  else
    {
      for (Options::iterator i = options.begin (); i != options.end (); ++i)
	{
	  saved.top ()[i->first] = GetGlobalOption (i->first);
	  SetGlobalOption (i->first, i->second);
	}
   } 
  client_saved.push (Options ());
  for (Options::iterator i = client_options.begin (); i != client_options.end (); ++i)
    {
      client_saved.top ()[i->first] = GetGlobalClientOption (i->first);
      SetGlobalClientOption (i->first, i->second);
    }
}

void
GLConfig::Pop ()
{
  for (Options::iterator i = saved.top ().begin (); i != saved.top ().end (); ++i)
    {
      SetGlobalOption (i->first, i->second);
    }
  saved.pop ();
  for (Options::iterator i = client_saved.top ().begin (); i != client_saved.top ().end (); ++i)
    {
      SetGlobalClientOption (i->first, i->second);
    }
  client_saved.pop ();
}

void
GLConfig::EnableDisplayList ()
{
  use_display_list = true;
  display_list.Compile ();
  for (Options::iterator i = options.begin (); i != options.end (); ++i)
    {
      if (i->second)
	glEnable (i->first);
      else
	glDisable (i->first);
    }
  display_list.Finish ();
}

void
GLConfig::DisableDisplayList ()
{
  use_display_list = false;
}

}
}
}
