/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: extensions.cc,v 1.2 2004/06/15 07:05:40 jd Exp $

   Created 06/08/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/graphics/glextensions.cc
  \brief gl extensions loader.
*/

#include "leg/support/graphics/extensions.h"
#include "leg/support/graphics/glextensions.h"
#include "leg/support/utils/errors.h"
#include <GL/glu.h>
#include <iostream>

using namespace std;

namespace leg
{
namespace support
{
namespace graphics
{

using leg::support::utils::Warning;

std::string	     Extensions::gl_extensions;
std::string	     Extensions::glu_extensions;
Extensions::ExtMap   Extensions::ext_map;

const GLubyte* multi_tex_coord_2fv_arb = (GLubyte*)	 "glMultiTexCoord2fvARB";
const GLubyte* active_texture_arb = (GLubyte*)		 "glActiveTextureARB";
const GLubyte* client_active_texture_arb = (GLubyte*)	 "glClientActiveTextureARB";
const GLubyte* bind_buffer_arb = (GLubyte*)		 "glBindBufferARB";
const GLubyte* delete_buffers_arb = (GLubyte*)		 "glDeleteBuffersARB";
const GLubyte* gen_buffers_arb = (GLubyte*)		 "glGenBuffersARB";
const GLubyte* is_buffer_arb = (GLubyte*)		 "glIsBufferARB";
const GLubyte* buffer_data_arb = (GLubyte*)		 "glBufferDataARB";
const GLubyte* buffer_sub_data_arb = (GLubyte*)		 "glBufferSubDataARB";
const GLubyte* get_buffer_sub_data_arb = (GLubyte*)	 "glGetBufferSubDataARB";
const GLubyte* map_buffer_arb = (GLubyte*)		 "glMapBufferARB";
const GLubyte* unmap_buffer_arb = (GLubyte*)		 "glUnmapBufferARB";
const GLubyte* get_buffer_parameter_iv_arb = (GLubyte*)	 "glGetBufferParameterivARB";
const GLubyte* get_buffer_pointer_v_arb = (GLubyte*)	 "glGetBufferPointervARB";
const GLubyte* gen_queries_arb = (GLubyte*)		 "glGenQueriesARB";
const GLubyte* delete_queries_arb = (GLubyte*)		 "glDeleteQueriesARB";
const GLubyte* is_query_arb = (GLubyte*)		 "glIsQueryARB";
const GLubyte* begin_query_arb = (GLubyte*)		 "glBeginQueryARB";
const GLubyte* end_query_arb = (GLubyte*)		 "glEndQueryARB";
const GLubyte* get_query_iv_arb = (GLubyte*)		 "glGetQueryivARB";
const GLubyte* get_query_object_iv_arb = (GLubyte*)	 "glGetQueryObjectivARB";
const GLubyte* get_query_object_uiv_arb = (GLubyte*)	 "glGetQueryObjectuivARB";

const GLubyte* vertex_pointer_ext = (GLubyte*)		 "glVertexPointerEXT";
const GLubyte* normal_pointer_ext = (GLubyte*)		 "glNormalPointerEXT";
const GLubyte* color_pointer_ext = (GLubyte*)		 "glColorPointerEXT";
const GLubyte* tex_coord_pointer_ext = (GLubyte*)	 "glTexCoordPointerEXT";
const GLubyte* index_pointer_ext = (GLubyte*)		 "glIndexPointerEXT";
const GLubyte* draw_arrays_ext = (GLubyte*)		 "glDrawArraysEXT";
const GLubyte* draw_range_elements_ext = (GLubyte*)	 "glDrawRangeElementsEXT";
const GLubyte* lock_arrays_ext = (GLubyte*)		 "glLockArraysEXT";
const GLubyte* unlock_arrays_ext = (GLubyte*)		 "glUnlockArraysEXT";
const GLubyte* bind_texture_ext = (GLubyte*)		 "glBindTextureEXT";
const GLubyte* are_textures_resident_ext = (GLubyte*)	 "glAreTexturesResidentEXT";
const GLubyte* delete_textures_ext = (GLubyte*)		 "glDeleteTexturesEXT";
const GLubyte* gen_textures_ext = (GLubyte*)		 "glGenTexturesEXT";
const GLubyte* is_texture_ext = (GLubyte*)		 "glIsTextureEXT";
const GLubyte* prioritize_textures_ext = (GLubyte*)	 "glPrioritizeTexturesEXT";
const GLubyte* active_stencil_face_ext = (GLubyte*)	 "glActiveStencilFaceEXT";

#ifndef WIN32
const GLubyte* allocate_memory_nv = (GLubyte*)		 "glXAllocateMemoryNV";
const GLubyte* free_memory_nv = (GLubyte*)		 "glXFreeMemoryNV";
#else
const GLubyte* allocate_memory_nv = (GLubyte*)		 "wglAllocateMemoryNV";
const GLubyte* free_memory_nv = (GLubyte*)		 "wglFreeMemoryNV";
#endif
const GLubyte* vertex_array_range_nv = (GLubyte*)	 "glVertexArrayRangeNV";
const GLubyte* flush_vertex_array_range_nv = (GLubyte*)	 "glFlushVertexArrayRangeNV";

#ifndef WIN32
PFNGLALLOCATEMEMORYNVPROC	       AgpAllocateMemory = 0;
PFNGLFREEMEMORYNVPROC		       AgpFreeMemory = 0;
#else
PFNGLALLOCATEMEMORYNVPROC	       AgpAllocateMemory = 0;
PFNGLFREEMEMORYNVPROC		       AgpFreeMemory = 0;
#endif

PFNGLVERTEXPOINTEREXTPROC	       VertexPointer = 0;
PFNGLNORMALPOINTEREXTPROC	       NormalPointer = 0;
PFNGLCOLORPOINTEREXTPROC	       ColorPointer = 0;
PFNGLTEXCOORDPOINTEREXTPROC	       TexelPointer = 0;
PFNGLINDEXPOINTEREXTPROC	       IndexPointer = 0;

PFNGLVERTEXARRAYRANGENV 	       VertexArrayRange = 0;
PFNGLFLUSHVERTEXARRAYRANGENVPROC       FlushVertexArrayRange = 0;

PFNGLDRAWRANGEELEMENTSEXTPROC	       DrawRangeElements = 0;
PFNGLDRAWARRAYSEXTPROC		       DrawArrays = 0;

PFNGLLOCKARRAYSEXTPROC		       LockArrays = 0;
PFNGLUNLOCKARRAYSEXTPROC	       UnlockArrays = 0;

PFNGLMULTITEXCOORD2FVARBPROC	       MultiTexCoord = 0;
PFNGLACTIVETEXTUREARBPROC	       ActiveTexture = 0;
PFNGLCLIENTACTIVETEXTUREARBPROC	       ClientActiveTexture = 0;
PFNGLBINDTEXTUREEXTPROC		       BindTexture = 0;
PFNGLARETEXTURESRESIDENTEXTPROC	       AreTexturesResident = 0;
PFNGLDELETETEXTURESEXTPROC	       DeleteTextures = 0;
PFNGLGENTEXTURESEXTPROC		       GenTextures = 0;
PFNGLISTEXTUREEXTPROC		       IsTexture = 0;
PFNGLPRIORITIZETEXTURESEXTPROC	       PrioritizeTextures = 0;

PFNGLBINDBUFFERARBPROC		       BindBuffer = 0;
PFNGLDELETEBUFFERSARBPROC	       DeleteBuffers = 0;
PFNGLGENBUFFERSARBPROC		       GenBuffers;
PFNGLISBUFFERARBPROC		       IsBuffer = 0;
PFNGLBUFFERDATAARBPROC		       BufferData = 0;
PFNGLBUFFERSUBDATAARBPROC	       BufferSubData = 0;
PFNGLGETBUFFERSUBDATAARBPROC	       GetBufferSubData = 0;
PFNGLMAPBUFFERARBPROC		       MapBuffer = 0;
PFNGLUNMAPBUFFERARBPROC		       UnmapBuffer = 0;
PFNGLGETBUFFERPARAMETERIVARBPROC       GetBufferParameteriv = 0;
PFNGLGETBUFFERPOINTERVARB	       GetBufferPointerv = 0;

PFNGLGENQUERIESARB		       GenQueries = 0;
PFNGLDELETEQUERIESARB		       DeleteQueries = 0;
PFNGLISQUERYARB			       IsQuery = 0;
PFNGLBEGINQUERYARB		       BeginQuery = 0;
PFNGLENDQUERYARB		       EndQuery = 0;
PFNGLGETQUERYIVARB		       GetQueryiv = 0;
PFNGLGETQUERYOBJECTIVARB	       GetQueryObjectiv = 0;
PFNGLGETQUERYOBJECTUIVARB	       GetQueryObjectuiv = 0;

PFNGLACTIVESTENCILFACEEXTPROC	       ActiveStencilFace = 0;

void
Extensions::Load (bool force)
{
   static bool loaded = false;

   if (loaded && !force)
      return;
   
   gl_extensions = reinterpret_cast<const char*> (glGetString (GL_EXTENSIONS));
   glu_extensions= reinterpret_cast<const char*> (gluGetString (GLU_EXTENSIONS));

   std::string gl_ext = gl_extensions + " " + glu_extensions;
   std::string tmp;
   unsigned int i = 0;

   while (1){
      i = gl_ext.find_first_of (" ");
      if (i < gl_ext.size()){
	 tmp = gl_ext.substr (0,i);
	 std::string cut = gl_ext.substr (i+1);
	 gl_ext = cut;
	 LoadGlExtension (tmp);
      }
      else 
	 break;
   }

   loaded = true;
}
   
void
Extensions::LoadGlExtension (const std::string& ext)
{
   if (ext == "")
      return;

   if (!Check (ext)){
      std::string msg = "Extension '" + ext + "' cannot be checked.";
      Warning (msg, "leg::support::graphics::Extensions::LoadGlExtension (const std::string&)");
      return;
   }

   if (ext == GL_ARB_MULTITEXTURE){
      MultiTexCoord = (PFNGLMULTITEXCOORD2FVARBPROC) GetFunctionPointer (multi_tex_coord_2fv_arb);
      ActiveTexture = (PFNGLACTIVETEXTUREARBPROC) GetFunctionPointer (active_texture_arb);
      ClientActiveTexture =(PFNGLCLIENTACTIVETEXTUREARBPROC)GetFunctionPointer(client_active_texture_arb);
      BindTexture = (PFNGLBINDTEXTUREEXTPROC)GetFunctionPointer (bind_texture_ext);
      AreTexturesResident =(PFNGLARETEXTURESRESIDENTEXTPROC)GetFunctionPointer (are_textures_resident_ext);
      DeleteTextures = (PFNGLDELETETEXTURESEXTPROC)GetFunctionPointer (delete_textures_ext);
      GenTextures = (PFNGLGENTEXTURESEXTPROC)GetFunctionPointer (gen_textures_ext);
      IsTexture = (PFNGLISTEXTUREEXTPROC)GetFunctionPointer (is_texture_ext);
      PrioritizeTextures = (PFNGLPRIORITIZETEXTURESEXTPROC)GetFunctionPointer (prioritize_textures_ext);

      if (  MultiTexCoord && 
	    ActiveTexture &&
	    ClientActiveTexture && 
	    BindTexture && 
	    AreTexturesResident && 
	    DeleteTextures && 
	    GenTextures && 
	    IsTexture && 
	    PrioritizeTextures){
	 ext_map[GL_ARB_MULTITEXTURE] = true;
      }
      else{
	 ext_map[GL_ARB_MULTITEXTURE] = false;
      }
   }
   else if (ext == GL_EXT_COMPILED_VERTEX_ARRAY){
      LockArrays = (PFNGLLOCKARRAYSEXTPROC)GetFunctionPointer (lock_arrays_ext);
      UnlockArrays = (PFNGLUNLOCKARRAYSEXTPROC)GetFunctionPointer (unlock_arrays_ext);

      if (LockArrays && UnlockArrays){
	 ext_map[GL_EXT_COMPILED_VERTEX_ARRAY] = true;
      }
      else{
	 ext_map[GL_EXT_COMPILED_VERTEX_ARRAY] = false;
      }
   }
   else if (ext == GL_EXT_DRAW_RANGE_ELEMENTS){
      DrawRangeElements = (PFNGLDRAWRANGEELEMENTSEXTPROC)GetFunctionPointer (draw_range_elements_ext);

      if (DrawRangeElements){
	 ext_map[GL_EXT_DRAW_RANGE_ELEMENTS] = true;
      }
      else{
	 ext_map[GL_EXT_DRAW_RANGE_ELEMENTS] = false;
      }
   }
   else if (ext == GL_EXT_VERTEX_ARRAY){
      VertexPointer = (PFNGLVERTEXPOINTEREXTPROC)GetFunctionPointer (vertex_pointer_ext);
      NormalPointer = (PFNGLNORMALPOINTEREXTPROC)GetFunctionPointer (normal_pointer_ext);
      ColorPointer = (PFNGLCOLORPOINTEREXTPROC)GetFunctionPointer (color_pointer_ext);
      TexelPointer = (PFNGLTEXCOORDPOINTEREXTPROC)GetFunctionPointer (tex_coord_pointer_ext);
      IndexPointer = (PFNGLINDEXPOINTEREXTPROC)GetFunctionPointer (index_pointer_ext);
      DrawArrays = (PFNGLDRAWARRAYSEXTPROC)GetFunctionPointer (draw_arrays_ext);

      if (VertexPointer && NormalPointer && ColorPointer && TexelPointer && IndexPointer && DrawArrays){
	 ext_map[GL_EXT_VERTEX_ARRAY] = true;
      }
      else{
	 ext_map[GL_EXT_VERTEX_ARRAY] = false;
      }
   }
   else if (ext == GL_NV_VERTEX_ARRAY_RANGE){
      VertexArrayRange = (PFNGLVERTEXARRAYRANGENV)GetFunctionPointer (vertex_array_range_nv);
      FlushVertexArrayRange = (PFNGLFLUSHVERTEXARRAYRANGENVPROC)GetFunctionPointer (flush_vertex_array_range_nv);
      AgpAllocateMemory = (PFNGLALLOCATEMEMORYNVPROC)GetFunctionPointer (allocate_memory_nv);
      AgpFreeMemory = (PFNGLFREEMEMORYNVPROC)GetFunctionPointer (free_memory_nv);

      if (VertexArrayRange && FlushVertexArrayRange && AgpAllocateMemory && AgpFreeMemory){
	 ext_map[GL_NV_VERTEX_ARRAY_RANGE] = true;
      }
      else{
	 ext_map[GL_NV_VERTEX_ARRAY_RANGE] = false;
      }
   }
   else if (ext == GL_ARB_VERTEX_BUFFER_OBJECT){
      BindBuffer = (PFNGLBINDBUFFERARBPROC)GetFunctionPointer (bind_buffer_arb);
      DeleteBuffers = (PFNGLDELETEBUFFERSARBPROC)GetFunctionPointer (delete_buffers_arb);
      GenBuffers = (PFNGLGENBUFFERSARBPROC)GetFunctionPointer (gen_buffers_arb);
      IsBuffer = (PFNGLISBUFFERARBPROC)GetFunctionPointer (is_buffer_arb);
      BufferData = (PFNGLBUFFERDATAARBPROC)GetFunctionPointer (buffer_data_arb);
      BufferSubData = (PFNGLBUFFERSUBDATAARBPROC)GetFunctionPointer (buffer_sub_data_arb);
      GetBufferSubData = (PFNGLGETBUFFERSUBDATAARBPROC)GetFunctionPointer (get_buffer_sub_data_arb);
      MapBuffer = (PFNGLMAPBUFFERARBPROC)GetFunctionPointer (map_buffer_arb); 
      UnmapBuffer = (PFNGLUNMAPBUFFERARBPROC)GetFunctionPointer (unmap_buffer_arb);
      GetBufferParameteriv = (PFNGLGETBUFFERPARAMETERIVARBPROC)GetFunctionPointer (get_buffer_parameter_iv_arb);
      GetBufferPointerv = (PFNGLGETBUFFERPOINTERVARB)GetFunctionPointer (get_buffer_pointer_v_arb);

      if (  BindBuffer &&  DeleteBuffers &&
	    GenBuffers && IsBuffer &&
	    BufferData && BufferSubData &&
	    GetBufferSubData && 
	    MapBuffer && UnmapBuffer &&
	    GetBufferParameteriv && GetBufferPointerv)
	 ext_map[GL_ARB_VERTEX_BUFFER_OBJECT] = true;
      else
	 ext_map[GL_ARB_VERTEX_BUFFER_OBJECT] = false;
   }
   else if (ext == GL_ARB_OCCLUSION_QUERY){
      GenQueries = (PFNGLGENQUERIESARB)GetFunctionPointer (gen_queries_arb);
      DeleteQueries = (PFNGLDELETEQUERIESARB)GetFunctionPointer (delete_queries_arb);
      IsQuery = (PFNGLISQUERYARB)GetFunctionPointer (is_query_arb);
      BeginQuery = (PFNGLBEGINQUERYARB)GetFunctionPointer (begin_query_arb);
      EndQuery = (PFNGLENDQUERYARB)GetFunctionPointer (end_query_arb);
      GetQueryiv = (PFNGLGETQUERYIVARB)GetFunctionPointer (get_query_iv_arb);
      GetQueryObjectiv = (PFNGLGETQUERYOBJECTIVARB)GetFunctionPointer (get_query_object_iv_arb);
      GetQueryObjectuiv = (PFNGLGETQUERYOBJECTUIVARB)GetFunctionPointer (get_query_object_uiv_arb);

      if (  GenQueries && DeleteQueries &&
	    IsQuery && BeginQuery &&
	    EndQuery && GetQueryiv &&
	    GetQueryiv && GetQueryObjectiv &&
	    GetQueryObjectuiv)
	 ext_map[GL_ARB_OCCLUSION_QUERY] = true;
      else
	 ext_map[GL_ARB_OCCLUSION_QUERY] = false;
   }
   else if (ext == GL_EXT_ACTIVE_STENCIL_FACE){
      ActiveStencilFace = (PFNGLACTIVESTENCILFACEEXTPROC)GetFunctionPointer (active_stencil_face_ext);

      if (ActiveStencilFace)
	 ext_map[GL_EXT_ACTIVE_STENCIL_FACE] = true;
      else
	 ext_map[GL_EXT_ACTIVE_STENCIL_FACE] = false;
   }
}

bool
Extensions::Check (const std::string& ext_name)
{
   return ((gluCheckExtension ((const GLubyte*)ext_name.c_str(),(const GLubyte*)gl_extensions.c_str()) == GL_TRUE) ||
	    (gluCheckExtension ((const GLubyte*)ext_name.c_str(),(const GLubyte*)glu_extensions.c_str()) == GL_TRUE));
}

bool
Extensions::IsExtensionSupported (const std::string& ext_name)
{
   return ext_map[ext_name] == true;
}

void
Extensions::CoutExtensionsSupport()
{
   ExtMap::const_iterator i;

   for (i = ext_map.begin(); i!=ext_map.end(); ++i)
      std::cout << "Extension " << i->first << (i->second?" is supported.":" is not supported.") << std::endl;
}

}// namespace graphics
}// namespace support
}// namespace leg

