/* This file is part of GNU Libraries and Engines for Games.

   $Id: $
   
   $Log: $

   Created /04 by J <>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file s
  \brief 
*/

#include "leg/libs/graphics/texture/rgb.h"
#include <cmath>

//TODO: translate into C++ all the stream stuff.

namespace leg
{
namespace libs
{
namespace graphics
{
namespace texture
{
 
Loader::TextureMap   Loader::texture_map;
std::string	     Loader::directory	  = "./textures/";

#define IMAGIC      0x01da
#define IMAGIC_SWAP 0xda01

#define SWAP_SHORT_BYTES(x) ((((x) & 0xff) << 8) | (((x) & 0xff00) >> 8))
#define SWAP_LONG_BYTES(x) (((((x) & 0xff) << 24) | (((x) & 0xff00) << 8)) | \
((((x) & 0xff0000) >> 8) | (((x) & 0xff000000) >> 24)))

struct _RgbImage
{
   unsigned short imagic;
   unsigned short type;
   unsigned short dim;
   unsigned short sizeX, sizeY, sizeZ;
   unsigned long min, max;
   unsigned long wasteBytes;
   char name[80];
   unsigned long colorMap;
   FILE *file;
   unsigned char *tmp[5];
   unsigned long rleEnd;
   unsigned long *rowStart;
   unsigned long *rowSize;
};

_RgbImage *ImageOpen (const char *fileName)
{
  _RgbImage *image;
  unsigned long *rowStart, *rowSize, ulTmp;
  int x, i;

  image = (_RgbImage *)malloc(sizeof(_RgbImage));
  if (image == NULL)
    {
      fprintf(stderr, "Out of memory!\n");
      exit(-1);
    }
  if ((image->file = fopen(fileName, "rb")) == NULL)
    {
      perror(fileName);
      exit(-1);
    }
  /*
   *	Read the image header
   */
  fread(image, 1, 12, image->file);
  /*
   *	Check byte order
   */
  if (image->imagic == IMAGIC_SWAP)
    {
      image->type = SWAP_SHORT_BYTES(image->type);
      image->dim = SWAP_SHORT_BYTES(image->dim);
      image->sizeX = SWAP_SHORT_BYTES(image->sizeX);
      image->sizeY = SWAP_SHORT_BYTES(image->sizeY);
      image->sizeZ = SWAP_SHORT_BYTES(image->sizeZ);
    }

  for ( i = 0 ; i <= image->sizeZ ; i++ )
    {
      image->tmp[i] = (unsigned char *)malloc(image->sizeX*256);
      if (image->tmp[i] == NULL )
	{
	  fprintf(stderr, "Out of memory!\n");
	  exit(-1);
	}
    }

  if ((image->type & 0xFF00) == 0x0100) /* RLE image */
    {
      x = image->sizeY * image->sizeZ * sizeof(long);
      image->rowStart = (unsigned long *)malloc(x);
      image->rowSize = (unsigned long *)malloc(x);
      if (image->rowStart == NULL || image->rowSize == NULL)
	{
	  fprintf(stderr, "Out of memory!\n");
	  exit(-1);
	}
      image->rleEnd = 512 + (2 * x);
      fseek(image->file, 512, SEEK_SET);
      fread(image->rowStart, 1, x, image->file);
      fread(image->rowSize, 1, x, image->file);
      if (image->imagic == IMAGIC_SWAP)
	{
	  x /= sizeof(long);
	  rowStart = image->rowStart;
	  rowSize = image->rowSize;
	  while (x--)
	    {
	      ulTmp = *rowStart;
	      *rowStart++ = SWAP_LONG_BYTES(ulTmp);
	      ulTmp = *rowSize;
	      *rowSize++ = SWAP_LONG_BYTES(ulTmp);
	    }
	}
    }
  return image;
}

void ImageClose( _RgbImage *image)
{
  int i;

  fclose(image->file);
  for ( i = 0 ; i <= image->sizeZ ; i++ )
    free(image->tmp[i]);
  free(image);
}

void ImageGetRow( _RgbImage *image, unsigned char *buf, int y, int z)
{
  unsigned char *iPtr, *oPtr, pixel;
  int count;

  if ((image->type & 0xFF00) == 0x0100)  /* RLE image */
    {
      fseek(image->file, image->rowStart[y+z*image->sizeY], SEEK_SET);
      fread(image->tmp[0], 1, (unsigned int)image->rowSize[y+z*image->sizeY],
	    image->file);

      iPtr = image->tmp[0];
      oPtr = buf;
      while (1)
	{
	  pixel = *iPtr++;
	  count = (int)(pixel & 0x7F);
	  if (!count)
	    return;
	  if (pixel & 0x80)
	    {
	      while (count--)
		{
		  *oPtr++ = *iPtr++;
		}
	    }
	  else
	    {
	      pixel = *iPtr++;
	      while (count--)
		{
		  *oPtr++ = pixel;
		}
	    }
	}
    }
  else /* verbatim image */
    {
      fseek(image->file, 512+(y*image->sizeX)+(z*image->sizeX*image->sizeY),
	    SEEK_SET);
      fread(buf, 1, image->sizeX, image->file);
    }
}

void ImageGetRawData( _RgbImage *image, unsigned char *data)
{
  int i, j, k;
  int remain=1;

  switch ( image->sizeZ )
    {
    case 1:
      remain = image->sizeX % 4;
      break;
    case 2:
      remain = image->sizeX % 2;
      break;
    case 3:
      remain = (image->sizeX * 3) & 0x3;
      if (remain)
	remain = 4 - remain;
      break;
    case 4:
      remain = 0;
      break;
    }

  for (i = 0; i < image->sizeY; i++)
    {
      for ( k = 0; k < image->sizeZ ; k++ )
	ImageGetRow(image, image->tmp[k+1], i, k);
      for (j = 0; j < image->sizeX; j++)
	for ( k = 1; k <= image->sizeZ ; k++ )
	  *data++ = *(image->tmp[k] + j);
      data += remain;
    }
}

// make sure the image is ok without scaling
int checkSize (int x)
{
	if (x == 2	 || x == 4 ||
		x == 8	 || x == 16 ||
		x == 32  || x == 64 ||
		x == 128 || x == 256 || x == 512)
		return 1;
	else return 0;
}

// reads in RGBA data and returns the same, 32 bit image
unsigned char *getRGBA (FILE *s, int size)
{
	unsigned char *rgba;
	unsigned char temp;
	int bread;
	int i;

	rgba = (unsigned char*)malloc (size * 4); // allocate memory for image

	//  no memory allocated?
	if (rgba == NULL)
		return 0;

	bread = fread (rgba, sizeof (unsigned char), size * 4, s);
	
	// make sure that all of the data is where it should be
	if (bread != size * 4)
	{
		free (rgba);
		return 0;
	}
	for (i = 0; i < size * 4; i += 4 )
	{
		temp = rgba[i];
		rgba[i] = rgba[i + 2];
		rgba[i + 2] = temp;
	}

	return rgba;
}

// reads in RGB data and returns RGB, 24bit image
unsigned char *getRGB (FILE *s, int size)
{
	unsigned char *rgb;
	unsigned char temp;
	int bread;
	int i;

	rgb = (unsigned char*)malloc (size * 3); // allocate memory for image
	
	//  no memory allocated?
	if (rgb == NULL)
		return 0;

	bread = fread (rgb, sizeof (unsigned char), size * 3, s); // read in the image data

	// make sure that all of the data is where it should be
	if (bread != size * 3)
	{
		// Bytes read != Bytes requested
		free (rgb);
		return 0;
	}

	for (i = 0; i < size * 3; i += 3)
	{
		temp = rgb[i];
		rgb[i] = rgb[i + 2];
		rgb[i + 2] = temp;
	}
	
	return rgb;
}

// class Image

BaseImage::BaseImage():
   size_x (0),
   size_y (0),
   size_z (0),
   data (0),
   size (0),
   color_type (rgb),
   color_bits (0)
{
}

BaseImage::BaseImage (const BaseImage& i)
{
   Copy (i);
}

BaseImage::~BaseImage()
{
}

void
BaseImage::Copy (const BaseImage& i)
{
   size_x = i.size_x;
   size_y = i.size_y;
   size_z = i.size_z;
   // TODO contine
}

// class Rgb

Rgb::Rgb(): BaseImage()
{
   unpack_alignment = 4;
}

void
Rgb::Load (const std::string& filename)
{
   _RgbImage *image;
   int sx;

   image = ImageOpen (filename.c_str());

   imagic = image->imagic;
   type = image->type;
   dim = image->dim;
   size_x = image->sizeX;
   size_y = image->sizeY;
   size_z = image->sizeZ;
   sx = ((image->sizeX) * (image->sizeZ) + 3) >> 2;
   size = sx * image->sizeY * sizeof (unsigned int);
  
   data = (unsigned char*) malloc (size);

   if (!data){
      fprintf(stderr, "Out of memory!\n");
      exit(-1);
   }

   ImageGetRawData (image, data);
   ImageClose (image);
}

// class ImageTga

ImageTga::ImageTga(): BaseImage()
{
   unpack_alignment = 1;
}

unsigned char*
ImageTga::GetData (FILE *s, int sz)
{
   if (color_bits == 32){
      color_type = rgba;
      return getRGBA (s, sz);
   }
   else if (color_bits == 24){
      color_type = rgb;
      return getRGB (s, sz);
   }      
   else 
      return 0;
}

// loads a 24bit or 32 bit targa file and uploads it to memory
// id is the texture id to bind too
// This supports only non RLE compressed file formats.
void
ImageTga::LoadTga (const std::string& name)
{
   unsigned char type[4];
   unsigned char info[7];
   unsigned char *imageData = NULL;
   int imageWidth, imageHeight;
   int imageBits, size;
   FILE *s;
   
   // Could not open the file!
   if (!(s = fopen (name.c_str(), "r+bt"))){
      std::cout << "File not found." << std::endl;
      return;//TGA_FILE_NOT_FOUND;
   }
   
   fread (&type, sizeof (char), 3, s);
   fseek (s, 12, SEEK_SET);
   fread (&info, sizeof (char), 6, s);
   
   // make sure that the colormap is 0 and type is 2, unmapped RGB
   if (type[1] != 0 || type[2] != 2){
      std::cout << "Bad image type." << std::endl;
      throw std::exception();
      return;//TGA_BAD_IMAGE_TYPE;
   }
   
   imageWidth = info[0] + info[1] * 256;
   imageHeight = info[2] + info[3] * 256;
   imageBits =	info[4]; // pixel depth
   size_x = imageWidth;
   size_y = imageHeight;
   size_z = 0;
   color_bits = imageBits;
   size = imageWidth * imageHeight; // image size
   
   // Dimension imageWidth x imageHeight is not a valid dimension!!
   if (!checkSize (imageWidth) || !checkSize (imageHeight)){
      std::cout << "Bad dimension." << std::endl;
      return;//TGA_BAD_DIMENSION;
   }
   
   // Image bits != Supported bits
   if (imageBits != 32 && imageBits != 24){
      std::cout << "Bad bits." << std::endl;
      return;//TGA_BAD_BITS;
   }
   
   // get the image data
   imageData = GetData (s, size);
   
   fclose (s);
   
   if (!imageData){
      std::cout << "Bad data." << std::endl;
      throw std::exception();
      return;//TGA_BAD_DATA;
   }
   
   data = imageData;
}

void
ImageTga::Load (const std::string& filename)
{
   LoadTga (filename.c_str());
}

// global functions

BaseImage*
LoadImage (const std::string& filename)
{
   BaseImage *image = 0;
   std::string tmp_name;
      
   // As jpg is not yet supported, but as many images are under this format,
   // we decide for the moment to truncate jpg into rgb, so that anyone will
   // be able to use the matching rgb image instead of the unuseable jpg.
   // Same thing for gif.
   
   size_t fs;
   fs = filename.size();
   
   if (  (filename.find (".jpg") < fs) || 
	 (filename.find (".JPG") < fs)){
      std::string t (filename, 0, filename.length() - 3);
      t += "rgb";
      tmp_name = t;
   }
   else if ((filename.find (".gif") < fs) || 
	    (filename.find (".GIF") < fs)){
      std::string t (filename, 0, filename.length() - 3);
      t += "rgb";
      tmp_name = t;
   }
   else
      tmp_name = filename;
   
   fs = tmp_name.size();
   
   if ((tmp_name.find (".rgb") < fs) || 
      (tmp_name.find (".RGB") < fs))
      image = new Rgb;
   else if ((tmp_name.find (".tga") < fs) || 
	 (tmp_name.find (".TGA") < fs))
      image = new ImageTga();
   else if ((tmp_name.find (".pcx") < fs) ||
	 (tmp_name.find (".PCX") < fs))
      image = new ImagePcx;
   else
      return 0;

   std::cout << "Loading image " << tmp_name << std::endl;
      
   image->Load (tmp_name);

   return image;
}

// only partial pcx file header
struct PcxHeader
{
   unsigned char manufacturer;
   unsigned char version;
   unsigned char encoding;
   unsigned char bits;
   unsigned char xMin;
   unsigned char yMin;
   unsigned char xMax;
   unsigned char yMax;
   unsigned char *palette;
};

struct TexturePcx // ex texture_t
{
   int width;		      // width of texture
   int height;		      // height of texture
   long int scaledWidth;
   long int scaledHeight;

   unsigned int texID;	      // the texture object id of this texture
   unsigned char *data;	      // the texture data
   unsigned char *palette;
};

unsigned char*
LoadPcxFile (const char *filename, PcxHeader *pcxHeader)
{
   int idx = 0;                  // counter index
   int c;                             // used to retrieve a char from the file
   int i;                             // counter index
   int numRepeat;      
   FILE *filePtr;                // file handle
   int width;                         // pcx width
   int height;                        // pcx height
   unsigned char *pixelData;     // pcx image data
   unsigned char *paletteData;   // pcx palette data

   // open PCX file
   filePtr = fopen(filename, "rb");
   if (filePtr == NULL)
      return NULL;

   // retrieve first character; should be equal to 10
   c = getc(filePtr);
   if (c != 10){
      fclose(filePtr);
      return NULL;
   }

   // retrieve next character; should be equal to 5
   c = getc(filePtr);
   if (c != 5){
      fclose(filePtr);
      return NULL;
   }

   // reposition file pointer to beginning of file
   rewind(filePtr);

   // read 4 characters of data to skip
   fgetc(filePtr);
   fgetc(filePtr);
   fgetc(filePtr);
   fgetc(filePtr);

   // retrieve leftmost x value of PCX
   pcxHeader->xMin = fgetc(filePtr);       // loword
   pcxHeader->xMin |= fgetc(filePtr) << 8; // hiword

   // retrieve bottom-most y value of PCX
   pcxHeader->yMin = fgetc(filePtr);       // loword
   pcxHeader->yMin |= fgetc(filePtr) << 8; // hiword

   // retrieve rightmost x value of PCX
   pcxHeader->xMax = fgetc(filePtr);       // loword
   pcxHeader->xMax |= fgetc(filePtr) << 8; // hiword

   // retrieve topmost y value of PCX
   pcxHeader->yMax = fgetc(filePtr);       // loword
   pcxHeader->yMax |= fgetc(filePtr) << 8; // hiword

   // calculate the width and height of the PCX
   width = pcxHeader->xMax - pcxHeader->xMin + 1;
   height = pcxHeader->yMax - pcxHeader->yMin + 1;

   // allocate memory for PCX image data
   pixelData = (unsigned char*)malloc(width*height);

   // set file pointer to 128th byte of file, where the PCX image data starts
   fseek(filePtr, 128, SEEK_SET);
     
   // decode the pixel data and store
   while (idx < (width*height)){
      c = getc(filePtr);
      if (c > 0xbf){
	 numRepeat = 0x3f & c;
         c = getc(filePtr);

         for (i = 0; i < numRepeat; i++){
	    pixelData[idx++] = c;
         }
      }
      else
	 pixelData[idx++] = c;

      fflush(stdout);
   }

   // allocate memory for the PCX image palette
   paletteData = (unsigned char*)malloc(768);

   // palette is the last 769 bytes of the PCX file
   fseek(filePtr, -769, SEEK_END);

   // verify palette; first character should be 12
   c = getc(filePtr);
   if (c != 12){
      fclose(filePtr);
      return NULL;
   }

   // read and store all of palette
   for (i = 0; i < 768; i++){
      c = getc(filePtr);
      paletteData[i] = c;
   }

   // close file and store palette in header
   fclose(filePtr);
   pcxHeader->palette = paletteData;

   // return the pixel image data
   return pixelData;
}

ImagePcx::ImagePcx(): BaseImage()
{
   color_type = rgba;
   color_bits = 16;
   unpack_alignment = 4;
}

void
ImagePcx::Load (const std::string& filename)
{
   TexturePcx *tmp_tex;

   tmp_tex = LoadPcx (filename);

   data = tmp_tex->data;
   size_x = tmp_tex->width;
   size_y = tmp_tex->height;

   free (tmp_tex->palette);
   tmp_tex->palette = 0;
}

TexturePcx*
ImagePcx::LoadPcx (const std::string& filename)
{
   PcxHeader texInfo;            // header of texture
   TexturePcx *thisTexture;       // the texture
   unsigned char *unscaledData;// used to calculate pcx
   int i;                             // index counter
   int j;                             // index counter
   int width;                         // width of texture
   int height;                        // height of texture

   thisTexture = new TexturePcx;

   thisTexture->data = LoadPcxFile(filename.c_str(), &texInfo);
   
   if (!thisTexture->data){
      return 0;
   }

   // store the texture information
   thisTexture->palette = texInfo.palette;
   thisTexture->width = texInfo.xMax - texInfo.xMin + 1;
   thisTexture->height = texInfo.yMax - texInfo.yMin + 1;

   // allocate memory for the unscaled data
   unscaledData = (unsigned char*)malloc(thisTexture->width*thisTexture->height*4);

   // store the unscaled data via the palette
   for (j = 0; j < thisTexture->height; j++){
      for (i = 0; i < thisTexture->width; i++){
	 unscaledData[4*(j*thisTexture->width+i)+0] = thisTexture->palette[3*thisTexture->data[j*thisTexture->width+i]+0];
         unscaledData[4*(j*thisTexture->width+i)+1] = thisTexture->palette[3*thisTexture->data[j*thisTexture->width+i]+1];
         unscaledData[4*(j*thisTexture->width+i)+2] = thisTexture->palette[3*thisTexture->data[j*thisTexture->width+i]+2];
         unscaledData[4*(j*thisTexture->width+i)+3] = 255;
      }
   }

   // find width and height's nearest greater power of 2
   width = thisTexture->width;
   height = thisTexture->height;

   // find width's
   i = 0;
   while (width){
      width /= 2;
      i++;
   }
   thisTexture->scaledHeight = (long)std::pow((float)2, (float)(i-1));

   // find height's
   i = 0;
   while (height){
      height /= 2;
      i++;
   }
   thisTexture->scaledWidth = (long)std::pow((float)2, (float)(i-1));

   // clear the texture data
   if (thisTexture->data){
      free (thisTexture->data);
      thisTexture->data = 0;
   }

   // reallocate memory for the texture data
   thisTexture->data = (unsigned char*)malloc(thisTexture->scaledWidth*thisTexture->scaledHeight*4);
     
   // use the GL utility library to scale the texture to the unscaled dimensions
   gluScaleImage (GL_RGBA, thisTexture->width, thisTexture->height, GL_UNSIGNED_BYTE, unscaledData, thisTexture->scaledWidth, thisTexture->scaledHeight, GL_UNSIGNED_BYTE, thisTexture->data);

   return thisTexture;
}

}
}
}
}
