/* This file is part of GNU Libraries and Engines for Games.

   $Id: light.cc,v 1.3 2004/07/03 20:15:05 jd Exp $
   $Log: light.cc,v $
   Revision 1.3  2004/07/03 20:15:05  jd
   graphics no more depend on scene, first release

   Revision 1.2  2004/05/22 06:54:16  jechk
   Graphics update, mainly framework stuff.

   Revision 1.1  2004/04/30 20:15:54  jechk
   Big merge.  See ChangeLog for details.


   Created 4/22/04 by Jeff Binder <bindej@rpi.edu>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file light.cc
  \brief A class representing lights.
*/

#include <list>

#include "leg/libs/graphics/light.h"
#include "leg/libs/graphics/scene.h"

namespace leg
{
namespace libs
{
namespace graphics
{
namespace lighting
{

using namespace std;

namespace internal
{
std::list<GLenum> free_lights;

void
GetLightInfo ()
{
  static bool setup = false;
  if (!setup)
    {
      setup = true;

      GLint max;
      glGetIntegerv (GL_MAX_LIGHTS, &max);

      for (GLint i = 0; i < max; ++i)
	{
	  free_lights.push_front (GL_LIGHT0 + i);
	}
    }
}
}

unsigned int
GetNumFreeLights ()
{
  return internal::free_lights.size ();
}

Light::Light (Point<3> loc)
  : loc (loc),
    enabled (false),
    ambient (0., 0., 0.),
    diffuse (1., 1., 1.),
    specular (1., 1., 1.),
    constant_atten (1.),
    linear_atten (0.),
    quadratic_atten (0.)
{
}

Light::~Light ()
{
  if (enabled)
    {
      Disable ();
    }
}

void
Light::Enable ()
{
  enabled = true;

  internal::GetLightInfo ();
  if (internal::free_lights.empty ())
    {
      throw Error ("No lights left.");
    }
  light = internal::free_lights.front ();
  internal::free_lights.pop_front ();

  glEnable (light);

  GLfloat amb[] = { ambient.r, ambient.g, ambient.b, ambient.a };
  glLightfv (light, GL_AMBIENT, amb);

  GLfloat diff[] = { diffuse.r, diffuse.g, diffuse.b, diffuse.a };
  glLightfv (light, GL_DIFFUSE, diff);

  GLfloat spec[] = { specular.r, specular.g, specular.b, specular.a };
  glLightfv (light, GL_SPECULAR, spec);

  glLightf (light, GL_CONSTANT_ATTENUATION, constant_atten);
  glLightf (light, GL_LINEAR_ATTENUATION, linear_atten);
  glLightf (light, GL_QUADRATIC_ATTENUATION, quadratic_atten);

  EnableLight ();
}

void
Light::Disable ()
{
  if (!enabled)
    {
      throw Error ("Attempt to disable a light that is not enabled.");
    }

  glDisable (light);
    
  internal::free_lights.push_front (light);
  enabled = false;
}

void
Light::Draw ()
{
  if (enabled)
    {
      glPushMatrix ();
      glTranslated (loc.x, loc.y, loc.z);
      DrawLight ();
      glPopMatrix ();
    }
}


void
Light::SetAmbient (Color c)
{
  ambient = c;
  if (enabled)
    {
      GLfloat amb[] = { ambient.r, ambient.g, ambient.b, ambient.a };
      glLightfv (light, GL_AMBIENT, amb);
    }
}

void
Light::SetDiffuse (Color c)
{
  diffuse = c;
  if (enabled)
    {
      GLfloat diff[] = { diffuse.r, diffuse.g, diffuse.b, diffuse.a };
      glLightfv (light, GL_DIFFUSE, diff);
    }
}

void
Light::SetSpecular (Color c)
{
  specular = c;
  if (enabled)
    {
      GLfloat spec[] = { specular.r, specular.g, specular.b, specular.a };
      glLightfv (light, GL_SPECULAR, spec);
    }
}

void
Light::SetConstantAttenuation (real a)
{
  constant_atten = a;
  if (enabled)
    {
      glLightf (light, GL_CONSTANT_ATTENUATION, constant_atten);
    }
}

void
Light::SetLinearAttenuation (real a)
{
  linear_atten = a;
  if (enabled)
    {
      glLightf (light, GL_LINEAR_ATTENUATION, linear_atten);
    }
}

void
Light::SetQuadraticAttenuation (real a)
{
  quadratic_atten = a;
  if (enabled)
    {
      glLightf (light, GL_QUADRATIC_ATTENUATION, quadratic_atten);
    }
}

void
Light::MessageFunc (string message, void *data)
{
  if (message == "enable")
    {
      Enable ();
    }
  else if (message == "disable")
    {
      Disable ();
    }
  else if (message == "added to scene")
    {
      //Scene *scene = static_cast<Scene *> (data);

// TODO find a way here (surely smart point it)
//      scene->SetEntityAsLight (this);  // this is not a smart pointer
    }
  else
    {;
//TODO idem almost
//      Entity::MessageFunc (message, data);
    }
}

}
}
}
}
