/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: mutex.h,v 1.2 2004/03/03 03:50:02 jechk Exp $

   Created 01/23/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/threads/mutex.h
  \brief Mutex support for threads.
*/

#ifndef LEG_SUPPORT_THREADS_MUTEX_H
#define LEG_SUPPORT_THREADS_MUTEX_H

#ifndef GNU_SOURCE
   #define GNU_SOURCE
#endif
#include <unistd.h>
#include <pthread.h>

namespace leg
{
namespace support
{
namespace threads
{
   //! Strenght of a mutex.
   /*!
    * The mutex strenght define the behavior of a mutex in a multithreaded 
    * environment. Fast mutex seems to be the best choice for games.
    */
   enum MutexStrength
   {
      //! Fast mutex.
      /*!
       * Two locks in the same threads will provoke a system hang. But this is
       * generally the default behavior we'll choose for mutexes: synchronizing
       * threads is the main time we loose regarding a single threaded program.
       */
      fast=	  PTHREAD_MUTEX_FAST_NP,

      //! Recursive locking mutex.
      /*!
       * Locks are countered and, so the last one that locks will wait until 
       * all the previous others have finished. This could be well, and is
       * somewhat like Semaphores, but recursion could be dangerous (infinite
       * recursions almost).
       */
      recursive=  PTHREAD_MUTEX_RECURSIVE_NP,

      //! Secured mutex.
      /*!
       * If could not lock, fails. Secured mutex could be well, but are slowler
       * than the others, so they are generally not used (almost for games).
       */
      secure=     PTHREAD_MUTEX_ERRORCHECK_NP
   };

   typedef pthread_mutex_t	 MutexType;	//!< Type of the mutex.
   typedef pthread_mutexattr_t	 MutexAttrType;	//!< Type of the mutex attributes.

   //! Mutex thread synchronizer.
   /*!
    * A mutex can protect some memory portions of a program running in a
    * multithreaded mode. Any shared memory should come along with a Mutex 
    * that will protect the memory from unsynchronized manipulations on it
    * with 2 or more threads. The mutex needs to remain the same during all
    * the memory data lifetime. A mutex can serve to protect different portions
    * of memory if it is used in a coherent manner.
    *
    * status:
    *	 safe.
    *
    * info:
    *	 none.
    *
    * @sa Thread,Condition,CondMutex.
    */
   class Mutex
   {
      //! the mutex identifior
      MutexType	     mutex;

      //! attributes of this mutex
      MutexAttrType  mutex_attributes;

      //! TODO: to be implemented
      MutexStrength  mutex_type;
      
      public:

      //! Default constructor.
      Mutex ();

      //! TODO: need to be tested
      Mutex (const Mutex& mut);
      
      ~Mutex ();

      //! need to be tested
      Mutex&
      operator= (const Mutex& src);

      //! Lock a mutex.
      /*!
       * Locking a mutex disallows other threads to use all the shared memories
       * from a code segment ranged between a same mutex Lock() and Unlock().
       * Be aware that any thread that will manipulate one memory segment from
       * the portion, and without Locking with the same Mutex is an error:
       * there will have no more synchronization.
       * When a thread is locking a mutex and a previous thread is already 
       * locking and running the same portion, we say that the thread is 
       * waiting for the other thread to finish. We can better say, this is the
       * mutex that is waiting (to lock), because a mutex can serve several 
       * memories in a program [however it's advised to use one mutex for one
       * memory portion only, for safety, for better understandings...]. But
       * using fast mutex will provoke the thread to wait until the OS 
       * scheduler is allowing it to run.
       *
       * This is real only when mutex are set into fast (our global policy).
       *
       * (please let me know if this is horrible to understand, so I can try to
       * change the doc).
       */
      int
      Lock ()
      {
	 return pthread_mutex_lock (&mutex);
      }

   #ifdef __USE_XOPEN2K
      int
      TimedLock (int t= 10)
      {
	 return pthread_mutex_timedlock (&mutex, (const timespec*)t);
      }
   #endif

      //! Unlock the mutex.
      /*!
       * Unlocking the mutex will 'free' the memory portion include from the
       * previous call to Lock() and this Unlock() to be useable by one thread
       * waiting for (locking) the same memory portion.
       */
      int 
      Unlock()
      {
	 return pthread_mutex_unlock (&mutex);
      }

      //! Try to lock the mutex.
      /*!
       * Trying to lock will provoke the code to continue if it's already 
       * locked by another thread.
       * This is really not advised to use this function.
       */
      int 
      TryLock ()
      {
	 return pthread_mutex_trylock (&mutex);
      }

      //! Set the strenght of the mutex.
      /*!
       * Change the behavior of the mutex. It's not advised to change it, or do
       * it with your own consciousness !
       */
      inline void 
      SetStrength (MutexStrength str)
      {
	 pthread_mutexattr_settype (&mutex_attributes, str);
      }

      private:

      //! Return the current strenght of the mutex. NOT AVAILABLE !!
      inline void
      GetStrength (MutexStrength *type)
      {
	 //pthread_mutexattr_gettype( &mut_attr, type);
      }

      public:

      // Return a pointer to the hard type pthread mutex.
      inline MutexType*
      GetMutex ()
      { 
	 return &mutex;
      }
      
      protected:

      void
      Copy (const Mutex& src);
      
      void
      Init ();

      void
      Destroy ();
   };
}// namespace threads
}// namespace support
}// namepsace leg
#endif

