/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: cond.h,v 1.2 2004/03/03 03:50:02 jechk Exp $
   $Log: cond.h,v $
   Revision 1.2  2004/03/03 03:50:02  jechk
   Changed some names, comments and other things for consistency.

   Revision 1.1  2004/03/03 02:05:22  jechk
   Merged many changes.  See ChangeLog for details.


   Created 01/23/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/threads/cond.h
  \brief Condition support for threads.
*/

#ifndef LEG_SUPPORT_THREADS_COND_H
#define LEG_SUPPORT_THREADS_COND_H

#include "mutex.h"

namespace leg
{
namespace support
{
namespace threads
{

typedef pthread_cond_t     CondType;		//!< Type of the condition.
typedef pthread_condattr_t CondAttrType;	//!< Type of the condition attributes.

//! Thread condition synchronization.
/*!
 * Condition synchronization to be used along with a mutex and 2 threads.
 *
 * status:
 *    fine.
 *
 * info:
 *    none.
 *
 * @sa Mutex,Thread,CondMutex.
 */
class Condition
{
   CondType	  cond;	   //!< matching condition waiting.

   CondAttrType	  attr;	   //!< attributes for the conditions.

   timespec	  *date;   //!< date.

   public:
      
   Condition ();

   Condition (const Condition& c);
		
   ~Condition ();

   Condition& 
   operator= (const Condition& c);

   //! Initializes the condition.
   /*! Init the condition in order to be used.
    */
   void
   Init ();

   //! Destroy the condition.
   /*! Destroy the condition that won't be useable anymore.
    */
   void
   Destroy ();

   //! Send the condition signal.
   /*! 
    * Send the condition signal to the thread that is waiting for it.
    *
    * @sa Wait()
    */
   inline int 
   Signal ()
   {
      return pthread_cond_signal (&cond);
   }

   //! Wait for the signal.
   /*!
    * Wait for the signal to be send. This has to be called in a
    * different thread from the one sending the signal.
    * Warning: if signal and wait are from the same thread, you
    * will surely encounter a system hang.
    *
    * @sa Signal(), TimedWait()
    */
   inline int 
   Wait (Mutex& mutex)
   {
      return pthread_cond_wait (&cond, mutex.GetMutex ());
   }

   //! Wait for a while the signal.
   /*!
    * Wait for some times the signal from another thread.
    * This may not be supported.
    */
   inline int 
   TimedWait (Mutex& mutex, const timespec *dt=0)
   {
      return pthread_cond_timedwait (&cond, mutex.GetMutex (), dt?dt:date);
   }

   protected:

   //! Copy the condition.
   /*!
    * Copy the condition from the argument.
    */
   void Copy (const Condition& c);
};

}// namespace threads
}// namespace support
}// namepsace leg
#endif

