/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: attribute.h,v 1.3 2004/06/01 02:05:34 jd Exp $
   
   $Log: attribute.h,v $
   Revision 1.3  2004/06/01 02:05:34  jd
   doc for namespaces

   Revision 1.2  2004/03/03 03:50:02  jechk
   Changed some names, comments and other things for consistency.

   Revision 1.1  2004/03/03 02:05:22  jechk
   Merged many changes.  See ChangeLog for details.


   Created 01/26/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/threads/attribute.h
  \brief First support of thread attributes for Leg.
*/

#ifndef LEG_SUPPORT_THREADS_ATTRIBUTE_H
#define LEG_SUPPORT_THREADS_ATTRIBUTE_H

#include "common.h"

//! The whole leg implementation.
/*!
 * All the implementation of Leg is done inside this namespace that is itself
 * divided into other sub-namespaces representing layers, in which there are
 * the main modules.
 *
 * status:
 *    safe.
 *
 * info:
 *    none.
 */
namespace leg
{

//! First layer implementation.
/*!
 * This layer provides first support for various stuffs that will be needed for
 * the whole development. All OS-dependant code is done here.
 * Each module of this layer is undependant from the others but support::utils
 * that tends to be useful for each part of the project.
 *
 * status:
 *    almost all the basis modules are done.
 *
 * info:
 *    needs audio,network and probably some others (joypads,AI...).
 */
namespace support
{

namespace threads
{

//! Attribute for threads.
/*!
 * Each thread has its corresponding attributes, all defined inside this class.
 * Thoses attributes allow to set and change the behavior of the thread. Have a
 * look at the Posix threads documentation for more details about them. An
 * attribute in itself, is nothing, so there will generally no need to use them
 * like that in any program.
 *
 * status:
 *    working.
 *
 * info:
 *    none.
 *
 * @sa Thread.
 */
class Attribute
{
   bool			constructed;	  //!< store the constructed state.
   
   protected:
   
   ThreadAttrType	attributes;	  //!< attributes of the thread.
   
   public:
   
   //! Constructor.
   /*!
    * Construct the attributes (calls Construct()).
    */
   Attribute ();

   protected:

   //! Copy constructor.
   /*!
    * There will generally have no reason of copying thread attributes, so it
    * is stored as a protected member.
    */
   Attribute (const Attribute&);

   public:

   //! Destructor.
   /*!
    * Calls Destroy ().
    */
   virtual ~Attribute ();

   protected:

   Attribute& operator= (const Attribute&);

   public:

   //! construct the attribute initialization.
   /*!
    * Do the initialization of the thread attribute.
    */
   void Construct ();

   //! Destroys the attributes.
   /*!
    * destroys the attributes. You won't be able to use it abymore but if you 
    * construct it again.
    */
   void Destroy ();

   //! Get the stored attributes.
   /*!
    * Return a reference to the stored attributes that may not be portable. So
    * refer to the documentation of Posix 1.c in order to have more details 
    * about them.
    */
   inline const ThreadAttrType&
   GetAttributes ()
   {
      return attributes;
   }

   //! Set the attributes.
   /*!
    * Set and change the attributes if exist. There will generally no need to
    * change them, so do it at your own risks.
    *
    * status:
    *	 unset.
    *
    * info:
    *	 avoid to use this function.
    */
   inline void
   SetAttributes (const ThreadAttrType& attr)
   {
      attributes = attr;
   }

   #ifndef WIN32
   //! Set the scheduling policy.
   /*!
    * Set the scheduling algorithm to be used for this thread attributes. This
    * will affects the owner thread, and this will affect the OS kernel 
    * scheduler (if supported) that will behave differently regarding the 
    * choosen algorithm.
    *
    * status:
    *	 safe under GNU/Linux only.
    *
    * info:
    *	 This function is not available on the Windows pthread port version.
    */
   void
   SetSchedulePolicy (ScheduleAlgorithm algo);

   //! Get the scheduling parameters.
   /*!
    * Returns the schedule parameters.
    *
    * info:
    *	 This function is not available under Windows.
    */
   void
   GetScheduleParam (ScheduleParam* sp);

   //! Set the schedule parameters.
   /*!
    * Change the schedule parameters.
    *
    * info:
    *	 This function is not available under Windows.
    */
   void
   SetScheduleParam (ScheduleParam* sp);
   #endif
};

}// namespace threads
}// namespace support
}// namespace leg

#endif
