/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: stateactor.h,v 1.4 2004/06/09 20:16:24 jd Exp $
   
   Created 06/01/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/states/stateactor.h
  \brief First support of state acting.
*/

#ifndef LEG_SUPPORT_STATES_STATE_ACTOR_H
#define LEG_SUPPORT_STATES_STATE_ACTOR_H

namespace leg
{
namespace support
{
namespace states
{

template 
<
   typename State,
   class Sync
> 
class BaseActorPolicy
{
   protected:

   Sync state_mutex; //!< state thread synchronizer. 
		     // Do not copy it when copying this !!
   
   public:
   
   BaseActorPolicy ()
   {
   }

   BaseActorPolicy (const BaseActorPolicy&)
   {
   }

   ~BaseActorPolicy()
   {
   }

   const BaseActorPolicy&
   operator = (const BaseActorPolicy&)
   {
      return *this;
   }

   protected:
   
   inline Sync&
   GetStateMutex()
   {
      return state_mutex;
   }
   
   inline void
   Activate (State& state, const State& value)
   {
      if (!(state & value))
	 state |= value;
   }

   inline void
   Deactivate (State& state, const State& value)
   {
      if (state & value)
	 state ^= value;
   }
};

//! State acting.
/*!
 * Act from an input: Change the state value regarding the input given in 
 * argument to Act(const Input&). The implementation of this action is made in
 * the super class Policy. This class is not intended to users, but is needed
 * by StateMachine that inherit from this class. Users will generally define
 * their own version of the policy.
 *
 * template arguments description:
 *    * Policy is the policy implementation of the state manipulation.
 *    * Input is a class representing the input (generally std::string).
 *    * State is a typename of State (generally unsigned int).
 *    * Sync is a thread synchronization tool for the state (generally Mutex).
 *
 * status:
 *    in development.
 *
 * info:
 *    It's now advised that all Policies inherit from BaseActorPolicy.
 *
 * @sa StateMachine, BaseActorPolicy
 */
template
<
   template <class,typename,class> class Policy,
   class Input,
   typename State,
   class Sync
>
class StateActor: virtual protected Policy<Input,State,Sync>
{
   protected:

   typedef Policy<Input,State,Sync> Parent;
   
   State state;	  //!< The state.

   StateActor ();
   
   StateActor (const StateActor& p):Parent (static_cast<const Parent&> (p)),
				    state (p.state)
   {
   }
   
   //! Default constructor.
   /*!
    * Needs the initial state value (generally 0).
    */
   explicit StateActor (State s):Parent(),
				 state (s)
   {
   }

   ~StateActor ()
   {
   }
   
   const StateActor&
   operator = (const StateActor& s)
   {
      Parent::operator = (static_cast<const Parent&> (s));
      state = s.state;

      return *this;
   }

   public:
   
   //! Act to an input.
   /*!
    * State manipulation regarding the input. This is just a 'bridge' function
    * that only calls to its parent, giving the correct arguments.
    */
   inline void
   Act (const Input& input)
   {
      Parent::Act (input,state);
   }
};

}
}
}
#endif
