/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: $

   Created 07/28/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/model/meshes.h
  \brief meshes for models.
*/

#ifndef LEG_SUPPORT_MODEL_MESHES_H
#define LEG_SUPPORT_MODEL_MESHES_H

#include <GL/gl.h>
#include <string>
#include <list>

namespace leg
{
namespace support
{
namespace model
{

typedef GLfloat real;
typedef GLuint  uint;

//! Mesh vertices/normals/colors/texels data
/*!
 * Simple structure containing main information about mesh data that can be
 * whether vertices, normals colors or texels.
 *
 * status:
 *    useable.
 *
 * info:
 *    none.
 */
struct VerticesData
{
   real *d;
   uint q;  // number of elements inside the array pointed by d.

   VerticesData();

   ~VerticesData();
};

typedef VerticesData Vertices;	 //!< Type used for storing vertices data
typedef VerticesData Normals;	 //!< For normals data
typedef VerticesData Colors;	 //!< For colors data
typedef VerticesData Texels;	 //!< For texels data

//! Main type of texture
/*!
 * This represents the main goals of a texture, if it's simply intended for
 * texture mapping, for reflection... Others will come later.
 */
enum TextureType
{
   none, // when no texture and cannot do otherwise only
   texture,
   reflection
};

//! Textures as seen by scene.
/*!
 * Any scene model that needs texture will use instances of this type in order
 * to do so. This is not intended to be directly used by any graphic engine.
 *
 * status:
 *    work in progress
 *
 * info:
 *    none
 */
struct Texture
{
   std::string name;	//!< name of this texture so that it can be used elsewhere
   TextureType type;	//!< type as refered by TextureType
   GLfloat     rot;	//!< any rotation around the normal of the texture plane
   real	       s[2];	//!< scale: t*=s
   real	       o[2];	//!< offset: t+=o
   real	       a;	//!< transparency
   GLuint      id;	//!< GL texture id (might not be used)
   
   //! Application of the texture
   enum Application
   {
      replace	  = GL_REPLACE,
      modulate	  = GL_MODULATE,
      blend	  = GL_BLEND,
      decal	  = GL_DECAL
   }	       application;

   Texture():  name (""),
	       type (none),
	       rot (0),
	       a (1),
	       id (0),
	       application (replace)
   {
      s[0] = s[1] = 1.;
      o[0] = o[1] = 0.;
   }

   Texture (const Texture& t)
   {
      Copy (t);
   }

   const Texture&
   operator = (const Texture& t)
   {
      Copy (t);

      return *this;
   }

   protected:

   void
   Copy (const Texture& t)
   {
      name = t.name;
      type = t.type;
      rot  = t.rot;
      s[0] = t.s[0];
      s[1] = t.s[1];
      o[0] = t.o[0];
      o[1] = t.o[1];
      a    = t.a;
      id   = t.id;
      application = t.application;
   }
};

//! Material as seen my model
/*!
 * Represents the main characteristics for handling materials.
 */
struct Material
{
   GLfloat a[4];  // ambient
   GLfloat d[4];  // diffuse
   GLfloat s[4];  // specular
   GLfloat h;     // shininess.

   Material()
   {
      unsigned int i;

      for (i = 0; i < 4; ++i)
	 a[i] = d[i] = s[i] = .0;
      h = 0;
   }

   Material (const Material& m)
   {
      Copy (m);
   }

   const Material&
   operator = (const Material& m)
   {
      Copy (m);

      return *this;
   }

   private:

   void
   Copy (const Material& m)
   {
      std::memcpy (a,m.a,4*sizeof (GLfloat));
      std::memcpy (d,m.d,4*sizeof (GLfloat));
      std::memcpy (s,m.s,4*sizeof (GLfloat));
      h = m.h;
   }
};

//! Mesh organisation representing what faces they've made from.
/*!
 * Actually only triangles are recognized.
 */
typedef GLenum MeshOrg;

const MeshOrg triangles	   = GL_TRIANGLES;
const MeshOrg tri_strip	   = GL_TRIANGLE_STRIP;
const MeshOrg tri_fan	   = GL_TRIANGLE_FAN;

/*
 * A mesh is a set of vertices data (vertices, normals, colors, texels) and a
 * common texture and material for the whole mesh. 
 * So, a mesh cannot have several materials anymore. The list of texture is
 * intended only for multitexturing, but not for sharing different textures
 * for different parts of this mesh.
 */
struct Meshes
{
   Vertices		vert;
   Normals		norm;
   Colors		col;
   Texels		tex;  // maybe put tex inside Texture within Model so that we do not have to
			      // deal with making a list for tex and match them with Texture.
   
   std::list<Texture>	texture; // would better be a vector or at least a map
   Material		material;
   MeshOrg		tri_org;
   uint			count;	// how many faces in the mesh.

   GLfloat		matrix[16];

   class Alterator
   {
      public:

      static void
      AddTexture (Meshes& mesh, const Texture& t)
      {
	 mesh.texture.push_back (t);
      }

      static void
      SetMaterial (Meshes& mesh, const Material& mat)
      {
	 mesh.material = mat;
      }

      static void
      Translate (Meshes& mesh, const GLdouble *const t)
      {
	 uint i;

	 for (i = 0; i < 9 * mesh.count; i+=3){
	    mesh.vert.d[i] += (GLfloat)t[0];
	    mesh.vert.d[i+1] += (GLfloat)t[1];
	    mesh.vert.d[i+2] += (GLfloat)t[2];
	 }
      }
      
      // plus many other funcs that a modeler can do (add vertices...)
   };

   private:
   
   Meshes();

   public:

   Meshes (uint faces_nb);

   Meshes (const Meshes& m);
   
   ~Meshes();
   
   const Meshes&
   operator = (const Meshes& m);

   protected:

   //TODO textures copy
   void
   Copy (const Meshes& m);

   friend class Alterator;
};

}
}
}

#endif
