/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: vector.h,v 1.4 2004/06/19 11:41:51 jd Exp $

   Created 12/5/03 by Jeff Binder <bindej@rpi.edu>
   
   Copyright (c) 2003, 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file leg/support/maths/vector.h
  \brief Classes to represent vector quantities (vectors, points, sizes).
*/

#ifndef LEG_SUPPORT_MATHS_VECTOR_H
#define LEG_SUPPORT_MATHS_VECTOR_H

#include <leg/support/maths/real.h>
#include <cmath>
#include <iomanip>

namespace leg
{

namespace support
{

namespace maths
{

template<unsigned int m, unsigned int n>
class Matrix;

//! An n-dimensional vector class.
/** A class to represent vector quantities with n elements.
*/
template<unsigned int n>
class Vector;

template<unsigned int n>
std::ostream&
operator << (std::ostream& os, const Vector<n>& v);

template <unsigned int n>
std::istream&
operator >> (std::istream& is, Vector<n>& v);

template<unsigned int n>
class BasicVector
{
public:
  real el[n];

  //! Copy constructor.
  BasicVector (const BasicVector<n> &v)
    {
      memcpy (this->el, v.el, n * sizeof (real));
    }

  //! Does not initialize the elements.
  BasicVector ()
    {
    }

  ~BasicVector ()
    {
    }


  //! Access the specified element of the vector.
  real&
  operator () (unsigned int i)
  {
    if (i >= n)
      {
	utils::Error ("Vector index out of bounds");
      }

    return el[i];
  }

  //! Access the specified element of the vector.
  real
  operator () (unsigned int i) const
  {
    if (i >= n)
      {
	utils::Error ("Vector index out of bounds");
      }

    return el[i];
  }


  //! Assignment equal operator.
  BasicVector<n> &
  operator = (const BasicVector<n> &r)
    {
      memcpy (el, r.el, n * sizeof *el);

      return *this;
    }
  
  //! Vector/real addition.
  Vector<n>
  operator + (real r) const
    {
      Vector<n> v;

      for (unsigned int i = 0; i < n; ++i)
	{
	  v.el[i] = el[i] + r;
	}
      
      return v;
    }
  
  //! Destructive vector/real addition.
  Vector<n>
  operator += (real r)
    {
      for (unsigned int i = 0; i < n; ++i)
	{
	  el[i] += r;
	}
      
      return *this;
    }
  
  //! Vector/real subtraction.
  Vector<n>
  operator - (real r) const
    {
      Vector<n> v;

      for (unsigned int i = 0; i < n; ++i)
	{
	  v.el[i] = el[i] - r;
	}
      
      return v;
    }
  
  //! Destructive vector/real subtraction.
  Vector<n>
  operator -= (real r)
    {
      for (unsigned int i = 0; i < n; ++i)
	{
	  el[i] -= r;
	}
      
      return *this;
    }
  
  //! Vector/real multiplication.
  Vector<n>
  operator * (real r) const
    {
      Vector<n> v;

      for (unsigned int i = 0; i < n; ++i)
	{
	  v.el[i] = el[i] * r;
	}

      return v;
    }
  
  //! Destructive vector/real multiplication.
  Vector<n>
  operator *= (real r)
    {
      for (unsigned int i = 0; i < n; ++i)
	{
	  el[i] *= r;
	}
      
      return *this;
    }

  //! Vector/real division.
  Vector<n>
  operator / (real r) const
    {
      Vector<n> v;

      r = 1. / r;
      for (unsigned int i = 0; i < n; ++i)
	{
	  v.el[i] = el[i] * r;
	}
      
      return v;
    }
  
  //! Destructive vector/real division.
  Vector<n>
  operator /= (real r)
    {
      r = 1. / r;
      for (unsigned int i = 0; i < n; ++i)
	{
	  el[i] *= r;
	}
      
      return *this;
    }
  
  //! Vector addition.
  Vector<n>
  operator + (const Vector<n> &r) const
    {
      Vector<n> v;

      for (unsigned int i = 0; i < n; ++i)
	{
	  v.el[i] = el[i] - r.el[i];
	}
      
      return v;
    }
  
  //! Destructive vector addition.
  Vector<n>
  operator += (const Vector<n> &r)
    {
      for (unsigned int i = 0; i < n; ++i)
	{
	  el[i] += r.el[i];
	}
      
      return *this;
    }
  
  //! Vector<n> subtraction.
  Vector<n>
  operator - (const Vector<n> &r) const
    {
      Vector<n> v;

      for (unsigned int i = 0; i < n; ++i)
	{
	  v.el[i] = el[i] - r.el[i];
	}

      return v;
    }
  
  //! Negates all of a vector's elements.
  Vector<n>
  operator - () const
    {
      Vector<n> v;

      for (unsigned int i = 0; i < n; ++i)
	{
	  v.el[i] = -el[i];
	}
      
      return *this;
    }
  
  //! Destructive vector subtraction.
  Vector<n>
  operator -= (const Vector<n> &r)
    {
      for (unsigned int i = 0; i < n; ++i)
	{
	  el[i] -= r.el[i];
	}
      
      return *this;
    }
  
  //! Checks for equality.
  bool
  operator == (const Vector<n> &r) const
    {
      for (unsigned int i = 0; i < n; ++i)
	{
	  if (el[i] != r.el[i])
	    {
	      return false;
	    }
	}
      
      return true;
    }
  
  //! Checks for inequality.
  bool
  operator != (const Vector<n> &r) const
    {
      for (unsigned int i = 0; i < n; ++i)
	{
	  if (el[i] != r.el[i])
	    {
	      return true;
	    }
	}
      
      return false;
    }

  //! Returns true for a null vector.
  bool
  operator ! () const
    {
      for (unsigned int i = 0; i < n; ++i)
	{
	  if (el[i])
	    {
	      return false;
	    }
	}
      
      return true;
    }


  //! Returns the angle between two vectors.
  real
  Angle (const Vector<n> &v) const
    {
      return acos (Dot (v) / (Magnitude () * v.Magnitude ()));
    }
  
  //! Dot product (a.k.a. inner product).
  real
  Dot (const Vector<n> &v) const
    {
      real d = 0;

      for (unsigned int i = 0; i < n; ++i)
	{
	  d += el[i] * v.el[i];
	}

      return d;
    }

  //! Returns the outer product of two vectors.
  template<unsigned int m>
  Matrix<n, m>
  Outer (const Vector<m> &v)
  {
    Matrix<n, m> a;

    for (unsigned int i = 0; i < n; ++i)
      {
	for (unsigned int j = 0; j < m; ++j)
	  {
	    a (i, j) = el[i] * v.el[j];
	  }
      }

    return a;
  }

  //! Calculates the average of two vectors.
  Vector<n>
  Avg (const Vector<n> &v) const
    {
      return (*this + v) / 2;
    }

  //! Reflects the vector about v.
  Vector<n>
  Reflect (const Vector<n> &v) const
    {
      return v * (v.Dot (*this) * 2) - *this;
    }

  //! Projects the vector onto v.
  Vector<n>
  Pr (const Vector<n> &v) const
    {
      return v * (Dot (v) / v.MagnitudeSq ());
    }


  //! Returns the magnitude of the vector squared.
  /** This is much more efficient than magnitude (), and should be
      used in its place whenever possible.
  */
  real
  MagnitudeSq () const
    {
      return Dot (*this);
    }

  //! Returns the vector's magnitude.
  real
  Magnitude () const
    {
      return sqrt (MagnitudeSq ());
    }

  //! Creates a new vector with a changed magnitude.
  Vector<n>
  Normal (real mag = 1.) const
    {
      real mult = mag / Magnitude ();
      return *this * mult;
    }

  //! Sets the magnitude of the vector.
  void
  Normalize (real mag = 1.)
    {
      real mult = mag / Magnitude ();

      *this *= mult;
    }

private:
  
  real
  Dot (const BasicVector<n> &v) const
    {
      real d = 0;

      for (unsigned int i = 0; i < n; ++i)
	{
	  d += el[i] * v.el[i];
	}

      return d;
    }

// December 15 2004, by Jdf.
// friend declaration of stream operators.

  friend 
  std::ostream&
  operator << <n> (std::ostream &, const Vector<n>&);

  friend
  std::istream &
  operator >> <n> (std::istream &, Vector<n>&);
};

// December 15 2004, by Jdf.
// Modified brackets so that they always end with a white-space.
template<unsigned int n>
std::ostream&
operator << (std::ostream &s, const Vector<n>& v)
{
  s << "[ ";
  for (unsigned int i = 0; i < n; ++i)
    {
      if (i != 0)
	{
	  s << " ";
	}
      s << std::setw (14) << v.el[i];
    }
  s << "] ";
  
  return s;
}

// December 15 2004, by Jdf.
// Add the function.
template <unsigned int n>
std::istream &
operator >> (std::istream &is, Vector<n>& v)
{
   char dumb;
   unsigned int i;
   
   is >> dumb;
   
   for (i = 0; i < n; ++i)
      is >> v.el[i];
   
   is >> dumb;

   return is;
}

template<>
class Vector<2> : public BasicVector<2>
{
public:
  real &x;
  real &y;

  //! Copy constructor.
  Vector (const BasicVector<2> &v)
    : BasicVector<2> (v),
      x (el[0]), y (el[1])
    {
    }

  //! Copy constructor.
  Vector (const Vector<2> &v)
    : BasicVector<2> (v),
      x (el[0]), y (el[1])
    {
    }

  //! Does not initialize the elements.
  Vector ()
    : x (el[0]), y (el[1])
    {
    }

  Vector (real x, real y)
    : x (el[0]), y (el[1])
    {
      el[0] = x;
      el[1] = y;
    }

  // This keeps the compiler from using the default assignment, which
  // doesn't work because of the reference members.
  Vector<2> &
  operator = (const Vector<2> &r)
    {
      memcpy (el, r.el, 2 * sizeof *el);
      return *this;
    }
};

template<>
class Vector<4>: public BasicVector<4>
{
  typedef BasicVector<4> Parent;
   
public:
  real& x;
  real& y;
  real& z;
  real& w;

  //! Copy constructor.
  Vector<4> (const Vector<4>& v)
    : Parent (static_cast<const Parent&> (v)),
      x (el[0]), y (el[1]), z (el[2]), w (el[3])
    {
    }

  //! Does not initialize the elements.
  Vector<4>()
    : Parent(),
      x (el[0]), y (el[1]), z (el[2]), w (el[3])
    {
    }

  explicit Vector<4> (int a, int b, int c, int d)
    : Parent(),
      x (el[0]), y (el[1]), z (el[2]), w (el[3])
    {
      x = a;
      y = b;
      z = c;
      w = d;
    }
   
  explicit Vector<4> (real a, real b, real c, real d)
    : Parent(),
      x (el[0]), y (el[1]), z (el[2]), w (el[3])
    {
      x = a;
      y = b;
      z = c;
      w = d;
    }

  Vector<4>&
  operator = (const Vector<4>& r)
    {
      memcpy (el, r.el, 4 * sizeof *el);
      return *this;
      
      //return static_cast<Vector<4>&> (BasicVector<4>::operator = (static_cast<const BasicVector<4>&> (r)));
    }

  //! Returns a 4-dimensional homogeneous copy of this vector.
  Vector<4>
  GetHomogeneousVector () const
  {
    Vector<4> v;

    if (!w)
       return v;
    
    v (0) = x/w;
    v (1) = y/w;
    v (2) = z/w;
    v (3) = 1;

    return v;
  }
};

template<>
class Vector<3> : public BasicVector<3>
{
public:
  real &x;
  real &y;
  real &z;

  //! Copy constructor.
  Vector<3> (const BasicVector<3> &v)
    : BasicVector<3> (v),
      x (el[0]), y (el[1]), z (el[2])
    {
    }

  //! Copy constructor.
  Vector<3> (const Vector<3> &v)
    : BasicVector<3> (v),
      x (el[0]), y (el[1]), z (el[2])
    {
    }

  //! Does not initialize the elements.
  Vector<3> ()
    : x (el[0]), y (el[1]), z (el[2])
    {
    }

  Vector<3> (real x, real y, real z)
    : x (el[0]), y (el[1]), z (el[2])
    {
      el[0] = x;
      el[1] = y;
      el[2] = z;
    }

  Vector<3> &
  operator = (const Vector<3> &r)
    {
      memcpy (el, r.el, 3 * sizeof *el);
      return *this;
    }


  //! Vector cross product.
  Vector<3>
  Cross (const Vector<3> &r) const
    {
      Vector<3> v (y * r.z - z * r.y,
		   z * r.x - x * r.z,
		   x * r.y - y * r.x);

      return v;
    }

  //! Returns a 4-dimensional homogeneous copy of this vector.
  Vector<4>
  GetHomogeneousVector () const
  {
    Vector<4> v;

    v (0) = x;
    v (1) = y;
    v (2) = z;
    v (3) = 1;

    return v;
  }

   inline void 
   RotateX (real alpha)
   {
      real p0 = y, p2 = z;
      real s = std::sin (alpha);
      real c = std::cos (alpha);
   
      y = (p0 * c) - (p2 * s);
      z = (p0 * s) + (p2 * c);
   }

   inline void 
   RotateY (real alpha)
   {
      real p0 = x, p2 = z;
      real s = std::sin (alpha);
      real c = std::cos (alpha);
   
      x = (p0 * c) + (p2 * s);
      z = (-p0* s) + (p2 * c);
   }

   inline void 
   RotateZ (real alpha)
   {
      real p0 = x, p2 = y;
      real s = std::sin (alpha);
      real c = std::cos (alpha);
   
      x = (p0 * c) - (p2 * s);
      y = (p0 * s) + (p2 * c);
   }
};

template<unsigned int n>
class Vector: public BasicVector<n>
{
public:

  //! Copy constructor.
  Vector<n> (const Vector<n> &v)
    : BasicVector<n> (v)
    {
    }

  //! Does not initialize the elements.
  Vector<n> ()
    : BasicVector<n> ()
    {
    }

  Vector<n> &
  operator = (const Vector<n> &r)
    {
       return static_cast<Vector<n>&> (BasicVector<n>::operator = (static_cast<const BasicVector<n>&> (r)));
    }
};

//! Creates an n-dimensional unit vector along the mth axis.
template<unsigned int n>
Vector<n>
UnitVector (unsigned int m)
{
  Vector<n> v;

  for (unsigned int i = 0; i < n; ++i)
    {
      v (i) = (i == m)? 1.: 0.;
    }

  return v;
}

//! An n-dimensional point class.
/** A class to represent points in n-space.
*/
template<unsigned int n>
class Point : public Vector<n>
{
   protected:

   using Vector<n>::el;
   
   public:

  //! Default constructor.
  Point (real x = 0., real y = 0., real z = 0.)
    : Vector<n> (x, y, z)
    {
    }

  //! Copy constructor.
  Point (const Point &p)
    : Vector<n> (p)
    {
    }

  ~Point ()
    {
    }
  
  Point& operator = (const Point &r)
  {
    memcpy (el, r.el, 3 * sizeof *el);
    return *this;
  }

  //! Returns the distance to another point squared.
  /** This is much more efficient than distanceTo (), and should be
      used in its place whenever possible.
  */
  real
  DistanceToSq (const Point &p) const
    {
      return (*this - p).MagnitudeSq ();
    }

  //! Returns the distance to another point.
  real
  DistanceTo (const Point &p) const
    {
      return sqrt (DistanceToSq (p));
    }
};

//! Specialization for 2D point.
template<>
class Point<2> : public Vector<2>
{
public:

  //! Default constructor.
  Point (real x = 0., real y = 0.)
    : Vector<2> (x, y)
    {
    }

  //! Copy constructor.
  Point (const Point &p)
    : Vector<2> (p)
    {
    }

  ~Point ()
    {
    }
  
  Point& operator = (const Point &r)
  {
    memcpy (el, r.el, 2 * sizeof *el);
    return *this;
  }

  //! Returns the distance to another point squared.
  /** This is much more efficient than distanceTo (), and should be
      used in its place whenever possible.
  */
  real
  DistanceToSq (const Point &p) const
    {
      return (*this - p).MagnitudeSq ();
    }

  //! Returns the distance to another point.
  real
  DistanceTo (const Point &p) const
    {
      return sqrt (DistanceToSq (p));
    }
};

//! An n-dimensional size class.
/** A class to represent the dimensions of n-dimensional objects.
*/
template<unsigned int n>
class Size : public Vector<n>
{
   protected:

   using Vector<n>::el;

   public:

  //! Default constructor.
  Size (real x = 0., real y = 0., real z = 0.)
    : Vector<n> (x, y, z)
    {
    }

  //! Copy constructor.
  Size (const Size &p)
    : Vector<n> (p)
    {
    }

  ~Size ()
    {
    }
  
  Size& operator = (const Size &r)
  {
    memcpy (el, r.el, 3 * sizeof *el);
    return *this;
  }

  //! Returns the area of a rectangular prism of this size.
  real
  Area ()
    {
      real a = 1.;
      for (unsigned int i = 0; i < n; ++i)
	{
	  a *= el[i];
	}

      return a;
    }
};

}

}

}

#endif /* LEG_MATHS_VECTOR_H */
