/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: real.h,v 1.6 2004/06/23 17:36:29 jd Exp $

   Created 12/5/03 by Jeff Binder <bindej@rpi.edu>
   
   Copyright (c) 2003, 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file leg/support/maths/real.h
  \brief Operations for real numbers.
*/

#ifndef LEG_SUPPORT_MATHS_REAL_H
#define LEG_SUPPORT_MATHS_REAL_H

#include <leg/support/utils/utils.h>
#include <cmath>
#include <cfloat>

namespace leg
{

namespace support
{

//! Mathematical utilites.
/*!
 * The sub namespace maths contains basic mathematical functions and
 * classes that represent mathematical objects commonly used in games.
 */
namespace maths
{

/// The default type for a real value.
typedef double real;

/// The minimum difference in values representable by a real type.
const real real_epsilon = DBL_EPSILON;

/// Returns the square of a value.
template<class T>
inline T
Sq (T s)
{
  return s * s;
}


/// Swaps two values.
template<class T>
void
Swap (T& a, T& b)
{
  T t = a;
  a = b;
  b = t;
}

namespace trigo
{

/// A constant containing the value pi.
const real pi = std::acos (0.0) * 2.;

const real pi_2 = pi / 2;
const real pi_3 = pi / 3;
const real pi_4 = pi / 4;
const real pi_5 = pi / 5;
const real pi_6 = pi / 6;

const real two_pi = 2 * pi;

const real deg_to_rad_coeff = pi / 180;
const real rad_to_deg_coeff = 180 / pi;

/// Convert degrees to radians.
inline real
DegToRad (real s)
{
  return s * deg_to_rad_coeff; // multiplication is much more faster than division.
}

/// Convert radians to degrees.
inline real
RadToDeg (real s)
{
  return s * rad_to_deg_coeff; // faster than division.
}

}

using namespace trigo;

/// Returns the smaller of its arguments.
inline real
Min (real a, real b)
{
  return a < b? a: b;
}

/// Returns the larger of its arguments.
inline real
Max (real a, real b)
{
  return a > b? a: b;
}

/// Interpolates between a and b using an interpolation factor of t.
inline real
Interp (real a, real b, real t)
{
  return a * (1 - t) + b * t;
}


//! Calculate the factorial of n.
//! Fast cpu-optimized factorial (theoretically).
inline unsigned int
Factorial (unsigned int n)
{
   return (n > 1) ? n * Factorial (n - 1) : n;
}

/// Calculates the number of permutations of k objects out of n. 
inline unsigned int
Permutation (unsigned int n, unsigned int k)
{
  unsigned int r = 1;

  if (k > n)
    {
      utils::Error ("Invalid arguments to Permutation","support::maths::Permutation (uint,uint)");
    }
  for (unsigned int i = n; i > n - k; --i)
    {
      r *= i;
    }

  return r;
}

/// Calculates the number of combinations of k objects out of n. 
inline unsigned int
Combination (unsigned int n, unsigned int k)
{
  return Permutation (n, k) / Factorial (k);
}

}

}

}

#endif /* LEG_SUPPORT_MATHS_REAL_H */
