/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: quaternion.h,v 1.6 2004/04/30 20:15:54 jechk Exp $
   $Log: quaternion.h,v $
   Revision 1.6  2004/04/30 20:15:54  jechk
   Big merge.  See ChangeLog for details.

   Revision 1.5  2004/03/09 03:54:57  jechk
   Made support::timing::Timer::timing private, minor changes.

   Revision 1.4  2004/03/08 23:07:07  jechk
   Added polynomial files.

   Revision 1.3  2004/03/08 22:26:36  jechk
   Maths update.  Mainly, added Polynomial class.

   Revision 1.2  2004/03/03 03:50:02  jechk
   Changed some names, comments and other things for consistency.

   Revision 1.1  2004/03/03 02:05:22  jechk
   Merged many changes.  See ChangeLog for details.


   Created 1/22/04 by Jeff Binder <bindej@rpi.edu>
   
   Copyright (c) 2003, 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file leg/support/maths/quaternion.h
  \brief A class representing quaternions.
*/

#ifndef LEG_SUPPORT_MATHS_QUATERNION_H
#define LEG_SUPPORT_MATHS_QUATERNION_H

#include <leg/support/maths/vector.h>

namespace leg
{

namespace support
{

namespace maths
{

template<unsigned int m, unsigned int n>
class Matrix;

//! A class to represent quaternion values.
class Quaternion
{
 public:
  real a, b, c, d;

  //! Creates a new zero quaternion.
  Quaternion ();

  //! Creates a new quaternion of the form a + bi + cj + dk.
  Quaternion (real a, real b, real c, real d);

  //! Creates a quaternion representing a rotation of angle radians about axis.
  /*!
   * Throws zero_vector_axis if axis is a zero vector.
   */
  Quaternion (real angle, Vector<3> axis);

  //! Creates a quaternion representing the rotation represented by the matrix.
  /*!
   * Throws zero_vector_axis if axis is not a rotation matrix.
   */
  Quaternion (const Matrix<4, 4> &m);

  //! Copy constructor.
  Quaternion (const Quaternion &q);

  ~Quaternion ();
  
  /// Quaternion/real addition.
  Quaternion
  operator + (real r) const
    {
      Quaternion q (a + r, b, c, d);
      
      return q;
    }
  
  /// Destructive quaternion/real addition.
  Quaternion
  operator += (real r)
    {
      a += r;
      
      return *this;
    }
  
  /// Quaternion/real subtraction.
  Quaternion
  operator - (real r) const
    {
      Quaternion q (a - r, b, c, d);
      
      return q;
    }
  
  /// Destructive quaternion/real subtraction.
  Quaternion
  operator -= (real r)
    {
      a -= r;
      
      return *this;
    }
  
  /// Quaternion/real multiplication.
  Quaternion
  operator *(real r) const
    {
      Quaternion q (a * r, b * r, c * r, d * r);

      return q;
    }
  
  /// Destructive quaternion/real multiplication.
  Quaternion
  operator *= (real r)
    {
      a *= r;
      b *= r;
      c *= r;
      d *= r;
      
      return *this;
    }

  /// Quaternion/real division.
  Quaternion
  operator / (real r) const
    {
      Quaternion q (a / r, b / r, c / r, d / r);
      
      return q;
    }
  
  /// Destructive quaternion/real division.
  Quaternion
  operator /= (real r)
    {
      a /= r;
      b /= r;
      c /= r;
      d /= r;
      
      return *this;
    }
  
  /// Checks for equality.
  bool
  operator == (const Quaternion &r) const
    {
      return (a == r.a) && (b == r.b) && (c == r.c) && (d == r.d);
    }
  
  /// Checks for inequality.
  bool
  operator != (const Quaternion &r) const
    {
      return (a != r.a) || (b != r.b) || (c != r.c) || (d != r.d);
    }

  /// Returns true for a null quaternion.
  bool
  operator ! () const
    {
      return (a == 0) && (b == 0) && (c == 0) && (d == 0);
    }

  //! Quaternion multiplication operator.
  Quaternion operator *(const Quaternion &q) const;

  //! Quaternion multiplication and assignment operator.
  Quaternion operator *= (const Quaternion &m);

  //! Quaternion addition operator.
  Quaternion operator +(const Quaternion &q) const;

  //! Quaternion addition and assignment operator.
  Quaternion operator +=(const Quaternion &q);

  //! Quaternion subtraction operator.
  Quaternion operator -(const Quaternion &q) const;

  //! Quaternion subtraction and assignment operator.
  Quaternion operator -=(const Quaternion &q);

  //! Returns the conjugate of the quaternion: a - bi - cj - dk.
  Quaternion GetConjugate () const;

  //! Returns the square of the absolute value of the quaternion.
  /** This is much more efficient than abs (), and should be
      used in its place whenever possible.
  */
  real AbsSq () const;

  //! Returns the absolute value of the quaternion.
  real Abs () const;

  //! Returns the multiplicative inverse of the quaternion.
  Quaternion Inverse () const;

  //! Returns the distance between two quaternions.
  real Distance (const Quaternion &q) const;

  //! Rotates a vector by the rotation represented by the quaternion.
  Vector<3> Rotate (const Vector<3> &v) const;

 private:
  
};

std::ostream &operator << (std::ostream &s, Quaternion v);

}

}

}

#endif // LEG_SUPPORT_MATHS_QUATERNION_H
