/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: line.h,v 1.1 2004/06/21 20:47:29 jechk Exp $
   $Log: line.h,v $
   Revision 1.1  2004/06/21 20:47:29  jechk
   Added new files in maths.



   Created 6/19/04 by Jeff Binder <bindej@rpi.edu>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file leg/support/maths/line.h
  \brief A class for lines, line segments, and rays.
*/

#ifndef LEG_SUPPORT_MATHS_LINE_H
#define LEG_SUPPORT_MATHS_LINE_H

#include <leg/support/maths/vector.h>

#include <vector>

namespace leg
{

namespace support
{

namespace maths
{

enum LineType
  {
    segment, ray, line
  };

template<unsigned int n>
class BasicLine;

template<unsigned int n>
class Line : public BasicLine<n>
{
public:
  Line (Point<n> a, Point<n> b, LineType type)
    : BasicLine<n> (a, b, type)
  {
  }
  
  Line (const Line<n> &line)
    : BasicLine<n> (line)
  {
  }
};

template<unsigned int n>
class BasicLine
{
public:
  Point<n> a;
  Point<n> b;

  BasicLine (Point<n> a, Point<n> b, LineType type)
    : a (a), b (b), type (type)
  {
  }
  
  BasicLine (const Line<n> &line)
    : a (line.a), b (line.b), type (line.type)
  {
  }

  ~BasicLine ()
  {
  }


  bool
  IsSegment ()
  {
    return type == segment;
  }

  bool
  IsRay ()
  {
    return type == ray;
  }

  bool
  IsLine ()
  {
    return type == line;
  }


  real Length ()
  {
    return (b - a).Magnitude ();
  }

  real LengthSq ()
  {
    return (b - a).MagnitudeSq ();
  }

  Point<n> Midpoint ()
  {
    return (a + b) * .5;
  }

  //! Returns whether the linear interpolation factor t is valid.
  /*! It is valid if it results in a valid point in this line (that
      is, not off the end of a line segment or ray).
  */
  bool IsValidT (real t)
  {
    return (type == line || t >= 0)
      && (type != segment || t <= 1);
  }

  Point<3> Interp (real t)
  {
    return a + (b - a) * t;
  }

private:
  LineType type;
};

template<>
class Line<2> : public BasicLine<2>
{
public:
  Line (Point<2> a, Point<2> b, LineType type)
    : BasicLine<2> (a, b, type)
  {
  }
  
  Line (const Line<2> &line)
    : BasicLine<2> (line)
  {
  }


  real InterpX (real t)
  {
    return a.x + (b.x - a.x) * t;
  }

  real InterpY (real t)
  {
    return a.y + (b.y - a.y) * t;
  }


  real Slope ()
  {
    return (b.y - a.y) / (b.x - a.x);
  }

  real YIntercept ()
  {
    return a.y - Slope () * a.x;
  }


  bool Intersects (Line<2> line)
  {
    real m, n, d;

    m = (line.b.x - line.a.x) * (a.y - line.a.y) -
      (line.b.y - line.a.y) * (a.x - line.a.x);
    n = (b.x - a.x) * (a.y - line.a.y) -
      (b.y - a.y) * (a.x - line.a.x);
    d = (line.b.y - line.a.y) * (b.x - a.x) -
      (line.b.x - line.a.x) * (b.y - a.y);

    if (d == 0)
      {
	return false;
      }

    m /= d;
    n /= d;

    return IsValidT (m) && line.IsValidT (n);
  }


  Point<2> ClosestPointOnLineTo_Point (Point<2> p)
  {
    real u;

    u = ((p.x - a.x) * (b.x - a.x) + (p.y - a.y) * (b.y - a.y))
      / LengthSq ();

    if (!IsLine () && u < 0)
      u = 0;
    
    if (IsSegment () && u > 1)
      u = 1;

    return Point<2> (a.x + u * (b.x - a.x), a.y + u * (b.y - a.y));
  }
};

template<>
class Line<3> : public BasicLine<3>
{
public:
  Line (Point<3> a, Point<3> b, LineType type)
    : BasicLine<3> (a, b, type)
  {
  }
  
  Line (const Line<3> &line)
    : BasicLine<3> (line)
  {
  }


  real InterpX (real t)
  {
    return a.x + (b.x - a.x) * t;
  }

  real InterpY (real t)
  {
    return a.y + (b.y - a.y) * t;
  }

  real InterpZ (real t)
  {
    return a.z + (b.z - a.z) * t;
  }
};

}

}

}

#endif // LEG_SUPPORT_MATHS_LINE_H
