/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: guile.h,v 1.5 2004/04/30 20:15:54 jechk Exp $
   $Log: guile.h,v $
   Revision 1.5  2004/04/30 20:15:54  jechk
   Big merge.  See ChangeLog for details.

   Revision 1.4  2004/03/11 06:44:52  jechk
   Made the Sq function generic; added some TODO comments.

   Revision 1.3  2004/03/04 07:22:36  jechk
   Clean up and fixes in Guile code.

   Revision 1.2  2004/03/03 03:50:02  jechk
   Changed some names, comments and other things for consistency.

   Revision 1.1  2004/03/03 02:05:22  jechk
   Merged many changes.  See ChangeLog for details.


   Created 12/8/03 by Jeff Binder <bindej@rpi.edu>
   
   Copyright (c) 2003, 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file leg/support/guile/guile.h
  \brief Support for the GNU Ubiquitous Intelligent Language for Extension.
*/

// TODO:
// Handling of Guile exceptions.
// Clean up binding interface---what about const functions?

#ifndef LEG_SUPPORT_GUILE_GUILE_H
#define LEG_SUPPORT_GUILE_GUILE_H

#include <libguile.h>
#include <map>
#include <list>
#include <vector>
#include <string>
#include <iostream>
#include <cstdarg>
#include <cstdlib>

#include <leg/support/threads/threads.h>
#include <loki/SmartPtr.h>

namespace leg
{

namespace support
{

//! Support for the GNU Ubiquitous Intelligent Language for Extension
/*!
 * The sub namespace guile included classes that can be used to
 * easily add support for Emacs-style extensibility with the GNU
 * Guile library, using the Scheme programming language.
 */
namespace guile
{

/*!
 * \defgroup guile support/guile: Basic Guile support
 * \ingroup support
 * 
 * Guile, the GNU Ubiquitous Intelligent Language for Extension, is an interpreter
 * for the Scheme programming language packaged in a library.  It is designed to be
 * used as an extension language similar to Emacs Lisp or The Gimp's Script-fu; that
 * is, a language that is used for writing plug-ins or defining file formats.
 * 
 * In games this is useful for a number of things, including defining scripted events
 * in games, storing complex game data in a human-readable format, and creating
 * easily moddable game engines.
 * 
 * Guile provides C functions for interfacing with Guile.  GNU Leg provides a
 * higher-level, easier to use C++ interface which is described here.
 * 
 * This documentation does not assume familiarity with Guile or Scheme, but don't
 * expect to learn either of them by reading this.
 * 
 * Scheme is a typeless language.  Any Scheme value can be stored in any Scheme variable.
 * Any Scheme value can also be stored is a C or C++ variable of type SCM.  When using
 * the Guile interface directly, you will have to convert your C++ values to and from Scheme using
 * functions and macros provided by Guile wherever C++ and Scheme code meets.
 * 
 * GNU Leg provides an easier way to manage this.  The class @class Scheme
 * stores and SCM and provides safer garbage collection.  Its member
 * function GetValue() can be used to convert an Scheme variable into
 * a C++ variable of the specified type.  For example,
 * 
 * \code
 * Scheme s;
 * ...
 * int i = s.GetValue<int> ();
 * \endcode
 * 
 * Note that although Scheme is typeless, the values stored in Scheme variable do have types.
 * That is, if, in the example above, s contains a list, a runtine error will occur when
 * the program attempts to extract an int value from it.
 * 
 * Scheme's constructor is used to convert a C++ value into Scheme.
 * This is simpler from the user's point of view because C++ is a
 * typed language; the compiler already knows what the type of the
 * argument is, so the user need not specify it.  Example:
 * 
 * \code
 * int i;
 * ...
 * Scheme s (i);
 * \endcode
 * 
 * Initially, only certain built-in types can be converted to and from Scheme.  However, the function
 * AddType() can be used to make pointers to class or struct types available to Guile.
 * For example:
 * 
 * \code
 * struct t
 * {
 *   t (int i) : i (i) {}
 *   int i;
 * };
 * ...
 * AddType<t> ("t");
 * ...
 * t *a = new t (2);
 * Scheme s (a);
 * \endcode
 * 
 * Note that unless accessors are greated, Guile code cannot do anything with the value s except
 * pass it back to C++ code; accessors must be created.  Fortunately, GNU Leg provides an easy
 * way to do this.  For more on this, see \link cfg Using C++ from Guile \endlink.
 * 
 * It is important to note that the garbage collector will not, by default, free objects of types
 * defined in this way.  They must be freed from within C++ code or a memory leak will result.
 *
 * And, of course, this is not useful if there is no way to run Guile code from C++.  For information
 * on this, see \link gfc Using Guile from C++ \endlink.
 * 
 * \see leg::support::guile
 * 
 * \{
 */

//! Converts a Guile value (type SCM) to a C++ value of type T.
/*!
 * If the value stored in a cannot be converted to type T, an error is
 * raised.  The optional arguments arg and where are included in this
 * error.  They are designed to be used in C functions called from
 * Guile.  arg should be the argument number of the value a, and where
 * should be the function to which it was passes.
 * 
 * Generally it is better to avoid using this function directly by
 * using the various templates described in \link gfc Using Guile from
 * C++ \endlink and \link cfg Using C++ from Guile \endlink.
 *
 * Supported types are int, char *, and any types defined with
 * AddType().
 * 
 * \see ToGuile
 * \see AddType
 */
template<class T> T FromGuile (SCM a, int arg = 1, const char *where = "(unknown)");

//! Converts a C++ value to a Guile value.
/*!
 * Displays an error if type T cannot be converted to a Guile value.
 * 
 * Generally it is better to avoid using this function directly by using the various
 * templates described in \link gfc Using Guile from C++ \endlink and
 * \link cfg Using C++ from Guile \endlink.
 *
 * Supported types are int, double, float, bool, char, char *, Scheme,
 * SCM, any types defined with AddGuileType(), and std::lists and
 * std::vectors of these types.
 * 
 * \see FromGuile
 * \see ToGuileSymbol
 * \see AddType
 */
template<class T> SCM ToGuile (T a);

//! A class representing Guile values
/*!
 * It is generally better to use this than to use SCM directly.
 * Guile values contained in instances of this class are automatically
 * marked by the garbage collector, wherever they are stored.  Values
 * stored directly in a SCM variable are only guaranteed not to be
 * garbage collected if they are on the stack.
*/
class Scheme
{
public:
  Scheme (const SCM &scm);

  //! Constructor.
  /*!
   * Scheme can be initialized with any of the types accepted
   * ToGuile().
   */
  template<class T>
  Scheme (T a)
    : Scheme (ToGuile (a))
  {
  }

  ~Scheme ();

  //! Gets the SCM value for this object.
  SCM
  GetSCM () const
  {
    return scm_val;
  }

  //! Gets the C++ value represented by the object.
  /*!
   * T can be any of the types accepted by FromGuile().
   */
  template<class T>
  T
  GetValue () const
  {
    return FromGuile<T> (scm_val);
  }

  //! Returns true if the object is of boolean type
  bool
  IsBoolean () const
  {
    return SCM_BOOLP (scm_val);
  }

  //! Returns true if the object is of character type
  bool
  IsCharacter () const
  {
    return SCM_CHARP (scm_val);
  }

  //! Returns true if the object is of symbol type
  bool
  IsSymbol () const
  {
    return SCM_SYMBOLP (scm_val);
  }

  //! Returns true if the object is of string type
  bool
  IsString () const
  {
    return SCM_STRINGP (scm_val);
  }

  //! Returns true if the object is of numeric type
  bool
  IsNumber () const
  {
    return SCM_NUMBERP (scm_val);
  }

  //! Returns true if the object is of integer type
  bool
  IsInteger () const
  {
    return SCM_INUMP (scm_val);
  }

  //! Returns true if the object is of floating-point type
  bool
  IsFloat () const
  {
    return SCM_REALP (scm_val);
  }

  //! Returns true if the object is of vector type
  bool
  IsVector () const
  {
    return SCM_VECTORP (scm_val);
  }

  //! Returns true if the object is of list type
  bool
  IsList () const
  {
    return scm_list_p (scm_val);
  }

private:
  SCM scm_val;
  SCM sym;
};

namespace internal
{
  // This must be called before Guile can be used.  It is safe to call
  // it multiple times; however, it is guaranteed to be called before
  // main is entered, so it should not typically be needed and is
  // undocumented.
  extern void StartGuile ();

  // These are the smob type identifiers returned by scm_make_smob_type, sorted by
  // the C++ type that they represent.
  extern std::map<std::string, long> *smob_type;

  // This stores the parent types for each smob type.
  extern std::map<std::string, std::list<std::string> > *smob_children;

  /*
   * This deserves some explanation.  C++ does not allow string literals to be passed as
   * arguments to templates.  This is a hack to get around that limitation.
   * 
   * Whenever a new function is bound, you'll notice, the variable s is set to the name
   * of the function, unreal is set to true, and the function being bound is called.
   * This function is, of course, an instance of one of the '*generic*' templates below.
   * unreal being true, the GRAB macro kicks in and sets the value of 'where' to the
   * name of the function, so that it can be used in error messages.
   * 
   * where is a static variable, but note that different instances of templates each get
   * their own instance of where.  This is exactly what we want, because each instance of
   * a template represents a single C++ function that is being bound.  So for our
   * purposes, this is as good as sending the string as a template parameter (well, it's
   * not nearly as clear, but oh, well...)
   * 
   * Note that if the same function is bound to multiple names in Guile, the same instance
   * of the template is used, and they end up sharing a where variable.  This causes error
   * messages for all of the Guile bindings to display the name of the most recent binding.
   * A warning message is displayed when this happens.
   */
  threads::Mutex grab_mutex;
  const char *s;
  bool unreal;
#define GRAB static const char *where = NULL;\
             if (unreal)\
             {\
               unreal = false;\
	       if (where)\
	         {\
	           std::cerr << "WARNING: binding '" << s << "' to a function already bound to '"\
                             << where << "'.  Error messages may display the wrong function name."\
                             << std::endl;\
	         }\
       	       where = s;\
	       return SCM_UNSPECIFIED;\
             }

  // Generic functions that are passed to scm_c_define_gsubr.
  template<class T>
  SCM
  NewGeneric0 ()
  {
    GRAB;

    return ToGuile (Loki::SmartPtr<T> (new T ()));
  }

  template<class T, class A>
  SCM
  NewGeneric1 (SCM a)
  {
    GRAB;

    return ToGuile (Loki::SmartPtr<T> (new T (FromGuile<A> (a, 1, where))));
  }

  template<class T, class A, class B>
  SCM
  NewGeneric2 (SCM a, SCM b)
  {
    GRAB;

    return ToGuile (Loki::SmartPtr<T> (new T (FromGuile<A> (a, 1, where),
					      FromGuile<B> (b, 2, where))));
  }

  template<class T, class A, class B, class C>
  SCM
  NewGeneric3 (SCM a, SCM b, SCM c)
  {
    GRAB;

    return ToGuile (Loki::SmartPtr<T> (new T (FromGuile<A> (a, 1, where),
					      FromGuile<B> (b, 2, where),
					      FromGuile<C> (c, 3, where))));
  }

  template<class T, class A, class B, class C, class D>
  SCM
  NewGeneric4 (SCM a, SCM b, SCM c, SCM d)
  {
    GRAB;

    return ToGuile (Loki::SmartPtr<T> (new T (FromGuile<A> (a, 1, where),
					      FromGuile<B> (b, 2, where),
					      FromGuile<C> (c, 3, where),
					      FromGuile<D> (d, 4, where))));
  }

  template<class T, class A, class B, class C, class D, class E>
  SCM
  NewGeneric5 (SCM a, SCM b, SCM c, SCM d, SCM e)
  {
    GRAB;

    return ToGuile (Loki::SmartPtr<T> (new T (FromGuile<A> (a, 1, where),
					      FromGuile<B> (b, 2, where),
					      FromGuile<C> (c, 3, where),
					      FromGuile<D> (d, 4, where),
					      FromGuile<E> (e, 5, where))));
  }

  template<class T, class A, class B, class C, class D, class E, class F>
  SCM
  NewGeneric6 (SCM a, SCM b, SCM c, SCM d, SCM e, SCM f)
  {
    GRAB;

    return ToGuile (Loki::SmartPtr<T> (new T (FromGuile<A> (a, 1, where),
					      FromGuile<B> (b, 2, where),
					      FromGuile<C> (c, 3, where),
					      FromGuile<D> (d, 4, where),
					      FromGuile<E> (e, 5, where),
					      FromGuile<F> (f, 6, where))));
  }


  template<class R, R (*f) ()>
  SCM
  Generic0 ()
  {
    GRAB;

    return ToGuile (f ());
  }

  template<class R, class A, R (*f) (A)>
  SCM
  Generic1 (SCM a)
  {
    GRAB;

    return ToGuile (f (FromGuile<A> (a, 1, where)));
  }

  template<class R, class A, class B, R (*f) (A, B)>
  SCM
  Generic2 (SCM a, SCM b)
  {
    GRAB;

    return ToGuile (f (FromGuile<A> (a, 1, where),
		       FromGuile<B> (b, 2, where)));
  }

  template<class R, class A, class B, class C, R (*f) (A, B, C)>
  SCM
  Generic3 (SCM a, SCM b, SCM c)
  {
    GRAB;

    return ToGuile (f (FromGuile<A> (a, 1, where),
		       FromGuile<B> (b, 2, where),
		       FromGuile<C> (c, 3, where)));
  }

  template<class R, class A, class B, class C, class D, R (*f) (A, B, C, D)>
  SCM
  Generic4 (SCM a, SCM b, SCM c, SCM d)
  {
    GRAB;

    return ToGuile (f (FromGuile<A> (a, 1, where),
		       FromGuile<B> (b, 2, where),
		       FromGuile<C> (c, 3, where),
		       FromGuile<D> (d, 4, where)));
  }

  template<class R, class A, class B, class C, class D, class E, R (*f) (A, B, C, D, E)>
  SCM
  Generic5 (SCM a, SCM b, SCM c, SCM d, SCM e)
  {
    GRAB;

    return ToGuile (f (FromGuile<A> (a, 1, where),
		       FromGuile<B> (b, 2, where),
		       FromGuile<C> (c, 3, where),
		       FromGuile<D> (d, 4, where),
		       FromGuile<E> (e, 5, where)));
  }

  template<class R, class A, class B, class C, class D, class E, class F, R (*f) (A, B, C, D, E, F)>
  SCM
  Generic6 (SCM a, SCM b, SCM c, SCM d, SCM e, SCM f1)
  {
    GRAB;

    return ToGuile (f (FromGuile<A> (a, 1, where),
		       FromGuile<B> (b, 2, where),
		       FromGuile<C> (c, 3, where),
		       FromGuile<D> (d, 4, where),
		       FromGuile<E> (e, 5, where),
		       FromGuile<F> (f1, 6, where)));
  }


  template<void (*f) ()>
  SCM
  VoidGeneric0 ()
  {
    GRAB;

    f ();
    return SCM_UNSPECIFIED;
  }

  template<class A, void (*f) (A)>
  SCM
  VoidGeneric1 ( SCM a)
  {
    GRAB;

    f (FromGuile<A> (a, 1, where));
    return SCM_UNSPECIFIED;
  }

  template<class A, class B, void (*f) (A, B)>
  SCM
  VoidGeneric2 (SCM a, SCM b)
  {
    GRAB;

    f (FromGuile<A> (a, 1, where),
       FromGuile<B> (b, 2, where));

    return SCM_UNSPECIFIED;
  }

  template<class A, class B, class C, void (*f) (A, B, C)>
  SCM
  VoidGeneric3 (SCM a, SCM b, SCM c)
  {
    GRAB;

    f (FromGuile<A> (a, 1, where),
       FromGuile<B> (b, 2, where),
       FromGuile<C> (c, 3, where));

    return SCM_UNSPECIFIED;
  }

  template<class A, class B, class C, class D, void (*f) (A, B, C, D)>
  SCM
  VoidGeneric4 (SCM a, SCM b, SCM c, SCM d)
  {
    GRAB;

    f (FromGuile<A> (a, 1, where),
       FromGuile<B> (b, 2, where),
       FromGuile<C> (c, 3, where),
       FromGuile<D> (d, 4, where));

    return SCM_UNSPECIFIED;
  }

  template<class A, class B, class C, class D, class E, void (*f) (A, B, C, D, E)>
  SCM
  VoidGeneric5 (SCM a, SCM b, SCM c, SCM d, SCM e)
  {
    GRAB;

    f (FromGuile<A> (a, 1, where),
       FromGuile<B> (b, 2, where),
       FromGuile<C> (c, 3, where),
       FromGuile<D> (d, 4, where),
       FromGuile<E> (e, 5, where));

    return SCM_UNSPECIFIED;
  }

  template<class A, class B, class C, class D, class E, class F, void (*f) (A, B, C, D, E, F)>
  SCM
  VoidGeneric6 (SCM a, SCM b, SCM c, SCM d, SCM e, SCM f1)
  {
    GRAB;

    f (FromGuile<A> (a, 1, where),
       FromGuile<B> (b, 2, where),
       FromGuile<C> (c, 3, where),
       FromGuile<D> (d, 4, where),
       FromGuile<E> (e, 5, where),
       FromGuile<F> (f1, 6, where));

    return SCM_UNSPECIFIED;
  }



  template<class T, class R, R (T::*f) ()>
  SCM
  MemberGeneric0 (SCM smob)
  {
    GRAB;

    Loki::SmartPtr<T> p = FromGuile<Loki::SmartPtr<T> > (smob, 1, where);

    return ToGuile (((*p).*f) ());
  }

  template<class T, class R, class A, R (T::*f) (A)>
  SCM
  MemberGeneric1 (SCM smob, SCM a)
  {
    GRAB;

    Loki::SmartPtr<T> p = FromGuile<Loki::SmartPtr<T> > (smob, 1, where);

    return ToGuile (((*p).*f) (FromGuile<A> (a, 2, where)));
  }

  template<class T, class R, class A, class B, R (T::*f) (A, B)>
  SCM
  MemberGeneric2 (SCM smob, SCM a, SCM b)
  {
    GRAB;

    Loki::SmartPtr<T> p = FromGuile<Loki::SmartPtr<T> > (smob, 1, where);

    return ToGuile (((*p).*f) (FromGuile<A> (a, 2, where),
			     FromGuile<B> (b, 3, where)));
  }

  template<class T, class R, class A, class B, class C, R (T::*f) (A, B, C)>
  SCM
  MemberGeneric3 (SCM smob, SCM a, SCM b, SCM c)
  {
    GRAB;

    Loki::SmartPtr<T> p = FromGuile<Loki::SmartPtr<T> > (smob, 1, where);

    return ToGuile (((*p).*f) (FromGuile<A> (a, 2, where),
			     FromGuile<B> (b, 3, where),
			     FromGuile<C> (c, 4, where)));
  }

  template<class T, class R, class A, class B, class C, class D, R (T::*f) (A, B, C, D)>
  SCM
  MemberGeneric4 (SCM smob, SCM a, SCM b, SCM c, SCM d)
  {
    GRAB;

    Loki::SmartPtr<T> p = FromGuile<Loki::SmartPtr<T> > (smob, 1, where);

    return ToGuile (((*p).*f) (FromGuile<A> (a, 2, where),
			     FromGuile<B> (b, 3, where),
			     FromGuile<C> (c, 4, where),
			     FromGuile<D> (d, 5, where)));
  }

  template<class T, class R, class A, class B, class C, class D, class E, R (T::*f) (A, B, C, D, E)>
  SCM
  MemberGeneric5 (SCM smob, SCM a, SCM b, SCM c, SCM d, SCM e)
  {
    GRAB;

    Loki::SmartPtr<T> p = FromGuile<Loki::SmartPtr<T> > (smob, 1, where);

    return ToGuile (((*p).*f) (FromGuile<A> (a, 2, where),
			     FromGuile<B> (b, 3, where),
			     FromGuile<C> (c, 4, where),
			     FromGuile<D> (d, 5, where),
			     FromGuile<E> (e, 6, where)));
  }

  template<class T, class R, class A, class B, class C, class D, class E, class F, R (T::*f) (A, B, C, D, E, F)>
  SCM
  MemberGeneric6 (SCM smob, SCM a, SCM b, SCM c, SCM d, SCM e, SCM f1)
  {
    GRAB;

    Loki::SmartPtr<T> p = FromGuile<Loki::SmartPtr<T> > (smob, 1, where);

    return ToGuile (((*p).*f) (FromGuile<A> (a, 2, where),
			     FromGuile<B> (b, 3, where),
			     FromGuile<C> (c, 4, where),
			     FromGuile<D> (d, 5, where),
			     FromGuile<E> (e, 6, where),
			     FromGuile<F> (f1, 7, where)));
  }


  template<class T, void (T::*f) ()>
  SCM
  VoidMemberGeneric0 (SCM smob)
  {
    GRAB;

    Loki::SmartPtr<T> p = FromGuile<Loki::SmartPtr<T> > (smob, 1, where);
    ((*p).*f) ();

    return SCM_UNSPECIFIED;
  }

  template<class T, class A, void (T::*f) (A)>
  SCM
  VoidMemberGeneric1 (SCM smob, SCM a)
  {
    GRAB;

    Loki::SmartPtr<T> p = FromGuile<Loki::SmartPtr<T> > (smob, 1, where);
    ((*p).*f) (FromGuile<A> (a, 2, where));

    return SCM_UNSPECIFIED;
  }

  template<class T, class A, class B, void (T::*f) (A, B)>
  SCM
  VoidMemberGeneric2 (SCM smob, SCM a, SCM b)
  {
    GRAB;

    Loki::SmartPtr<T> p = FromGuile<Loki::SmartPtr<T> > (smob, 1, where);
    ((*p).*f) (FromGuile<A> (a, 2, where),
	     FromGuile<B> (b, 3, where));

    return SCM_UNSPECIFIED;
  }

  template<class T, class A, class B, class C, void (T::*f) (A, B, C)>
  SCM
  VoidMemberGeneric3 (SCM smob, SCM a, SCM b, SCM c)
  {
    GRAB;

    Loki::SmartPtr<T> p = FromGuile<Loki::SmartPtr<T> > (smob, 1, where);
    ((*p).*f) (FromGuile<A> (a, 2, where),
	     FromGuile<B> (b, 3, where),
	     FromGuile<C> (c, 4, where));

    return SCM_UNSPECIFIED;
  }

  template<class T, class A, class B, class C, class D, void (T::*f) (A, B, C, D)>
  SCM
  VoidMemberGeneric4 (SCM smob, SCM a, SCM b, SCM c, SCM d)
  {
    GRAB;

    Loki::SmartPtr<T> p = FromGuile<Loki::SmartPtr<T> > (smob, 1, where);
    ((*p).*f) (FromGuile<A> (a, 2, where),
	     FromGuile<B> (b, 3, where),
	     FromGuile<C> (c, 4, where),
	     FromGuile<D> (d, 5, where));

    return SCM_UNSPECIFIED;
  }

  template<class T, class A, class B, class C, class D, class E, void (T::*f) (A, B, C, D, E)>
  SCM
  VoidMemberGeneric5 (SCM smob, SCM a, SCM b, SCM c, SCM d, SCM e)
  {
    GRAB;

    Loki::SmartPtr<T> p = FromGuile<Loki::SmartPtr<T> > (smob, 1, where);
    ((*p).*f) (FromGuile<A> (a, 2, where),
	     FromGuile<B> (b, 3, where),
	     FromGuile<C> (c, 4, where),
	     FromGuile<D> (d, 5, where),
	     FromGuile<E> (e, 6, where));

    return SCM_UNSPECIFIED;
  }

  template<class T, class A, class B, class C, class D, class E, class F, void (T::*f) (A, B, C, D, E, F)>
  SCM
  VoidMemberGeneric6 (SCM smob, SCM a, SCM b, SCM c, SCM d, SCM e, SCM f1)
  {
    GRAB;

    Loki::SmartPtr<T> p = FromGuile<Loki::SmartPtr<T> > (smob, 1, where);
    ((*p).*f) (FromGuile<A> (a, 2, where),
	     FromGuile<B> (b, 3, where),
	     FromGuile<C> (c, 4, where),
	     FromGuile<D> (d, 5, where),
	     FromGuile<E> (e, 6, where),
	     FromGuile<F> (f1, 7, where));

    return SCM_UNSPECIFIED;
  }

#undef GRAB


  template<class T>
  size_t
  FreeGeneric (SCM smob)
  {
    delete (Loki::SmartPtr<T> *) SCM_SMOB_DATA (smob);
    return 0;
  }

  template<class T, void (T::*f) (Scheme, scm_print_state *)>
  int
  PrintGeneric (SCM smob, SCM port, scm_print_state *state)
  {
    Loki::SmartPtr<T> p = FromGuile<Loki::SmartPtr<T> > (smob, 1, "print");
    ((*p).*f) (Scheme (port), state);
    return 1;
  }

  template<class T>
  SCM
  EqualpGeneric (SCM smob, SCM smob2)
  {
    Loki::SmartPtr<T> p = FromGuile<Loki::SmartPtr<T> > (smob, 1, "equal?");
    Loki::SmartPtr<T> p2 = FromGuile<Loki::SmartPtr<T> > (smob2, 2, "equal?");
    return (*p == *p2)? SCM_BOOL_T: SCM_BOOL_F;
  }
}



//! Template for conversion to and from Guile.
/*!
 * Specializations of this class are used to convert objects of type T to and
 * from Guile.  You may define your own specializations to add new conversions.
 * Note that if you want to embed your own type in Guile rather than convert it
 * to a Scheme structure, you should use AddType() instead.
 * 
 * Specializations of this class must contain functions ToGuile and FromGuile.
 * They have the same interface as those at the top level.
 */
template<class T>
class Converter
{
};

template<>
struct Converter<SCM>
{
  SCM
  ToGuile (SCM a)
  {
    return a;
  }

  SCM
  FromGuile (SCM a, int arg, const char *where)
  {
    return a;
  }
};

template<>
struct Converter<Scheme>
{
  SCM
  ToGuile (Scheme a)
  {
    return a.GetSCM ();
  }

  Scheme
  FromGuile (SCM a, int arg, const char *where)
  {
    return Scheme (a);
  }
};

template<>
struct Converter<bool>
{
  SCM
  ToGuile (bool a)
  {
    return SCM_BOOL (a);
  }

  int
  FromGuile (SCM a, int arg, const char *where)
  {
    return SCM_NFALSEP (a);
  }
};

template<>
struct Converter<char>
{
  SCM
  ToGuile (char a)
  {
    return SCM_MAKE_CHAR (a);
  }

  char
  FromGuile (SCM a, int arg, const char *where)
  {
    SCM_ASSERT (SCM_CHARP (a), a, SCM_ARG1 + arg - 1, where);
    
    return SCM_CHAR (a);
  }
};

template<>
struct Converter<int>
{
  SCM
  ToGuile (int a)
  {
    return SCM_MAKINUM (a);
  }

  int
  FromGuile (SCM a, int arg, const char *where)
  {
    SCM_ASSERT (SCM_NUMBERP (a), a, SCM_ARG1 + arg - 1, where);
    
    return SCM_INUMP (a)? SCM_INUM (a): static_cast<int> (SCM_REAL_VALUE (a));
  }
};

template<>
struct Converter<unsigned int>
{
  SCM
  ToGuile (unsigned int a)
  {
    return SCM_MAKINUM (a);
  }

  unsigned int
  FromGuile (SCM a, int arg, const char *where)
  {
    SCM_ASSERT (SCM_NUMBERP (a), a, SCM_ARG1 + arg - 1, where);
    
    return SCM_INUMP (a)? SCM_INUM (a): static_cast<int> (SCM_REAL_VALUE (a));
  }
};
  
template<>
struct Converter<double>
{
  SCM
  ToGuile (double a)
  {
    return scm_make_real (a);
  }

  double
  FromGuile (SCM a, int arg, const char *where)
  {
    SCM_ASSERT (SCM_NUMBERP (a), a, SCM_ARG1 + arg - 1, where);

    return SCM_INUMP (a)? static_cast<double> (SCM_INUM (a)): SCM_REAL_VALUE (a);
  }
};

template<>
struct Converter<float>
{
  SCM
  ToGuile (float a)
  {
    return scm_make_real (a);
  }

  float
  FromGuile (SCM a, int arg, const char *where)
  {
    SCM_ASSERT (SCM_NUMBERP (a), a, SCM_ARG1 + arg - 1, where);

    return SCM_INUMP (a)? static_cast<float> (SCM_INUM (a)): SCM_REAL_VALUE (a);
  }
};

template<>
struct Converter<const char *>
{
  SCM
  ToGuile (const char *a)
  {
    return scm_makfrom0str (a);
  }

  const char *
  FromGuile (SCM a, int arg, const char *where)
  {
    bool string_or_symbol = (SCM_STRINGP (a) || SCM_SYMBOLP (a));
    SCM_ASSERT (string_or_symbol? SCM_BOOL_T: SCM_BOOL_F, a, SCM_ARG1 + arg - 1, where);

    return SCM_STRINGP (a)? SCM_STRING_CHARS (a): SCM_SYMBOL_CHARS (a);
  }
};

template<>
struct Converter<std::string>
{
  SCM
  ToGuile (std::string a)
  {
    return Converter<const char *> ().ToGuile (a.c_str ());
  }

  std::string
  FromGuile (SCM a, int arg, const char *where)
  {
    return std::string (Converter<const char *> ().FromGuile (a, arg, where));
  }
};

template<class T>
struct Converter<std::vector<T> >
{
  SCM
  ToGuile (std::vector<T> a)
  {
    SCM s = scm_make_vector (Converter<int> ().ToGuile (a.size ()), SCM_BOOL_F);
    int ii = 0;
    
    for (typename std::vector<T>::iterator i = a.begin (); i != a.end (); ++i)
      {
	scm_vector_set_x (s, Converter<int> ().ToGuile (ii++),
					   Converter<T> ().ToGuile (*i));
      }
    
    return s;
  }

  std::vector<T>
  FromGuile (SCM a, int arg, const char *where)
  {
    SCM_ASSERT (scm_vector_p (a),
		a, SCM_ARG1 + arg - 1, where);

    std::vector<T> v;
    int s = Converter<int> ().FromGuile (scm_vector_length (a), arg, where);

    for (int i = 0; i < s; ++i)
      {
	v.push_back (Converter<T> ().FromGuile
		     (scm_vector_ref (a, Converter<int> ().ToGuile (i)),
		      arg, where));
      }

    return v;
  }
};

template<class T>
struct Converter<std::list<T> >
{
  SCM
  ToGuile (std::list<T> a)
  {
    SCM s = SCM_EOL;
    
    for (typename std::list<T>::reverse_iterator i = a.rbegin (); i != a.rend (); ++i)
      {
	s = scm_cons (Converter<T> ().ToGuile (*i), s);
      }
    
    return s;
  }

  std::list<T>
  FromGuile (SCM a, int arg, const char *where)
  {
    SCM_ASSERT (scm_list_p (a),
		a, SCM_ARG1 + arg - 1, where);

    std::list<T> l;

    SCM s = a;
    while (SCM_NNULLP (s))
      {
	l.push_back (Converter<T> ().FromGuile (SCM_CAR (s), arg, where));
	s = SCM_CDR (s);
      }

    return l;
  }
};

template<class T>
struct Converter<Loki::SmartPtr<T> >
{
private:
  bool
  IsDescendantOf (SCM a, std::string type)
  {
    for (std::list<std::string>::iterator i = (*internal::smob_children)[type].begin ();
	 i != (*internal::smob_children)[type].end (); ++i)
      {
	if (SCM_SMOB_PREDICATE ((unsigned int) (*internal::smob_type)[*i], a)
	    || IsDescendantOf (a, *i))
	  {
	    return true;
	  }
      }
    return false;
  }

public:
  SCM
  ToGuile (const Loki::SmartPtr<T>& a)
  {
    std::string type = typeid (Loki::SmartPtr<T>).name ();
    
    if (internal::smob_type->find (type) == internal::smob_type->end ())
      {
	std::cerr << "ERROR: Don't know how to convert type '" << type << "' to Guile." << std::endl;
	exit (1);
      }
    else
      {
	Loki::SmartPtr<T> *ptr = new Loki::SmartPtr<T> (a);

	SCM smob;
	SCM_NEWSMOB (smob, (*internal::smob_type)[type], ptr);
	return smob;
      }
  }

  Loki::SmartPtr<T>
  FromGuile (SCM a, int arg, const char *where)
  {
    std::string type = typeid(Loki::SmartPtr<T>).name ();

    SCM_ASSERT ((SCM_SMOB_PREDICATE ((unsigned int) (*internal::smob_type)[type], a)
		 || IsDescendantOf (a, type)),
		a, SCM_ARG1 + arg - 1, where);

    return *(Loki::SmartPtr<T> *) SCM_SMOB_DATA (a);
  }
};

template<class T>
SCM
ToGuile (T a)
{
  return Converter<T> ().ToGuile (a);
}

template<class T>
T
FromGuile (SCM a, int arg, const char *where)
{
  return Converter<T> ().FromGuile (a, arg, where);
}



Scheme
ToGuileSymbol (const char *a)
{
  return scm_string_to_symbol (ToGuile (a));
}

//! Converts a string to a Scheme symbol.
/*!
 * This function is provided because a string could either be converted to a
 * symbol or to a string.  The default is a string; this function converts to
 * a symbol.  It is safe to send the result of this function to a
 * callGuile*Function* function.
 */
Scheme
ToGuileSymbol (std::string a)
{
  return scm_string_to_symbol (ToGuile (a));
}



//! Makes a new type convertable to Guile.
/*!
 * Given a type T (which should be a class or struct), makes the type
 * T* work with ToGuile() and FromGuile().  Guile values created in this way are opaque,
 * but accessors can be defined.
 */
template<class T>
void
AddType (char *name)
{
  std::string type = typeid (Loki::SmartPtr<T>).name ();
  (*internal::smob_type)[type] = scm_make_smob_type (name, sizeof (Loki::SmartPtr<T>));

  size_t (*fcn) (SCM) = internal::FreeGeneric<T>;
  scm_set_smob_free ((*internal::smob_type)[typeid (Loki::SmartPtr<T>).name ()], fcn);
}

//! Declares that T is descended from Parent.
template<class T, class Parent>
void
AddTypeParent ()
{
  std::string type = typeid (Loki::SmartPtr<T>).name ();
  std::string parent = typeid (Loki::SmartPtr<Parent>).name ();

  (*internal::smob_children)[parent].push_front (type);
}

//! Sets the print function for a type.
/*!
 * Sets the print function for a type previously defined with addType().
 * This function is called when the object is to be displayed.  The default
 * function displays the name of the type and the object's address in Scheme-space.
 */
template<class T, void (T::*f) (Scheme, scm_print_state *)>
void
SetTypePrintFcn ()
{
  int (*fcn) (SCM, SCM, scm_print_state *) = internal::PrintGeneric<T, f>;
  scm_set_smob_print ((*internal::smob_type)[typeid (Loki::SmartPtr<T>).name ()], fcn);
}

//! Registers an operator == for a type.
/*!
 * Tells Guile that the type T, previously defined with addType(), has an
 * operator == defined that can be used by 'equal?'.  This operator should work
 * on the base type T, not on T*.
 */
template<class T>
void
SetTypeComparable ()
{
  SCM (*fcn) (SCM, SCM) = internal::EqualpGeneric<T>;
  scm_set_smob_equalp ((*internal::smob_type)[typeid (Loki::SmartPtr<T>).name ()], fcn);
}

/*! \} */



/*!
 * \defgroup gfc Using Guile from C++
 * \ingroup guile
 * 
 * These functions are used for calling Guile code from C++ code.
 * 
 * The most basic functions, loadFile() and EvalExpression() are
 * fairly self-explanatory.  However, when it is necessary to send C++
 * values into Scheme code, the callFunction family of functions are
 * useful.
 * 
 * These functions call a Guile function, automatically converting the arguments from C++,
 * and converting the return value to C++.  For example:
 *
 * \code
 * int i, j;
 * ...
 * char *s = callFunction2<char *> ("some-guile-function", i, j);
 * callVoidFunction2 ("some-other-guile-function", i, j);
 * \endcode
 *
 * The '2' in the name indicates that the function is called with 2 arguments.  The 'Void' in
 * the name indicates that the function's return value is ignored.
 *
 * Note that the string can not only contain the name of a Guile function, but it can
 * contain any expression that evaulates to a function.  So, using Scheme's support
 * for closures, it is easy to run a snippet of Scheme code with access to C++ variables.
 * For example:
 *
 * \code
 * int i, j;
 * ...
 * int s = callFunction2<int> ("(lambda (a b) (+ (* a 2) b))", i, j);
 * \endcode
 * 
 * \{
 */

//! Loads Guile code from file.
void
LoadFile (std::string file)
{
  scm_primitive_load (scm_makfrom0str (file.c_str ()));
}

//! Evaluates a Guile expression, returning the value.
/*!
 * \arg R specifies the type of the value returned.  An error is raies if the expression
 * does not return a value compatible with that type.
 */
template<class R>
R
EvalExpression (std::string exp)
{
  return FromGuile<R> (scm_eval_string (scm_makfrom0str (exp.c_str ())), 1, "EvalExpression");
}

template<>
void
EvalExpression<void> (std::string exp)
{
  scm_eval_string (scm_makfrom0str (exp.c_str ()));
}

//! Calls a Guile function of no arguments and returns the function's value.
/*!
 * \arg R specifies the type of the value returned.  An error is raies if the expression
 * does not return a value compatible with that type.
 * 
 * \arg fcn is an expression that evalueates to a Guile function.  This could be the name
 * of a function, or a lamba expression, for instance.
 */
template<class R>
R
CallFunction0 (const char *fcn)
{
  return FromGuile<R> (scm_primitive_eval (SCM_LIST1 (scm_eval_string (scm_makfrom0str (fcn)))),
		       1, "CallFunction0");
}

//! Calls a Guile function of 1 argument and returns the function's value.
/*!
 * Note that the argument A can be automatically detected, and need not be specified.
 * 
 * \arg R specifies the type of the value returned.  An error is raies if the expression
 * does not return a value compatible with that type.
 * 
 * \arg fcn is an expression that evalueates to a Guile function.  This could be the name
 * of a function, or a lamba expression, for instance.
 */
template<class R, class A>
R
CallFunction1 (const char *fcn, A a)
{
  return FromGuile<R> (scm_primitive_eval (SCM_LIST2 (scm_eval_string (scm_makfrom0str (fcn)),
					    SCM_LIST2 (scm_sym_quote, ToGuile (a)))),
		       1, "CallFunction1");
}

//! Calls a Guile function of 2 arguments and returns the function's value.
/*!
 * Note that the arguments A and B can be automatically detected, and need not be specified.
 * 
 * \arg R specifies the type of the value returned.  An error is raies if the expression
 * does not return a value compatible with that type.
 * 
 * \arg fcn is an expression that evalueates to a Guile function.  This could be the name
 * of a function, or a lamba expression, for instance.
 */
template<class R, class A, class B>
R
CallFunction2 (const char *fcn, A a, B b)
{
  return FromGuile<R> (scm_primitive_eval (SCM_LIST3 (scm_eval_string (scm_makfrom0str (fcn)),
					    SCM_LIST2 (scm_sym_quote, ToGuile (a)),
					    SCM_LIST2 (scm_sym_quote, ToGuile (v)))),
		       1, "CallFunction2");
}

//! Calls a Guile function of 3 arguments and returns the function's value.
/*!
 * Note that the arguments A, B, and C can be automatically detected, and need not be specified.
 * 
 * \arg R specifies the type of the value returned.  An error is raies if the expression
 * does not return a value compatible with that type.
 * 
 * \arg fcn is an expression that evalueates to a Guile function.  This could be the name
 * of a function, or a lamba expression, for instance.
 */
template<class R, class A, class B, class C>
R
CallFunction3 (const char *fcn, A a, B b, C c)
{
  return FromGuile<R> (scm_primitive_eval (SCM_LIST4 (scm_eval_string (scm_makfrom0str (fcn)),
					    SCM_LIST2 (scm_sym_quote, ToGuile (a)),
					    SCM_LIST2 (scm_sym_quote, ToGuile (b)),
					    SCM_LIST2 (scm_sym_quote, ToGuile (c)))),
		       1, "CallFunction3");
}

//! Calls a Guile function of 4 arguments and returns the function's value.
/*!
 * Note that the arguments A, B, C, and D can be automatically detected, and need not be specified.
 * 
 * \arg R specifies the type of the value returned.  An error is raies if the expression
 * does not return a value compatible with that type.
 * 
 * \arg fcn is an expression that evalueates to a Guile function.  This could be the name
 * of a function, or a lamba expression, for instance.
 */
template<class R, class A, class B, class C, class D>
R
CallFunction4 (const char *fcn, A a, B b, C c, D d)
{
  return FromGuile<R> (scm_primitive_eval (SCM_LIST5 (scm_eval_string (scm_makfrom0str (fcn)),
					    SCM_LIST2 (scm_sym_quote, ToGuile (a)),
					    SCM_LIST2 (scm_sym_quote, ToGuile (b)),
					    SCM_LIST2 (scm_sym_quote, ToGuile (c)),
					    SCM_LIST2 (scm_sym_quote, ToGuile (d)))),
		       1, "CallFunction4");
}

//! Calls a Guile function of no arguments, ignoring the return value.
/*!
 * \arg fcn is an expression that evalueates to a Guile function.  This could be the name
 * of a function, or a lamba expression, for instance.
 */
void
CallVoidFunction0 (const char *fcn)
{
  scm_primitive_eval (SCM_LIST1 (scm_eval_string (scm_makfrom0str (fcn))));
}

//! Calls a Guile function of 1 argument, ignoring the return value.
/*!
 * Note that the argument A can be automatically detected, and need not be specified.
 * 
 * \arg fcn is an expression that evalueates to a Guile function.  This could be the name
 * of a function, or a lamba expression, for instance.
 */
template<class A>
void
CallVoidFunction1 (const char *fcn, A a)
{
  scm_primitive_eval (SCM_LIST2 (scm_eval_string (scm_makfrom0str (fcn)),
		       SCM_LIST2 (scm_sym_quote, ToGuile (a))));
}

//! Calls a Guile function of 2 arguments, ignoring the return value.
/*!
 * Note that the arguments A and B can be automatically detected, and need not be specified.
 * 
 * \arg fcn is an expression that evalueates to a Guile function.  This could be the name
 * of a function, or a lamba expression, for instance.
 */
template<class A, class B>
void
CallVoidFunction2 (const char *fcn, A a, B b)
{
  scm_primitive_eval (SCM_LIST3 (scm_eval_string (scm_makfrom0str (fcn)),
		       SCM_LIST2 (scm_sym_quote, ToGuile (a)),
		       SCM_LIST2 (scm_sym_quote, ToGuile (b))));
}

//! Calls a Guile function of 3 arguments, ignoring the return value.
/*!
 * Note that the arguments A, B, and C can be automatically detected, and need not be specified.
 * 
 * \arg fcn is an expression that evalueates to a Guile function.  This could be the name
 * of a function, or a lamba expression, for instance.
 */
template<class A, class B, class C>
void
CallVoidFunction3 (const char *fcn, A a, B b, C c)
{
  scm_primitive_eval (SCM_LIST4 (scm_eval_string (scm_makfrom0str (fcn)),
		       SCM_LIST2 (scm_sym_quote, ToGuile (a)),
		       SCM_LIST2 (scm_sym_quote, ToGuile (b)),
		       SCM_LIST2 (scm_sym_quote, ToGuile (c))));
}

//! Calls a Guile function of 4 arguments, ignoring the return value.
/*!
 * Note that the arguments A, B, C, and D can be automatically detected, and need not be specified.
 * 
 * \arg fcn is an expression that evalueates to a Guile function.  This could be the name
 * of a function, or a lamba expression, for instance.
 */
template<class A, class B, class C, class D>
void
CallVoidFunction4 (const char *fcn, A a, B b, C c, D d)
{
  scm_primitive_eval (SCM_LIST5 (scm_eval_string (scm_makfrom0str (fcn)),
		       SCM_LIST2 (scm_sym_quote, ToGuile (a)),
		       SCM_LIST2 (scm_sym_quote, ToGuile (b)),
		       SCM_LIST2 (scm_sym_quote, ToGuile (c)),
		       SCM_LIST2 (scm_sym_quote, ToGuile (d))));
}

/*! \} */



/*!
 * \defgroup cfg Using C++ from Guile
 * \ingroup guile
 * 
 * These functions are used for making C++ code available to Guile code.
 * 
 * Since C++ code must be compiled, it cannot simply be run from within Scheme code.
 * To make C++ code available to scheme, bindings must be created.  This means that
 * C++ functions are given Guile names and made available to Guile code.
 *
 * This is done using the various add*Binding* functions available here.  Versions
 * are available for functions, constructors, and member functions.  Their usages are all
 * fairly similar.  Here is an example:
 * 
 * \code
 * int
 * some_function (double a, char *b)
 * {
 *   ...
 * }
 * ...
 * addBinding2<int, double, char *, some_function> ("some-function");
 * \endcode
 * 
 * \{
 */

//! Creates a Guile binding for a constructor of no arguments.
/*!
 * Creates a Guile binding for a constructor of type T* which takes no arguments.
 * name is the name as it will appear in Guile.
 */
template<class T>
void
AddConstructorBinding0 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) () = internal::NewGeneric0<T>;
  scm_c_define_gsubr (internal::s = name, 0, 0, 0, (SCM (*) ()) fcn);

  // See the comment for the GRAB macro above for an explanation of this hack.
  internal::unreal = true;
  fcn ();
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a constructor of 1 argument.
/*!
 * Creates a Guile binding for a constructor of type T* which takes 1 argument of type A.
 * name is the name as it will appear in Guile.
 */
template<class T, class A>
void
AddConstructorBinding1 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM) = internal::NewGeneric1<T, A>;
  scm_c_define_gsubr (internal::s = name, 1, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a constructor of 2 arguments.
/*!
 * Creates a Guile binding for a constructor of type T* which takes 2 arguments of type A and B.
 * name is the name as it will appear in Guile.
 */
template<class T, class A, class B>
void
AddConstructorBinding2 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM) = internal::NewGeneric2<T, A, B>;
  scm_c_define_gsubr (internal::s = name, 2, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a constructor of 3 arguments.
/*!
 * Creates a Guile binding for a constructor of type T* which takes 3 arguments of type A, B and C.
 * name is the name as it will appear in Guile.
 */
template<class T, class A, class B, class C>
void
AddConstructorBinding3 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM) = internal::NewGeneric3<T, A, B, C>;
  scm_c_define_gsubr (internal::s = name, 3, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a constructor of 4 arguments.
/*!
 * Creates a Guile binding for a constructor of type T* which takes 4 arguments of type A, B, C, and D.
 * name is the name as it will appear in Guile.
 */
template<class T, class A, class B, class C, class D>
void
AddConstructorBinding4 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM, SCM) = internal::NewGeneric4<T, A, B, C, D>;
  scm_c_define_gsubr (internal::s = name, 4, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a constructor of 5 arguments.
/*!
 * Creates a Guile binding for a constructor of type T* which takes 5
 * arguments of type A, B, C, D and E.
 * name is the name as it will appear in Guile.
 */
template<class T, class A, class B, class C, class D, class E>
void
AddConstructorBinding5 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM, SCM, SCM) = internal::NewGeneric5<T, A, B, C, D, E>;
  scm_c_define_gsubr (internal::s = name, 5, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a constructor of 6 arguments.
/*!
 * Creates a Guile binding for a constructor of type T* which takes 6
 * arguments of type A, B, C, D, E and F.
 * name is the name as it will appear in Guile.
 */
template<class T, class A, class B, class C, class D, class E, class F>
void
AddConstructorBinding6 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM, SCM, SCM, SCM) = internal::NewGeneric6<T, A, B, C, D, E, F>;
  scm_c_define_gsubr (internal::s = name, 6, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}



//! Creates a Guile binding for a function of no arguments.
/*!
 * Creates a Guile binding for a function f returning type R which takes no arguments.
 * name is the name as it will appear in Guile.
 */
template<class R, R (*f) ()>
void
AddBinding0 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) () = internal::Generic0<R, f>;
  scm_c_define_gsubr (internal::s = name, 0, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn ();
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a function of 1 argument.
/*!
 * Creates a Guile binding for a function f returning type R which takes 1 argument of type A.
 * name is the name as it will appear in Guile.
 */
template<class R, class A, R (*f) (A)>
void
AddBinding1 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM) = internal::Generic1<R, A, f>;
  scm_c_define_gsubr (internal::s = name, 1, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a function of 2 arguments.
/*!
 * Creates a Guile binding for a function f returning type R which takes 2 arguments of types A and B.
 * name is the name as it will appear in Guile.
 */
template<class R, class A, class B, R (*f) (A, B)>
void
AddBinding2 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM) = internal::Generic2<R, A, B, f>;
  scm_c_define_gsubr (internal::s = name, 2, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a function of 3 arguments.
/*!
 * Creates a Guile binding for a function f returning type R which takes 3 arguments of types A, B, and C.
 * name is the name as it will appear in Guile.
 */
template<class R, class A, class B, class C, R (*f) (A, B, C)>
void
AddBinding3 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM) = internal::Generic3<R, A, B, C, f>;
  scm_c_define_gsubr (internal::s = name, 3, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a function of 4 arguments.
/*!
 * Creates a Guile binding for a function f returning type R which
 * takes 4 arguments of types A, B, C, and D.
 * name is the name as it will appear in Guile.
 */
template<class R, class A, class B, class C, class D, R (*f) (A, B, C, D)>
void
AddBinding4 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM, SCM) = internal::Generic4<R, A, B, C, D, f>;
  scm_c_define_gsubr (internal::s = name, 4, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a function of 5 arguments.
/*!
 * Creates a Guile binding for a function f returning type R which
 * takes 5 arguments of types A, B, C, D and E.
 * name is the name as it will appear in Guile.
 */
template<class R, class A, class B, class C, class D, class E, R (*f) (A, B, C, D, E)>
void
AddBinding5 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM, SCM, SCM) = internal::Generic5<R, A, B, C, D, E, f>;
  scm_c_define_gsubr (internal::s = name, 5, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a function of 6 arguments.
/*!
 * Creates a Guile binding for a function f returning type R which
 * takes 6 arguments of types A, B, C, D and E.
 * name is the name as it will appear in Guile.
 */
template<class R, class A, class B, class C, class D, class E, class F, R (*f) (A, B, C, D, E, F)>
void
AddBinding6 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM, SCM, SCM, SCM) = internal::Generic6<R, A, B, C, D, E, F, f>;
  scm_c_define_gsubr (internal::s = name, 6, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}


//! Creates a Guile binding for a void function of no arguments.
/*!
 * Creates a Guile binding for a function f returning void which takes no arguments.
 * name is the name as it will appear in Guile.
 */
template<void (*f) ()>
void
AddVoidBinding0 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) () = internal::VoidGeneric0<f>;
  scm_c_define_gsubr (internal::s = name, 0, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn ();
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a void function of 1 argument.
/*!
 * Creates a Guile binding for a function f returning void which takes 1 argument of type A.
 * name is the name as it will appear in Guile.
 */
template<class A, void (*f) (A)>
void
AddVoidBinding1 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM) = internal::VoidGeneric1<A, f>;
  scm_c_define_gsubr (internal::s = name, 1, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a void function of 2 arguments.
/*!
 * Creates a Guile binding for a function f returning void which takes 2 arguments of types A and B.
 * name is the name as it will appear in Guile.
 */
template<class A, class B, void (*f) (A, B)>
void
AddVoidBinding2 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM) = internal::VoidGeneric2<A, B, f>;
  scm_c_define_gsubr (internal::s = name, 2, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a void function of 3 arguments.
/*!
 * Creates a Guile binding for a function f returning void which takes 3 arguments of types A, B, and C.
 * name is the name as it will appear in Guile.
 */
template<class A, class B, class C, void (*f) (A, B, C)>
void
AddVoidBinding3 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM) = internal::VoidGeneric3<A, B, C, f>;
  scm_c_define_gsubr (internal::s = name, 3, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a void function of 4 arguments.
/*!
 * Creates a Guile binding for a function f returning void which
 * takes 4 arguments of types A, B, C, and D.
 * name is the name as it will appear in Guile.
 */
template<class A, class B, class C, class D, void (*f) (A, B, C, D)>
void
AddVoidBinding4 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM, SCM) = internal::VoidGeneric4<A, B, C, D, f>;
  scm_c_define_gsubr (internal::s = name, 4, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a void function of 5 arguments.
/*!
 * Creates a Guile binding for a function f returning void which
 * takes 5 arguments of types A, B, C, D and E.
 * name is the name as it will appear in Guile.
 */
template<class A, class B, class C, class D, class E, void (*f) (A, B, C, D, E)>
void
AddVoidBinding5 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM, SCM, SCM) = internal::VoidGeneric5<A, B, C, D, E, f>;
  scm_c_define_gsubr (internal::s = name, 5, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a void function of 6 arguments.
/*!
 * Creates a Guile binding for a function f returning void which
 * takes 6 arguments of types A, B, C, D, E and F.
 * name is the name as it will appear in Guile.
 */
template<class A, class B, class C, class D, class E, class F, void (*f) (A, B, C, D, E, F)>
void
AddVoidBinding6 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM, SCM, SCM, SCM) = internal::VoidGeneric6<A, B, C, D, E, F, f>;
  scm_c_define_gsubr (internal::s = name, 6, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}



//! Creates a Guile binding for a member function of no arguments.
/*!
 * Creates a Guile binding for a member function f of class T
 * returning type R which takes no arguments.
 * name is the name as it will appear in Guile.
 */
template<class T, class R, R (T::*f) ()>
void
AddMemberBinding0 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM) = internal::MemberGeneric0<T, R, f>;
  scm_c_define_gsubr (internal::s = name, 1, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a member function of 1 argument.
/*!
 * Creates a Guile binding for a member function f of class T
 * returning type R which takes 1 argument of type A.
 * name is the name as it will appear in Guile.
 */
template<class T, class R, class A, R (T::*f) (A)>
void
AddMemberBinding1 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM) = internal::MemberGeneric1<T, R, A, f>;
  scm_c_define_gsubr (internal::s = name, 2, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a member function of 2 arguments.
/*!
 * Creates a Guile binding for a member function f of class T
 * returning type R which takes 2 arguments of types A and B.
 * name is the name as it will appear in Guile.
 */
template<class T, class R, class A, class B, R (T::*f) (A, B)>
void
AddMemberBinding2 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM) = internal::MemberGeneric2<T, R, A, B, f>;
  scm_c_define_gsubr (internal::s = name, 3, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a member function of 3 arguments.
/*!
 * Creates a Guile binding for a member function f of class T
 * returning type R which takes 3 arguments of types A, B, and C.
 * name is the name as it will appear in Guile.
 */
template<class T, class R, class A, class B, class C, R (T::*f) (A, B, C)>
void
AddMemberBinding3 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM, SCM) = internal::MemberGeneric3<T, R, A, B, C, f>;
  scm_c_define_gsubr (internal::s = name, 4, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a member function of 4 arguments.
/*!
 * Creates a Guile binding for a member function f of class T
 * returning type R which takes 4 arguments of types A, B, C, and D.
 * name is the name as it will appear in Guile.
 */
template<class T, class R, class A, class B, class C, class D, R (T::*f) (A, B, C, D)>
void
AddMemberBinding4 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM, SCM, SCM) = internal::MemberGeneric4<T, R, A, B, C, D, f>;
  scm_c_define_gsubr (internal::s = name, 5, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a member function of 5 arguments.
/*!
 * Creates a Guile binding for a member function f of class T
 * returning type R which takes 5 arguments of types A, B, C, D and E.
 * name is the name as it will appear in Guile.
 */
template<class T, class R, class A, class B, class C, class D, class E, R (T::*f) (A, B, C, D, E)>
void
AddMemberBinding5 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM, SCM, SCM, SCM) = internal::MemberGeneric5<T, R, A, B, C, D, E, f>;
  scm_c_define_gsubr (internal::s = name, 6, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a member function of 6 arguments.
/*!
 * Creates a Guile binding for a member function f of class T
 * returning type R which takes 6 arguments of types A, B, C, D, E and F.
 * name is the name as it will appear in Guile.
 */
template<class T, class R, class A, class B, class C, class D, class E, class F, R (T::*f) (A, B, C, D, E, F)>
void
AddMemberBinding6 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM, SCM, SCM, SCM, SCM) = internal::MemberGeneric6<T, R, A, B, C, D, E, F, f>;
  scm_c_define_gsubr (internal::s = name, 7, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}


//! Creates a Guile binding for a void member function of no arguments.
/*!
 * Creates a Guile binding for a member function f of class T
 * returning void which takes no arguments.
 * name is the name as it will appear in Guile.
 */
template<class T, void (T::*f) ()>
void
AddVoidMemberBinding0 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM) = internal::VoidMemberGeneric0<T, f>;
  scm_c_define_gsubr (internal::s = name, 1, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a void member function of 1 argument.
/*!
 * Creates a Guile binding for a member function f of class T
 * returning void which takes 1 argument of type A.
 * name is the name as it will appear in Guile.
 */
template<class T, class A, void (T::*f) (A)>
void
AddVoidMemberBinding1 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM) = internal::VoidMemberGeneric1<T, A, f>;
  scm_c_define_gsubr (internal::s = name, 2, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a void member function of 2 arguments.
/*!
 * Creates a Guile binding for a member function f of class T
 * returning void which takes 2 arguments of types A and B.
 * name is the name as it will appear in Guile.
 */
template<class T, class A, class B, void (T::*f) (A, B)>
void
AddVoidMemberBinding2 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM) = internal::VoidMemberGeneric2<T, A, B, f>;
  scm_c_define_gsubr (internal::s = name, 3, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a void member function of 3 arguments.
/*!
 * Creates a Guile binding for a member function f of class T
 * returning void which takes 3 arguments of types A, B, and C.
 * name is the name as it will appear in Guile.
 */
template<class T, class A, class B, class C, void (T::*f) (A, B, C)>
void
AddVoidMemberBinding3 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM, SCM) = internal::VoidMemberGeneric3<T, A, B, C, f>;
  scm_c_define_gsubr (internal::s = name, 4, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a void member function of 4 arguments.
/*!
 * Creates a Guile binding for a member function f of class T
 * returning void which takes 4 arguments of types A, B, C, and D.
 * name is the name as it will appear in Guile.
 */
template<class T, class A, class B, class C, class D, void (T::*f) (A, B, C, D)>
void
AddVoidMemberBinding4 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM, SCM, SCM) = internal::VoidMemberGeneric4<T, A, B, C, D, f>;
  scm_c_define_gsubr (internal::s = name, 5, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a void member function of 5 arguments.
/*!
 * Creates a Guile binding for a member function f of class T
 * returning void which takes 5 arguments of types A, B, C, D and E.
 * name is the name as it will appear in Guile.
 */
template<class T, class A, class B, class C, class D, class E, void (T::*f) (A, B, C, D, E)>
void
AddVoidMemberBinding5 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM, SCM, SCM, SCM) = internal::VoidMemberGeneric5<T, A, B, C, D, E, f>;
  scm_c_define_gsubr (internal::s = name, 6, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

//! Creates a Guile binding for a void member function of 6 arguments.
/*!
 * Creates a Guile binding for a member function f of class T
 * returning void which takes 6 arguments of types A, B, C, D, E and F.
 * name is the name as it will appear in Guile.
 */
template<class T, class A, class B, class C, class D, class E, class F, void (T::*f) (A, B, C, D, E, F)>
void
AddVoidMemberBinding6 (const char *name)
{
  internal::grab_mutex.Lock ();
  SCM (*fcn) (SCM, SCM, SCM, SCM, SCM, SCM, SCM) = internal::VoidMemberGeneric6<T, A, B, C, D, E, F, f>;
  scm_c_define_gsubr (internal::s = name, 7, 0, 0, (SCM (*) ()) fcn);
  internal::unreal = true;
  fcn (SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F, SCM_BOOL_F);
  internal::grab_mutex.Unlock ();
}

/*! \} */

} // namespace guile

} // namespace support

} // namespace leg

#endif /* LEG_SUPPORT_GUILE_GUILE_H */

