/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: glconfig.h,v 1.2 2004/04/30 20:15:54 jechk Exp $
   $Log: glconfig.h,v $
   Revision 1.2  2004/04/30 20:15:54  jechk
   Big merge.  See ChangeLog for details.

   Revision 1.1  2004/03/28 21:30:07  jechk
   Added the support/graphics module.



   Created 3/24/04 by Jeff Binder <bindej@rpi.edu>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file glconfig.h
  \brief OpenGL settings management.
*/

#include <list>
#include <map>
#include <stack>

#include <GL/gl.h>

#include <leg/support/graphics/displaylist.h>

#ifndef LEG_SUPPORT_GRAPHICS_GLCONFIG_H
#define LEG_SUPPORT_GRAPHICS_GLCONFIG_H

namespace leg
{
namespace support
{
namespace graphics
{

//! Sets a GL option immediately.
/*! This is more efficient than calling GL because it may be able to
 * avoid calling OpenGL functions if the option is already in the
 * correct state.
 * 
 * It is important that this function is used because otherwise Leg
 * can lose track of an option's value.  In inner loops it might be
 * better to use GL functions directly (if it is known that the value
 * will change), but care must be taken when using them.
 */
void SetGlobalOption (GLenum option, bool flag);

//! Gets the value of a GL option.
/*! This is more efficient than calling GL because it may be able to
 * avoid calling OpenGL functions if Leg already knows the value of
 * the option.
 */
bool GetGlobalOption (GLenum option);

//! Similar to SetGlobalOption() but for client states.
void SetGlobalClientOption (GLenum option, bool flag);

//! Similar to GetGlobalOption() but for client states.
bool GetGlobalClientOption (GLenum option);

//! Maintains GL state information.
/*!
 * GLConfig objects store OpenGL options, and allow them to be set and
 * unset efficiently (without excess OpenGL calls).
 */
class GLConfig
{
 public:
  GLConfig () : use_display_list (false) {}

  GLConfig (const GLConfig &copy)
    : options (copy.options)
  {
  }

  ~GLConfig ()
  {
  }

  //! Sets the value of an option stored in the GLConfig.
  /*! This does not immediately change OpenGL's state.  It simply
   * stores the option for the next time Push () is called.
   */
  void SetOption (GLenum option, bool flag);

  //! Gets the value of an option that is stored in the GLConfig.
  bool GetOption (GLenum option);

  //! Returns true if the specified option is stored in the GLConfig.
  bool IsOptionStored (GLenum option);

  //! Removes an option from the GLConfig.
  /*! Options are not managed by a GLConfig unless you add them to
   * it.  This function makes the GLConfig stop managing the option if
   * it already is doing so.
   */
  void RemoveOption (GLenum option);

  //! Similar to SetOption() but for client states.
  void SetClientOption (GLenum option, bool flag);

  //! Similar to GetOption() but for client states.
  bool GetClientOption (GLenum option);

  //! Similar to IsOptionStored() but for client states.
  bool IsClientOptionStored (GLenum option);

  //! Similar to RemoveOption() but for client states.
  void RemoveClientOption (GLenum option);

  //! Pushes this configuration onto the stack.
  /*! All of the options stored in this GLConfig are set to their
   * respective values.  The current values of the options are saved
   * on the stack.
   */
  void Push ();

  //! Pops this configuration off the stack.
  /*! All options aremanaged by this GLConfig are reset to what they
   * were before it was pushed.
   */
  void Pop ();

  //! Enables the use of display lists.
  /*! There is a little overhead to enabling display lists, but they
   * increase the performance of the Push operation.  Display lists
   * are automatically disabled when the settings are changed.
   */
  void EnableDisplayList ();

  //! Disables the use of display lists.
  void DisableDisplayList ();

 private:
  typedef std::map<GLenum, bool> Options;

  Options options;
  std::stack<Options> saved;

  Options client_options;
  std::stack<Options> client_saved;

  bool use_display_list;
  DisplayList display_list;
};

}
}
}

#endif // LEG_SUPPORT_GRAPHICS_GLCONFIG_H
