/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: color.h,v 1.2 2004/05/22 06:54:16 jechk Exp $
   $Log: color.h,v $
   Revision 1.2  2004/05/22 06:54:16  jechk
   Graphics update, mainly framework stuff.

   Revision 1.1  2004/03/28 21:30:07  jechk
   Added the support/graphics module.



   Created 03/26/04 by Jeff Binder <bindej@rpi.edu>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file leg/support/graphics/color.h
  \brief Class to represent RGBA colors.
*/

#include <leg/support/maths/real.h>

#ifndef LEG_SUPPORT_GRAPHICS_COLOR_H
#define LEG_SUPPORT_GRAPHICS_COLOR_H

namespace leg
{
namespace support
{
namespace graphics
{

class Color
{
public:

  //! Initializes the color to black.
  Color ()
    : r (el[0]), g (el[1]), b (el[2]), a (el[3])
  {
    el[0] = 0.;
    el[1] = 0.;
    el[2] = 0.;
    el[3] = 1.;
  }

  //! The alpha channel is set to 1.
  Color (maths::real r, maths::real g, maths::real b)
    : r (el[0]), g (el[1]), b (el[2]), a (el[3])
  {
    el[0] = r;
    el[1] = g;
    el[2] = b;
    el[3] = 1.;
  }

  Color (maths::real r, maths::real g, maths::real b, maths::real a)
    : r (el[0]), g (el[1]), b (el[2]), a (el[3])
  {
    el[0] = r;
    el[1] = g;
    el[2] = b;
    el[3] = a;
  }

  Color (const Color& c)
    : r (el[0]), g (el[1]), b (el[2]), a (el[3])
  {
    el[0] = c.r;
    el[1] = c.g;
    el[2] = c.b;
    el[3] = c.a;
  }

  ~Color ()
  {}


  /// Assignment equal operator.
  Color &
  operator = (const Color &n)
  {
    r = n.r;
    g = n.g;
    b = n.b;
    a = n.a;
    
    return *this;
  }

  Color operator *(const maths::real& n) const
  {
    Color c (r * n, g * n, b * n, a * n);
    
    return c;
  }

  Color operator *= (const maths::real& n)
  {
    r *= n;
    g *= n;
    b *= n;
    a *= n;
    
    return *this;
  }

  Color operator / (const maths::real& n) const
  {
    Color c (r / n, g / n, b / n, a / n);
    
    return c;
  }

  Color operator /= (const maths::real& n)
  {
    r /= n;
    g /= n;
    b /= n;
    a /= n;
    
    return *this;
  }


  Color operator + (const Color& n) const
  {
    Color c (r + n.r, g + n.g, b + n.b, a + n.a);
    
    return c;
  }

  Color operator += (const Color& n)
  {
      r += n.r;
      g += n.g;
      b += n.b;
      a += n.a;
      
      return *this;
  }

  Color operator - (const Color& n) const
  {
    Color c (r - n.r, g - n.g, b - n.b, a - n.a);
    
    return c;
  }

  Color operator -= (const Color& n)
  {
      r -= n.r;
      g -= n.g;
      b -= n.b;
      a -= n.a;
      
      return *this;
  }

  Color operator *(const Color& n) const
  {
    Color c (r * n.r, g * n.g, b * n.b, a * n.a);
    
    return c;
  }

  Color operator *= (const Color& n)
  {
      r *= n.r;
      g *= n.g;
      b *= n.b;
      a *= n.a;
      
      return *this;
  }

  Color operator / (const Color& n) const
  {
    Color c (r / n.r, g / n.g, b / n.b, a / n.a);
    
    return c;
  }

  Color operator /= (const Color& n)
  {
      r /= n.r;
      g /= n.g;
      b /= n.b;
      a /= n.a;
      
      return *this;
  }


  bool operator == (const Color& n)
  {
    return (r == n.r) && (g == n.g) && (b == n.b) && (a == n.a);
  }


  //! Set this as the OpenGL color.
  void Set () const;

  maths::real el[4];

  maths::real &r;
  maths::real &g;
  maths::real &b;
  maths::real &a;
};

//! Returns the current GL color.
Color GetGlobalColor ();

std::ostream &operator << (std::ostream &s, Color v);

}
}
}

#endif /* LEG_SUPPORT_GRAPHICS_COLOR_H */
